import React, { useState, useEffect } from "react";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Switch,
  Paper,
  IconButton,
  Button,
  MenuItem,
  TextField,
  Typography,
  Modal,
  Fade,
  Backdrop,
  Box,
} from "@mui/material";
import axios from "axios";
import EditIcon from "@mui/icons-material/Edit";
import DeleteIcon from "@mui/icons-material/Delete";
import AddIcon from "@mui/icons-material/Add";

const apiUrl = process.env.REACT_APP_API_URL;

// Áreas predefinidas del consultorio
const RolUsuario = [
  { value: "administrador", label: "Administrador" },
  { value: "supervisor", label: "Supervisor" },
  { value: "cliente", label: "Cliente" },
  { value: "vendedor", label: "Vendedor/a" },
];

const RegisterUser = () => {
  const [users, setUsers] = useState([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [modalType, setModalType] = useState("add"); // Puede ser 'add' o 'edit'
  const [selectedUser, setSelectedUser] = useState({
    usuario: "",
    email: "",
    rol: "",
    password: "",
  });

  const handleToggleActive = async (userId, isActive) => {
    try {
      const token = localStorage.getItem("access_token");
      const response = await axios.put(
        `${apiUrl}/api/v1/usuarios${userId}/active`,
        { is_active: isActive },
        {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        }
      );

      if (response.status === 200) {
        setUsers((prevUsers) =>
          prevUsers.map((user) =>
            user.id === userId ? { ...user, is_active: isActive } : user
          )
        );
        toast.success("Estado del usuario actualizado exitosamente.");
      }
    } catch (error) {
      console.error("Error al actualizar el estado del usuario:", error);
      toast.error("Error al actualizar el estado del usuario.");
    }
  };

  useEffect(() => {
    const fetchUsers = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(`${apiUrl}/api/v1/usuarios/`, {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        });

        const data = response.data;
        setUsers(data);
      } catch (error) {
        toast.error("Error al cargar Usuarios.");
      }
    };

    fetchUsers();
  }, []);

  const handleEdit = (user) => {
    setIsModalOpen(true);
    setModalType("edit");
    setSelectedUser(user);
  };

  const handleDelete = async (userId) => {
    try {
      if (
        window.confirm("¿Estás seguro de que deseas eliminar este usuario?")
      ) {
        const token = localStorage.getItem("access_token");
        const response = await axios.delete(
          `${apiUrl}/api/v1/usuarios/${userId}`,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );

        if (response.status === 200) {
          setUsers(users.filter((user) => user.id !== userId));
          toast.success("Usuario eliminado exitosamente");
        }
      }
    } catch (error) {
      toast.error("Error al eliminar el Usuario");
    }
  };

  const handleAddNew = () => {
    setIsModalOpen(true);
    setModalType("add");
    setSelectedUser({
      usuario: "",
      email: "",
      rol: "",
      password: "",
      is_active: true,
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    const token = localStorage.getItem("access_token");

    try {
      if (modalType === "add") {
        const response = await axios.post(
          `${apiUrl}/api/v1/usuarios/`,
          selectedUser,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );
        if (response.status === 200) {
          setUsers([...users, response.data]);
          toast.success("Usuario agregado exitosamente");
        }
      } else if (modalType === "edit") {
        const response = await axios.put(
          `${apiUrl}/api/v1/usuarios/${selectedUser.id}`,
          selectedUser,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );
        if (response.status === 200) {
          setUsers(
            users.map((user) =>
              user.id === selectedUser.id ? response.data : user
            )
          );
          toast.success("Usuario actualizado exitosamente");
        }
      }
      setIsModalOpen(false);
    } catch (error) {
      toast.error("Error al guardar el Usuario");
    }
  };

  return (
    <>
      <Typography
        sx={{
          marginTop: "10px",
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h5"
      >
        Usuarios
      </Typography>
      <Button
        variant="contained"
        color="primary"
        startIcon={<AddIcon />}
        onClick={handleAddNew}
        sx={{ marginBottom: "20px", float: "right" }} // Cambié style por sx
      >
        Nuevo Usuario
      </Button>

      <TableContainer component={Paper}>
        <Table>
          <TableHead>
            <TableRow>
              <TableCell>Usuario</TableCell>
              <TableCell>Email</TableCell>
              <TableCell>Rol</TableCell>
              <TableCell>Activo</TableCell>
              <TableCell align="right">Acciones</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {users.map((user) => (
              <TableRow key={user.id}>
                <TableCell>{user.usuario}</TableCell>
                <TableCell>{user.email}</TableCell>
                <TableCell>{user.rol}</TableCell>
                <TableCell>
                  <Switch
                    checked={Boolean(user.is_active)}
                    onChange={(e) =>
                      handleToggleActive(user.id, e.target.checked)
                    }
                    color="primary"
                  />
                </TableCell>
                <TableCell align="right">
                  <IconButton onClick={() => handleEdit(user)}>
                    <EditIcon style={{ color: "green" }} />
                  </IconButton>
                  <IconButton onClick={() => handleDelete(user.id)}>
                    <DeleteIcon style={{ color: "#de2323" }} />
                  </IconButton>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>
      <ToastContainer position="bottom-right" />

      {/* Modal para agregar/editar usuario */}
      <Modal
        open={isModalOpen}
        onClose={() => setIsModalOpen(false)}
        closeAfterTransition
        slots={{ backdrop: Backdrop }}
        slotProps={{
          backdrop: {
            timeout: 1500,
          },
        }}
      >
        <Fade in={isModalOpen}>
          <Box
            sx={{
              backgroundColor: "white",
              padding: 3,
              margin: "auto",
              width: 320,
              position: "absolute",
              top: "50%",
              left: "50%",
              transform: "translate(-50%, -50%)",
              boxShadow: 24,
              borderRadius: 2,
            }}
          >
            <Typography variant="h6" component="h2">
              {modalType === "add" ? "Agregar Usuario" : "Editar Usuario"}
            </Typography>
            <form onSubmit={handleSubmit}>
              <TextField
                fullWidth
                label="Usuario"
                value={selectedUser.usuario}
                size="small"
                onChange={(e) =>
                  setSelectedUser({ ...selectedUser, usuario: e.target.value })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Email"
                value={selectedUser.email}
                size="small"
                onChange={(e) =>
                  setSelectedUser({ ...selectedUser, email: e.target.value })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                select
                label="Rol"
                value={selectedUser.rol}
                size="small"
                onChange={(e) =>
                  setSelectedUser({ ...selectedUser, rol: e.target.value })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              >
                {RolUsuario.map((rol) => (
                  <MenuItem key={rol.value} value={rol.value}>
                    {rol.label}
                  </MenuItem>
                ))}
              </TextField>
              <TextField
                fullWidth
                label="Contraseña"
                type="password"
                size="small"
                value={selectedUser.password || ""}
                onChange={(e) =>
                  setSelectedUser({ ...selectedUser, password: e.target.value })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <Box sx={{ textAlign: "right", mt: 2 }}>
                <Button
                  fullWidth
                  type="submit"
                  variant="contained"
                  color="primary"
                >
                  {modalType === "add" ? "Agregar" : "Guardar Cambios"}
                </Button>
              </Box>
            </form>
          </Box>
        </Fade>
      </Modal>
    </>
  );
};

export default RegisterUser;
