import React, { useState, useEffect } from "react";
import { jwtDecode } from "jwt-decode";
import { useNavigate } from "react-router-dom";
import "../App.css";
import {
  TextField,
  Button,
  Box,
  CircularProgress,
  Typography,
  Toolbar,
  AppBar,
  FormControl,
  InputLabel,
  OutlinedInput,
  InputAdornment,
  IconButton,
} from "@mui/material";
import { Visibility, VisibilityOff } from "@mui/icons-material";
import { toast } from "react-toastify";
import axios from "axios";
import Footer from "../components/Footer";
import CrearAdmin from "./CreateAdmin";

const Login = () => {
  const [username, setUsername] = useState("");
  const [password, setPassword] = useState("");
  const [userAdmin, setUserAdmin] = useState(null);
  const [showPassword, setShowPassword] = useState(false);
  const [loading, setLoading] = useState(false);
  const [decodedToken, setDecodedToken] = useState(null); // <-- Estado para almacenar token decodificado
  const navigate = useNavigate();
  const apiUrl = process.env.REACT_APP_API_URL;

  // Verificar administrador al montar
  useEffect(() => {
    const verificarAdmin = async () => {
      try {
        const response = await axios.get(`${apiUrl}/api/v1/verificar-admin`);
        setUserAdmin(response.data.admin_creado);
      } catch {
        setUserAdmin(true);
      }
    };
    verificarAdmin();
  }, [apiUrl]);

  // Procesar solicitudes caducadas
  useEffect(() => {
    const procesarSolicitudesCaducadas = async () => {
      try {
        await axios.post(
          `${apiUrl}/api/v1/solicitudes/procesar-solicitudes-caducadas`
        );
      } catch (error) {
        console.error(error);
      }
    };
    procesarSolicitudesCaducadas();
  }, [apiUrl]);

  // Manejar redirección cuando haya un token decodificado
  useEffect(() => {
    if (!decodedToken) return;

    const rol = decodedToken.rol?.toLowerCase() || "";

    const timer = setTimeout(() => {
      if (rol === "administrador" || rol === "asistente") {
        navigate("/administracion");
      } else if (rol === "supervisor" || rol === "vendedor") {
        navigate("/ventas");
      } else {
        navigate("/clientes");
      }
    }, 2000); // <-- 2 segundos de retraso

    // Cleanup para evitar problemas si el componente se desmonta antes
    return () => clearTimeout(timer);
  }, [decodedToken, navigate]);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);

    try {
      const loginData = new URLSearchParams();
      loginData.append("username", username);
      loginData.append("password", password);

      const response = await axios.post(`${apiUrl}/api/v1/login`, loginData, {
        headers: { "Content-Type": "application/x-www-form-urlencoded" },
      });

      const { data } = response.data;
      if (!data.access_token) throw new Error("No se recibió el token.");

      const token = data.access_token;
      localStorage.setItem("access_token", token);

      const decoded = jwtDecode(token);
      setDecodedToken(decoded); // <-- Esto dispara el useEffect de redirección

      toast.success("Login exitoso!");
    } catch (error) {
      const mensaje =
        error.response?.data?.mensaje || "Error al iniciar sesión";
      const detalle = error.response?.data?.error || "";
      toast.error(`${mensaje}${detalle ? ": " + detalle : ""}`);
    } finally {
      setLoading(false);
    }
  };

  if (userAdmin === false) {
    return (
      <CrearAdmin apiUrl={apiUrl} onCreated={() => window.location.reload()} />
    );
  }

  return (
    <>
      <AppBar position="static" sx={{ backgroundColor: "#0A4575" }}>
        <Toolbar>
          <img
            src="https://www.cementeriopuertasalcielo.com/wp-content/uploads/2023/02/Logo2x-03-1.webp"
            alt="Logo"
            style={{
              width: "270px",
              marginTop: 15,
              marginBottom: 15,
              marginRight: 20,
            }}
          />
          <Box sx={{ flexGrow: 1 }} />
        </Toolbar>
      </AppBar>

      <Box
        sx={{
          backgroundImage: `url('https://www.cementeriopuertasalcielo.com/wp-content/uploads/2024/01/CW16-scaled.jpg')`,
          backgroundRepeat: "no-repeat",
          backgroundSize: "cover",
          backgroundPosition: "center",
          minHeight: "calc(100vh - 80px)",
          display: "flex",
          justifyContent: "center",
          alignItems: "center",
        }}
      >
        <Box
          sx={{
            display: "flex",
            boxShadow: 3,
            maxWidth: "600px",
            width: "100%",
            backgroundColor: "#fff",
          }}
        >
          <Box
            sx={{
              backgroundColor: "#0A4575",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
              padding: 2,
              width: "270px",
            }}
          >
            <img
              src="https://www.cementeriopuertasalcielo.com/wp-content/uploads/2023/02/LogoFooter-1.webp"
              alt="Logo del Sistema"
              width="70%"
            />
          </Box>

          <Box sx={{ marginTop: 2, marginBottom: 2, padding: 3, flex: 1 }}>
            <Typography variant="h5" align="center" gutterBottom>
              Inicio de Sesión
            </Typography>

            <Box component="form" onSubmit={handleSubmit} sx={{ mt: 3 }}>
              <TextField
                fullWidth
                label="Usuario"
                variant="outlined"
                margin="normal"
                size="small"
                value={username}
                onChange={(e) => setUsername(e.target.value)}
                required
                sx={{ boxShadow: 3 }}
              />

              <FormControl
                fullWidth
                variant="outlined"
                margin="normal"
                size="small"
                sx={{ boxShadow: 3 }}
              >
                <InputLabel htmlFor="password">Contraseña</InputLabel>
                <OutlinedInput
                  id="password"
                  type={showPassword ? "text" : "password"}
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  label="Contraseña"
                  endAdornment={
                    <InputAdornment position="end">
                      <IconButton
                        onClick={() => setShowPassword(!showPassword)}
                        edge="end"
                      >
                        {showPassword ? <VisibilityOff /> : <Visibility />}
                      </IconButton>
                    </InputAdornment>
                  }
                />
              </FormControl>

              <Box sx={{ textAlign: "center", mt: 3 }}>
                <Button
                  type="submit"
                  variant="contained"
                  color="primary"
                  fullWidth
                  disabled={loading}
                >
                  {loading ? <CircularProgress size={24} /> : "Iniciar Sesión"}
                </Button>
              </Box>
            </Box>
          </Box>
        </Box>
      </Box>

      <Box sx={{ position: "fixed", bottom: 0, width: "100%" }}>
        <Footer />
      </Box>
    </>
  );
};

export default Login;
