import React, { useState, useEffect } from "react";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Switch,
  Paper,
  IconButton,
  Button,
  TextField,
  Typography,
  Modal,
  Fade,
  Backdrop,
  Box,
} from "@mui/material";
import axios from "axios";
import { Link } from "react-router-dom";
import EditIcon from "@mui/icons-material/Edit";
import DeleteIcon from "@mui/icons-material/Delete";
import AddIcon from "@mui/icons-material/Add";
import { toast } from "react-toastify";

const apiUrl = process.env.REACT_APP_API_URL;

const ListVendedores = () => {
  const [vendedores, setVendedores] = useState([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [modalType, setModalType] = useState("add"); // Puede ser 'add' o 'edit'
  const [selectedVendedor, setSelectedVendedor] = useState({
    nombre: "",
    cedula: "",
    telefono: "",
    email: "",
  });

  const handleToggleActive = async (vendedorId, isActive) => {
    try {
      const token = localStorage.getItem("access_token");
      const response = await axios.put(
        `${apiUrl}/api/v1/vendedor/${vendedorId}/active`,
        { is_active: isActive },
        {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        }
      );

      if (response.status === 200) {
        setVendedores((prevVendedores) =>
          prevVendedores.map((vendedor) =>
            vendedor.id === vendedorId
              ? { ...vendedor, is_active: isActive }
              : vendedor
          )
        );
        toast.success("Estado del vendedor actualizado exitosamente.");
      }
    } catch (error) {
      console.error("Error al actualizar el estado del vendedor:", error);
      toast.error("Error al actualizar el estado del vendedor.");
    }
  };

  useEffect(() => {
    const fetchVendedores = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(`${apiUrl}/api/v1/vendedor/`, {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        });

        const data = response.data;
        setVendedores(data);
      } catch (error) {
        toast.error("Error al cargar vendedores.");
      }
    };

    fetchVendedores();
  }, []);

  const handleEdit = (vendedor) => {
    setIsModalOpen(true);
    setModalType("edit");
    setSelectedVendedor(vendedor);
  };

  const handleDelete = async (vendedorId) => {
    try {
      if (
        window.confirm("¿Estás seguro de que deseas eliminar este vendedor?")
      ) {
        const token = localStorage.getItem("access_token");
        const response = await axios.delete(
          `${apiUrl}/api/v1/vendedor/${vendedorId}`,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );

        if (response.status === 200) {
          setVendedores(
            vendedores.filter((vendedor) => vendedor.id !== vendedorId)
          );
          toast.success("Vendedor eliminado exitosamente");
        }
      }
    } catch (error) {
      toast.error("Error al eliminar el vendedor");
    }
  };

  const handleAddNew = () => {
    setIsModalOpen(true);
    setModalType("add");
    setSelectedVendedor({
      nombre: "",
      cedula: "",
      telefono: "",
      email: "",
      is_active: true,
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    const token = localStorage.getItem("access_token");

    try {
      if (modalType === "add") {
        const response = await axios.post(
          `${apiUrl}/api/v1/vendedor/`,
          selectedVendedor,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );
        if (response.status === 200) {
          setVendedores([...vendedores, response.data]);
          toast.success("Vendedor agregado exitosamente");
        }
      } else if (modalType === "edit") {
        const response = await axios.put(
          `${apiUrl}/api/v1/vendedor/${selectedVendedor.id}`,
          selectedVendedor,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );
        if (response.status === 200) {
          setVendedores(
            vendedores.map((vendedor) =>
              vendedor.id === selectedVendedor.id ? response.data : vendedor
            )
          );
          toast.success("Vendedor actualizado exitosamente");
        }
      }
      setIsModalOpen(false);
    } catch (error) {
      toast.error("Error al guardar el vendedor");
    }
  };

  return (
    <Box
      component="div"
      sx={{
        maxWidth: "1050px",
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        sx={{
          marginTop: "10px",
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h5"
      >
        Vendedores
      </Typography>
      <Button
        variant="contained"
        color="primary"
        startIcon={<AddIcon />}
        onClick={handleAddNew}
        sx={{ marginBottom: "20px", float: "right" }}
      >
        Nuevo Vendedor
      </Button>

      <TableContainer component={Paper}>
        <Table>
          <TableHead sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}>
            <TableRow>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Nombre
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Cédula
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Teléfono
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Email
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Activo
              </TableCell>
              <TableCell
                align="right"
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Acciones
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {vendedores.map((vendedor) => (
              <TableRow key={vendedor.id}>
                <TableCell>{vendedor.nombre}</TableCell>
                <TableCell>{vendedor.cedula}</TableCell>
                <TableCell>{vendedor.telefono}</TableCell>
                <TableCell>{vendedor.email}</TableCell>
                <TableCell>
                  <Switch
                    checked={Boolean(vendedor.is_active)}
                    onChange={(e) =>
                      handleToggleActive(vendedor.id, e.target.checked)
                    }
                    color="primary"
                  />
                </TableCell>
                <TableCell align="right">
                  <IconButton onClick={() => handleEdit(vendedor)}>
                    <EditIcon style={{ color: "green" }} />
                  </IconButton>
                  <IconButton onClick={() => handleDelete(vendedor.id)}>
                    <DeleteIcon style={{ color: "#de2323" }} />
                  </IconButton>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>
      <Button
        variant="contained"
        component={Link}
        to="/administracion"
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>

      {/* Modal para agregar/editar vendedor */}
      <Modal
        open={isModalOpen}
        onClose={() => setIsModalOpen(false)}
        closeAfterTransition
        slots={{ backdrop: Backdrop }}
        slotProps={{
          backdrop: {
            timeout: 1500,
          },
        }}
      >
        <Fade in={isModalOpen}>
          <Box
            sx={{
              backgroundColor: "white",
              padding: 3,
              margin: "auto",
              width: 320,
              position: "absolute",
              top: "50%",
              left: "63%",
              transform: "translate(-50%, -50%)",
              boxShadow: 24,
              borderRadius: 2,
            }}
          >
            <Typography variant="h6" component="h2">
              {modalType === "add" ? "Agregar Vendedor" : "Editar Vendedor"}
            </Typography>
            <form onSubmit={handleSubmit}>
              <TextField
                fullWidth
                label="Nombre"
                value={selectedVendedor.nombre}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    nombre: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Cédula"
                value={selectedVendedor.cedula || ""}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    cedula: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Teléfono"
                value={selectedVendedor.telefono || ""}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    telefono: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Email"
                value={selectedVendedor.email}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    email: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <Box sx={{ textAlign: "right", mt: 2 }}>
                <Button
                  fullWidth
                  type="submit"
                  variant="contained"
                  color="primary"
                >
                  {modalType === "add" ? "Agregar" : "Guardar Cambios"}
                </Button>
              </Box>
            </form>
          </Box>
        </Fade>
      </Modal>
    </Box>
  );
};

export default ListVendedores;
