import { useEffect, useState } from "react";
import {
  Card,
  CardContent,
  Typography,
  Grid,
  Box,
  CircularProgress,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";

const colors = ["#0088FE", "#00C49F", "#FFBB28", "#FF8042"];
const emptyColor = ["#CCCCCC"]; // Color gris para gráfico vacío

const VentasDashboard = () => {
  const [stats, setStats] = useState(null);
  const [loading, setLoading] = useState(true);
  const apiUrl = process.env.REACT_APP_API_URL || "http://localhost:8001";

  useEffect(() => {
    const token = localStorage.getItem("access_token");
    if (!token) {
      toast.error("No se encontró el token de autenticación.");
      setLoading(false);
      return;
    }

    axios
      .get(`${apiUrl}/api/v1/vendedores/ventas/dashboard`, {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      })
      .then((res) => {
        //console.log("Datos recibidos:", res.data); // Para depuración
        setStats(res.data);
        setLoading(false);
      })
      .catch((err) => {
        toast.error(
          err.response?.status === 401
            ? "Sesión expirada, por favor inicia sesión nuevamente."
            : "Error al cargar datos."
        );
        setLoading(false);
      });
  }, []);

  if (loading) {
    return (
      <Box display="flex" justifyContent="center" mt={5}>
        <CircularProgress />
      </Box>
    );
  }

  if (!stats) {
    return (
      <Box display="flex" justifyContent="center" mt={5}>
        <Typography variant="h6" color="error">
          No se pudieron cargar los datos.
        </Typography>
      </Box>
    );
  }

  const resumen = [
    {
      titulo: "Solicitudes Creadas",
      valor: stats.solicitudes - stats.contratos,
    },
    {
      titulo: "Contratos Activos",
      valor: stats.contratos,
    },
    {
      titulo: "Solicitudes Caducadas",
      valor: stats.solicitudes_caducadas,
    },
    {
      titulo: "Total Ingresos (USD)",
      valor: `$ ${stats.total_ingresos.toLocaleString()}`, // Mostrar en dólares
    },
  ];

  return (
    <Box>
      <Grid container spacing={2}>
        {resumen.map((item, idx) => (
          <Grid item xs={12} sm={6} md={3} key={idx}>
            <Card sx={{ borderRight: "6px solid #0A4575", boxShadow: 3 }}>
              <CardContent>
                <Typography
                  sx={{
                    fontFamily: "Bakbak One, sans-serif",
                    color: "#0A4575",
                  }}
                >
                  {item.titulo}
                </Typography>
                <Typography variant="h5" fontWeight="bold" textAlign="end">
                  {item.valor}
                </Typography>
              </CardContent>
            </Card>
          </Grid>
        ))}
      </Grid>
      <Typography
        variant="h4"
        gutterBottom
        sx={{ marginTop: "15px", marginLeft: "15px", color: "#ffffff" }}
      >
        Bienvenido al Módulo Administrativo
      </Typography>
      <Typography
        variant="body1"
        sx={{ marginLeft: "15px", marginBottom: "1rem", color: "#ffffff" }}
      >
        Aquí podrás gestionar todos los servicios.
      </Typography>
    </Box>
  );
};

export default VentasDashboard;
