import React, { useEffect, useState } from "react";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Switch,
  TablePagination,
  Button,
} from "@mui/material";
import { ArrowBack } from "@mui/icons-material";
import axios from "axios";
import { Link } from "react-router-dom";
import { toast } from "react-toastify";

const apiUrl = process.env.REACT_APP_API_URL;

const AdminSolicitudesPage = () => {
  const [solicitudes, setSolicitudes] = useState({
    items: [],
    total: 0,
    page: 0,
    per_page: 0,
  });
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [loading, setLoading] = useState(false);

  // 💡 1. Mover la función fuera del useEffect
  const fetchSolicitudes = async () => {
    try {
      setLoading(true);
      const token = localStorage.getItem("access_token");
      if (!token) {
        toast.error("No se encontró el token de autenticación.");
        return;
      }

      const config = {
        headers: { Authorization: `Bearer ${token}` },
      };

      const response = await axios.get(
        `${apiUrl}/api/v1/admin/solicitudes?page=${
          page + 1
        }&per_page=${rowsPerPage}`,
        config
      );

      //console.log("Respuesta del endpoint /solicitudes:", response.data);
      setSolicitudes(response.data);
    } catch (error) {
      toast.error("Error al obtener la lista de solicitudes.");
      console.error("Error al obtener solicitudes:", error);
      setSolicitudes({ items: [], total: 0, page: 0, per_page: 0 });
    } finally {
      setLoading(false);
    }
  };

  // 💡 2. Ahora sí puedes usarlo dentro del useEffect
  useEffect(() => {
    fetchSolicitudes();
  }, [page, rowsPerPage]);

  // Manejar cambio del switch para convertir solicitud en contrato
  const handleConvertirContrato = async (solicitudId, checked) => {
    if (!checked) return; // Solo permitir activar el switch (no desactivar)

    try {
      const token = localStorage.getItem("access_token");
      const config = {
        headers: { Authorization: `Bearer ${token}` },
      };

      const payload = {
        SolicitudId: solicitudId,
      };

      await axios.post(
        `${apiUrl}/api/v1/admin/convertir-contrato`,
        payload,
        config
      );

      // Actualizar estado local
      setSolicitudes((prev) => ({
        ...prev,
        items: prev.items.map((cliente) => ({
          ...cliente,
          contratos: cliente.solicitudes.map((solicitud) =>
            solicitud.id === solicitudId
              ? { ...solicitud, estado: "contrato", estadoPago: "contrato" }
              : solicitud
          ),
        })),
      }));

      toast.success(
        `Solicitud ${solicitudId} convertida a contrato exitosamente`
      );

      await fetchSolicitudes();
    } catch (error) {
      toast.error(
        `Error al convertir la solicitud: ${
          error.response?.data?.detail || error.message
        }`
      );
      console.error("Error al convertir solicitud:", error);
    }
  };

  const handleChangePage = (event, newPage) => {
    setPage(newPage);
  };

  const handleChangeRowsPerPage = (event) => {
    setRowsPerPage(parseInt(event.target.value, 10));
    setPage(0);
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
        maxWidth: "1200px",
      }}
    >
      <Typography
        variant="h6"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        GESTIONAR SOLICITUDES A CONTRATOS
      </Typography>
      <TableContainer component={Paper} sx={{ boxShadow: 3, width: "100%" }}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Nº Solicitud</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
              <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
              <TableCell sx={{ color: "#fff" }}>Fecha de Creación</TableCell>
              <TableCell sx={{ color: "#fff" }}>Monto Total</TableCell>
              <TableCell sx={{ color: "#fff" }}>Modalidad de Pago</TableCell>
              <TableCell sx={{ color: "#fff" }}>Estado</TableCell>
              <TableCell sx={{ color: "#fff" }}>Acción</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {solicitudes.items.length > 0 ? (
              solicitudes.items.flatMap((cliente) =>
                cliente.solicitudes.map((solicitud) => (
                  <TableRow key={solicitud.id}>
                    <TableCell>{solicitud.numero_solicitud}</TableCell>
                    <TableCell>{cliente.nombre}</TableCell>
                    <TableCell>{solicitud.tipo_servicio}</TableCell>
                    <TableCell>
                      {new Date(solicitud.fecha_creacion).toLocaleDateString(
                        "es-ES",
                        {
                          day: "2-digit",
                          month: "2-digit",
                          year: "numeric",
                        }
                      )}
                    </TableCell>
                    <TableCell>${solicitud.precioPlan || 0}</TableCell>
                    <TableCell>
                      {solicitud.detalles && solicitud.detalles.length > 0
                        ? solicitud.detalles[0].detalles_adicionales
                            .modalidadPago
                        : "No especificado"}
                    </TableCell>
                    <TableCell>{solicitud.estado || "Sin estado"}</TableCell>
                    <TableCell>
                      <Switch
                        checked={solicitud.estado === "contrato"}
                        onChange={(e) =>
                          handleConvertirContrato(
                            solicitud.id,
                            e.target.checked
                          )
                        }
                        disabled={solicitud.estado === "contrato" || loading}
                        color="primary"
                        sx={{
                          "& .MuiSwitch-switchBase.Mui-checked": {
                            color: "#0A4575",
                          },
                          "& .MuiSwitch-switchBase.Mui-checked + .MuiSwitch-track":
                            {
                              backgroundColor: "#0A4575",
                            },
                        }}
                      />
                    </TableCell>
                  </TableRow>
                ))
              )
            ) : (
              <TableRow>
                <TableCell colSpan={8} align="center">
                  No hay solicitudes disponibles.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
        <TablePagination
          rowsPerPageOptions={[5, 10, 25]}
          component="div"
          count={solicitudes.total || 0}
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={handleChangePage}
          onRowsPerPageChange={handleChangeRowsPerPage}
          labelRowsPerPage="Filas por página:"
          labelDisplayedRows={({ from, to, count }) =>
            `${from}-${to} de ${count}`
          }
        />
      </TableContainer>{" "}
      <Button
        variant="contained"
        component={Link}
        to="/administracion"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
    </Box>
  );
};

export default AdminSolicitudesPage;
