import { useEffect, useState } from "react";
import axios from "axios";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  TextField,
  MenuItem,
  TablePagination,
  Stack,
  Button,
} from "@mui/material";
import { ArrowBack } from "@mui/icons-material";
import ContentPasteSearchIcon from "@mui/icons-material/ContentPasteSearch";
import { Link } from "react-router-dom";
import dayjs from "dayjs";

const Auditoria = () => {
  const [registros, setRegistros] = useState([]);
  const [rol, setRol] = useState("");
  const [fechaDesde, setFechaDesde] = useState("");
  const [fechaHasta, setFechaHasta] = useState("");
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const apiUrl = process.env.REACT_APP_API_URL || "http://localhost:8000";

  const fetchAuditoria = async () => {
    try {
      const res = await axios.get(`${apiUrl}/api/v1/admin/auditoria`, {
        params: {
          rol,
          desde: fechaDesde,
          hasta: fechaHasta,
        },
      });
      setRegistros(res.data || []);
    } catch (err) {
      console.error("Error al obtener la bitácora", err);
    }
  };

  useEffect(() => {
    fetchAuditoria();
  }, []);

  const handleChangePage = (event, newPage) => {
    setPage(newPage);
  };

  const handleChangeRowsPerPage = (event) => {
    setRowsPerPage(parseInt(event.target.value, 10));
    setPage(0);
  };

  const registrosFiltrados = registros.slice(
    page * rowsPerPage,
    page * rowsPerPage + rowsPerPage
  );

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 2,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Box
        sx={{
          display: "flex",
          alignItems: "center",
          mb: 2,
        }}
      >
        <ContentPasteSearchIcon
          sx={{ color: "#0A4575", fontSize: 30, mr: 1 }}
        />{" "}
        {/* Ícono de usuarios */}
        <Typography
          sx={{
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
          variant="h5"
        >
          AUDITORIA DE MOVIMIENTOS
        </Typography>
      </Box>

      {/* Filtros */}
      <Stack direction={{ xs: "column", sm: "row" }} spacing={2} mb={2}>
        <TextField
          label="Desde"
          type="date"
          size="small"
          value={fechaDesde}
          onChange={(e) => setFechaDesde(e.target.value)}
          InputLabelProps={{ shrink: true }}
          sx={{ boxShadow: 3 }}
        />
        <TextField
          label="Hasta"
          size="small"
          type="date"
          value={fechaHasta}
          onChange={(e) => setFechaHasta(e.target.value)}
          InputLabelProps={{ shrink: true }}
          sx={{ boxShadow: 3 }}
        />
        <TextField
          select
          label="Rol"
          size="small"
          value={rol}
          onChange={(e) => setRol(e.target.value)}
          sx={{ minWidth: 150, boxShadow: 3 }}
        >
          <MenuItem value="">Todos</MenuItem>
          <MenuItem value="administrador">Administrador</MenuItem>
          <MenuItem value="supervisor">Supervisor</MenuItem>
          <MenuItem value="vendedor">Vendedor</MenuItem>
          <MenuItem value="cliente">Cliente</MenuItem>
        </TextField>
        <Button
          variant="contained"
          size="small"
          onClick={fetchAuditoria}
          sx={{ whiteSpace: "nowrap" }}
        >
          Aplicar filtros
        </Button>
      </Stack>

      {/* Tabla */}
      <TableContainer component={Paper} elevation={3}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0b5dafff" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Usuario</TableCell>
              <TableCell sx={{ color: "#fff" }}>Rol</TableCell>
              <TableCell sx={{ color: "#fff" }}>Acción</TableCell>
              <TableCell sx={{ color: "#fff" }}>Descripción</TableCell>
              <TableCell sx={{ color: "#fff" }}>Fecha</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {registrosFiltrados.map((r, index) => (
              <TableRow key={index}>
                <TableCell>{r.username}</TableCell>
                <TableCell>{r.rol}</TableCell>
                <TableCell>{r.accion}</TableCell>
                <TableCell>{r.descripcion}</TableCell>
                <TableCell>
                  {dayjs(r.fecha).format("DD/MM/YYYY HH:mm")}
                </TableCell>
              </TableRow>
            ))}
            {registrosFiltrados.length === 0 && (
              <TableRow>
                <TableCell colSpan={6} align="center">
                  No hay registros para los filtros seleccionados.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
        <TablePagination
          component="div"
          count={registros.length}
          page={page}
          onPageChange={handleChangePage}
          rowsPerPage={rowsPerPage}
          onRowsPerPageChange={handleChangeRowsPerPage}
          labelRowsPerPage="Filas por página"
        />
      </TableContainer>
      <Button
        variant="contained"
        component={Link}
        to="/administracion"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
    </Box>
  );
};

export default Auditoria;
