import React, { useEffect, useState } from "react";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Typography,
  Box,
  CircularProgress,
  IconButton,
  Collapse,
  TablePagination,
  Button,
} from "@mui/material";
import { ArrowBack, ExpandMore, ExpandLess } from "@mui/icons-material";
import axios from "axios";
import { Link } from "react-router-dom";
import { toast } from "react-toastify";
import { jwtDecode } from "jwt-decode";

const apiUrl = process.env.REACT_APP_API_URL || "http://localhost:8000";

const ContratosTable = () => {
  const [contratos, setContratos] = useState([]);
  const [loading, setLoading] = useState(true);
  const [openRows, setOpenRows] = useState({});
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [totalCount, setTotalCount] = useState(0);

  useEffect(() => {
    const fetchContratosActivos = async () => {
      try {
        const token = localStorage.getItem("access_token");
        if (!token) {
          toast.error("No se encontró el token de autenticación.");
          setLoading(false);
          return;
        }

        const decoded = jwtDecode(token);
        const vendedorId = decoded.sub ? Number(decoded.sub) : null;

        if (!vendedorId) {
          toast.error("No se pudo obtener el ID del usuario.");
          setLoading(false);
          return;
        }

        const config = {
          headers: { Authorization: `Bearer ${token}` },
          params: { skip: page * rowsPerPage, limit: rowsPerPage },
        };

        const response = await axios.get(
          `${apiUrl}/api/v1/admin/contratos/activos`,
          config
        );

        if (!Array.isArray(response.data)) {
          throw new Error("La respuesta de la API no es un arreglo.");
        }

        setContratos(response.data);
        // Nota: Como el endpoint no devuelve el total, asumimos que hay más páginas si se devuelven rowsPerPage elementos
        setTotalCount(
          response.data.length === rowsPerPage
            ? (page + 1) * rowsPerPage + 1
            : page * rowsPerPage + response.data.length
        );
      } catch (error) {
        console.error("Error al obtener contratos activos:", error);
        toast.error(
          error.response?.data?.detail ||
            "Error al cargar los contratos activos."
        );
      } finally {
        setLoading(false);
      }
    };

    fetchContratosActivos();
  }, [page, rowsPerPage]);

  const handleToggleRow = (id) => {
    setOpenRows((prev) => ({ ...prev, [id]: !prev[id] }));
  };

  const handleChangePage = (event, newPage) => {
    setPage(newPage);
  };

  const handleChangeRowsPerPage = (event) => {
    setRowsPerPage(parseInt(event.target.value, 10));
    setPage(0);
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
        maxWidth: "1200px",
      }}
    >
      <Typography
        variant="h6"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        CONTRATOS ACTIVOS
      </Typography>
      {loading ? (
        <Box sx={{ display: "flex", justifyContent: "center", mt: 4 }}>
          <CircularProgress />
        </Box>
      ) : contratos.length === 0 ? (
        <Typography variant="body1" color="text.secondary">
          No hay contratos disponibles.
        </Typography>
      ) : (
        <>
          <TableContainer component={Paper}>
            <Table size="small">
              <TableHead sx={{ backgroundColor: "#0A4575" }}>
                <TableRow>
                  <TableCell sx={{ color: "#fff" }}>
                    Número de Contrato
                  </TableCell>
                  <TableCell sx={{ color: "#fff" }}>Vendedor</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Método de Pago</TableCell>
                  <TableCell sx={{ color: "#fff" }}>
                    Fecha de Creación
                  </TableCell>
                  <TableCell sx={{ color: "#fff" }}>Estado</TableCell>
                  <TableCell sx={{ color: "#fff" }}></TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {contratos.map((contrato) => (
                  <React.Fragment key={contrato.id}>
                    <TableRow>
                      <TableCell>{contrato.numero_contrato || "N/A"}</TableCell>
                      <TableCell>{contrato.vendedor_nombre || "N/A"}</TableCell>
                      <TableCell>{contrato.cliente?.nombre || "N/A"}</TableCell>
                      <TableCell>{contrato.tipo_servicio || "N/A"}</TableCell>
                      <TableCell>{contrato.modalidadPago || "N/A"}</TableCell>
                      <TableCell>
                        {contrato.fecha_creacion
                          ? new Date(
                              contrato.fecha_creacion
                            ).toLocaleDateString()
                          : "N/A"}
                      </TableCell>
                      <TableCell>
                        <Button
                          variant="contained"
                          sx={{ backgroundColor: "#4CAF50" }}
                        >
                          CONTRATO
                        </Button>
                      </TableCell>
                      <TableCell>
                        <IconButton
                          onClick={() => handleToggleRow(contrato.id)}
                        >
                          {openRows[contrato.id] ? (
                            <ExpandLess />
                          ) : (
                            <ExpandMore />
                          )}
                        </IconButton>
                      </TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell
                        colSpan={9}
                        style={{ paddingBottom: 0, paddingTop: 0 }}
                      >
                        <Collapse
                          in={openRows[contrato.id]}
                          timeout="auto"
                          unmountOnExit
                        >
                          <Box sx={{ margin: 1 }}>
                            <Typography
                              variant="h6"
                              gutterBottom
                              sx={{
                                fontFamily: "Bakbak One, sans-serif",
                                color: "#0A4575",
                              }}
                            >
                              Detalles del Contrato
                            </Typography>
                            {contrato.detalles &&
                            contrato.detalles.length > 0 ? (
                              <Table size="small">
                                <TableHead sx={{ backgroundColor: "#0A4575" }}>
                                  <TableRow>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Código
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Descripción
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Método de Pago
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Pago Inicial
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Cuotas
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Monto por Cuotas
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Descuento
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Precio con Descuento
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Precio Total
                                    </TableCell>
                                  </TableRow>
                                </TableHead>
                                <TableBody>
                                  {contrato.detalles.map((detalle) => (
                                    <TableRow key={detalle.id}>
                                      <TableCell>
                                        {detalle.codigo || "N/A"}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.detalles_adicionales
                                          .descripcion || "N/A"}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.metodoPago || "N/A"}
                                      </TableCell>
                                      <TableCell>
                                        ${detalle.pagoInicial}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.cuotas || "$0"}
                                      </TableCell>
                                      <TableCell>
                                        ${detalle.montoPorCuotas}
                                      </TableCell>
                                      <TableCell>
                                        ${detalle.descuento}
                                      </TableCell>
                                      <TableCell>
                                        ${detalle.precioConDescuento}
                                      </TableCell>
                                      <TableCell>
                                        ${detalle.precioPlan}
                                      </TableCell>
                                    </TableRow>
                                  ))}
                                </TableBody>
                              </Table>
                            ) : (
                              <Typography
                                variant="body2"
                                color="text.secondary"
                              >
                                No hay detalles disponibles para este contrato.
                              </Typography>
                            )}
                          </Box>
                        </Collapse>
                      </TableCell>
                    </TableRow>
                  </React.Fragment>
                ))}
              </TableBody>
            </Table>
          </TableContainer>
          <TablePagination
            rowsPerPageOptions={[5, 10, 25]}
            component="div"
            count={totalCount}
            rowsPerPage={rowsPerPage}
            page={page}
            onPageChange={handleChangePage}
            onRowsPerPageChange={handleChangeRowsPerPage}
            labelRowsPerPage="Filas por página:"
            labelDisplayedRows={({ from, to, count }) =>
              `${from}–${to} de ${count}`
            }
          />
          <Button
            variant="contained"
            component={Link}
            to="/administracion"
            startIcon={<ArrowBack />}
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Volver
          </Button>
        </>
      )}
    </Box>
  );
};

export default ContratosTable;
