import React, { useState, useEffect } from "react";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Switch,
  Paper,
  IconButton,
  Button,
  MenuItem,
  TextField,
  Typography,
  Modal,
  Fade,
  Backdrop,
  Box,
  InputAdornment,
} from "@mui/material";
import {
  People as PeopleIcon,
  PersonAdd,
  Edit as EditIcon,
  Delete as DeleteIcon,
  ArrowBack,
  Visibility,
  VisibilityOff,
} from "@mui/icons-material";
import axios from "axios";
import { Link } from "react-router-dom";
import { toast } from "react-toastify";

const apiUrl = process.env.REACT_APP_API_URL;

// Áreas predefinidas del consultorio
const Rolusername = [
  { value: "admin", label: "Administrador" },
  { value: "supervisor", label: "Supervisor" },
  { value: "cliente", label: "Cliente" },
  { value: "vendedor", label: "Vendedor/a" },
];

const GestionarUsuarios = () => {
  const [users, setUsers] = useState([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [showPassword, setShowPassword] = useState(false);
  const [modalType, setModalType] = useState("add");
  const [selectedUser, setSelectedUser] = useState({
    username: "",
    email: "",
    rol: "",
    password: "",
    cedula: "",
    direccion: "",
    telefono: "",
    is_active: true,
  });

  useEffect(() => {
    const fetchUsers = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(`${apiUrl}/api/v1/usuarios`, {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        });
        setUsers(response.data);
      } catch (error) {
        toast.error("Error al cargar usuarios.");
      }
    };

    fetchUsers();
  }, []);

  const handleToggleActive = async (userId, isActive) => {
    try {
      const token = localStorage.getItem("access_token");
      const response = await axios.put(
        `${apiUrl}/api/v1/usuarios/${userId}/active`,
        { is_active: isActive },
        {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        }
      );

      if (response.status === 200) {
        setUsers((prevUsers) =>
          prevUsers.map((user) =>
            user.id === userId ? { ...user, is_active: isActive } : user
          )
        );
        toast.success("Estado del usuario actualizado exitosamente.");
      }
    } catch (error) {
      console.error("Error al actualizar el estado del usuario:", error);
      toast.error("Error al actualizar el estado del usuario.");
    }
  };

  const handleEdit = (user) => {
    setIsModalOpen(true);
    setModalType("edit");
    setSelectedUser({
      ...user,
      password: "", // Limpiar el campo de contraseña en modo edición
    });
  };

  const handleDelete = async (userId) => {
    try {
      if (
        window.confirm("¿Estás seguro de que deseas eliminar este usuario?")
      ) {
        const token = localStorage.getItem("access_token");
        const response = await axios.delete(
          `${apiUrl}/api/v1/usuarios/${userId}`,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );

        if (response.status === 200) {
          setUsers(users.filter((user) => user.id !== userId));
          toast.success("Usuario eliminado exitosamente");
        }
      }
    } catch (error) {
      toast.error("Error al eliminar el usuario");
    }
  };

  const handleAddNew = () => {
    setIsModalOpen(true);
    setModalType("add");
    setSelectedUser({
      username: "",
      email: "",
      rol: "",
      password: "",
      cedula: "",
      direccion: "",
      telefono: "",
      is_active: true,
    });
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    const token = localStorage.getItem("access_token");

    try {
      if (modalType === "add") {
        const response = await axios.post(
          `${apiUrl}/api/v1/usuarios/`,
          {
            vendedor_id: selectedUser.userId,
            username: selectedUser.username,
            email: selectedUser.email,
            rol: selectedUser.rol,
            password: selectedUser.password,
            cedula: selectedUser.cedula || null,
            direccion: selectedUser.direccion || null,
            telefono: selectedUser.telefono || null,
          },
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );
        if (response.status === 200) {
          setUsers([...users, response.data]);
          toast.success("Usuario agregado exitosamente");
        }
      } else if (modalType === "edit") {
        const response = await axios.put(
          `${apiUrl}/api/v1/usuarios/${selectedUser.id}`,
          {
            username: selectedUser.username,
            email: selectedUser.email,
            rol: selectedUser.rol,
            password: selectedUser.password || undefined, // No enviar contraseña si está vacía
          },
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );
        if (response.status === 200) {
          setUsers(
            users.map((user) =>
              user.id === selectedUser.id ? response.data : user
            )
          );
          toast.success("Usuario actualizado exitosamente");
        }
      }
      setIsModalOpen(false);
    } catch (error) {
      toast.error("Error al guardar el usuario");
    }
  };

  return (
    <Box
      component="div"
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          mb: 3,
        }}
      >
        <Box
          sx={{
            display: "flex",
            alignItems: "center",
          }}
        >
          <PeopleIcon sx={{ color: "#0A4575", fontSize: 30, mr: 1 }} />{" "}
          {/* Ícono de usuarios */}
          <Typography
            sx={{
              fontFamily: "Bakbak One, sans-serif",
              color: "#0A4575",
            }}
            variant="h5"
          >
            USUARIOS
          </Typography>
        </Box>
        <Box sx={{ display: "flex", gap: 2 }}>
          <Button
            variant="contained"
            color="primary"
            startIcon={<PersonAdd />}
            onClick={handleAddNew}
            sx={{ float: "right" }}
          >
            Nuevo Usuario
          </Button>
        </Box>
      </Box>
      <TableContainer component={Paper} elevation={3}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}>
            <TableRow>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Nombre de Usuario
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Email
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Rol
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Activo
              </TableCell>
              <TableCell
                align="right"
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Editar / Eliminar
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {users.map((user) => (
              <TableRow key={user.id}>
                <TableCell>{user.username}</TableCell>
                <TableCell>{user.email}</TableCell>
                <TableCell>{user.rol}</TableCell>
                <TableCell>
                  <Switch
                    checked={Boolean(user.is_active)}
                    onChange={(e) =>
                      handleToggleActive(user.id, e.target.checked)
                    }
                    color="primary"
                  />
                </TableCell>
                <TableCell align="right">
                  <IconButton onClick={() => handleEdit(user)}>
                    <EditIcon style={{ color: "green" }} />
                  </IconButton>
                  <IconButton onClick={() => handleDelete(user.id)}>
                    <DeleteIcon style={{ color: "#de2323" }} />
                  </IconButton>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>
      <Button
        variant="contained"
        component={Link}
        to="/administracion"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
      <Modal
        open={isModalOpen}
        onClose={() => setIsModalOpen(false)}
        closeAfterTransition
        slots={{ backdrop: Backdrop }}
        slotProps={{
          backdrop: {
            timeout: 1500,
          },
        }}
      >
        <Fade in={isModalOpen}>
          <Box
            sx={{
              backgroundColor: "white",
              padding: 3,
              margin: "auto",
              width: 320,
              position: "absolute",
              top: "50%",
              left: "63%",
              transform: "translate(-50%, -50%)",
              boxShadow: 24,
              borderRadius: 2,
            }}
          >
            <Typography variant="h6" component="h2">
              {modalType === "add" ? "Agregar Usuario" : "Editar Usuario"}
            </Typography>
            <form onSubmit={handleSubmit}>
              <TextField
                fullWidth
                label="Nombre y Apellido"
                value={selectedUser.username}
                size="small"
                required
                onChange={(e) =>
                  setSelectedUser({ ...selectedUser, username: e.target.value })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Email"
                value={selectedUser.email}
                size="small"
                required
                onChange={(e) =>
                  setSelectedUser({ ...selectedUser, email: e.target.value })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                select
                label="Rol"
                value={selectedUser.rol}
                size="small"
                required
                onChange={(e) =>
                  setSelectedUser({ ...selectedUser, rol: e.target.value })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              >
                {Rolusername.map((rol) => (
                  <MenuItem key={rol.value} value={rol.value}>
                    {rol.label}
                  </MenuItem>
                ))}
              </TextField>
              {modalType === "add" &&
                (selectedUser.rol === "vendedor" ||
                  selectedUser.rol === "supervisor") && (
                  <>
                    <TextField
                      fullWidth
                      label="Cédula"
                      value={selectedUser.cedula || ""}
                      size="small"
                      required
                      onChange={(e) =>
                        setSelectedUser({
                          ...selectedUser,
                          cedula: e.target.value,
                        })
                      }
                      margin="normal"
                      sx={{ boxShadow: 3 }}
                    />
                    <TextField
                      fullWidth
                      label="Dirección"
                      value={selectedUser.direccion || ""}
                      size="small"
                      required
                      onChange={(e) =>
                        setSelectedUser({
                          ...selectedUser,
                          direccion: e.target.value,
                        })
                      }
                      margin="normal"
                      sx={{ boxShadow: 3 }}
                    />
                    <TextField
                      fullWidth
                      label="Teléfono"
                      value={selectedUser.telefono || ""}
                      size="small"
                      required
                      onChange={(e) =>
                        setSelectedUser({
                          ...selectedUser,
                          telefono: e.target.value,
                        })
                      }
                      margin="normal"
                      sx={{ boxShadow: 3 }}
                    />
                  </>
                )}
              <TextField
                type={showPassword ? "text" : "password"}
                label="Contraseña"
                size="small"
                variant="outlined"
                sx={{ boxShadow: 3 }}
                fullWidth
                value={selectedUser.password || ""}
                onChange={(e) =>
                  setSelectedUser({
                    ...selectedUser,
                    password: e.target.value,
                  })
                }
                margin="normal"
                required={modalType === "add"}
                InputProps={{
                  endAdornment: (
                    <InputAdornment position="end">
                      <IconButton
                        onClick={() => setShowPassword(!showPassword)}
                        edge="end"
                        aria-label="toggle password visibility"
                      >
                        {showPassword ? <VisibilityOff /> : <Visibility />}
                      </IconButton>
                    </InputAdornment>
                  ),
                }}
              />
              <Box sx={{ textAlign: "right", mt: 2 }}>
                <Button
                  fullWidth
                  type="submit"
                  variant="contained"
                  color="primary"
                >
                  {modalType === "add" ? "Agregar" : "Guardar Cambios"}
                </Button>
              </Box>
            </form>
          </Box>
        </Fade>
      </Modal>
    </Box>
  );
};

export default GestionarUsuarios;
