import React, { useState, useEffect } from "react";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Switch,
  Paper,
  IconButton,
  Button,
  TextField,
  Typography,
  Modal,
  Fade,
  Backdrop,
  Box,
  MenuItem,
} from "@mui/material";
import {
  People as PeopleIcon,
  Edit as EditIcon,
  Delete as DeleteIcon,
  ArrowBack,
} from "@mui/icons-material";
import axios from "axios";
import { Link } from "react-router-dom";
import { toast } from "react-toastify";

const apiUrl = process.env.REACT_APP_API_URL;

const GestionarVendedores = () => {
  const [vendedores, setVendedores] = useState([]);
  const [supervisores, setSupervisores] = useState([]);
  const [mensaje, setMensaje] = useState([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedVendedor, setSelectedVendedor] = useState({
    nombre: "",
    cedula: "",
    direccion: "",
    telefono: "",
    email: "",
    supervisor_id: null,
    is_active: true,
  });

  useEffect(() => {
    const fetchVendedores = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(`${apiUrl}/api/v1/vendedores`, {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        });

        setMensaje(response.data.mensaje);
        setVendedores(response.data.vendedores || []);
        // Filtrar solo los usuarios con rol "supervisor"
        setSupervisores(
          (response.data.usuarios || []).filter(
            (usuario) => usuario.rol === "supervisor"
          )
        );

        if (response.data.vendedores.length === 0) {
          toast.info(response.data.mensaje);
        }
      } catch (error) {
        const msg =
          error.response?.data?.mensaje || "Error al cargar vendedores";
        const errDetail = error.response?.data?.error || "";
        toast.error(`${msg}${errDetail ? ": " + errDetail : ""}`);
      }
    };

    fetchVendedores();
  }, []);

  const handleToggleActive = async (vendedorId, isActive) => {
    try {
      const token = localStorage.getItem("access_token");
      const response = await axios.put(
        `${apiUrl}/api/v1/vendedores/${vendedorId}/active`,
        { is_active: isActive },
        {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        }
      );

      if (response.status === 200) {
        setVendedores((prevVendedores) =>
          prevVendedores.map((vendedor) =>
            vendedor.id === vendedorId
              ? { ...vendedor, is_active: isActive }
              : vendedor
          )
        );
        toast.success("Estado del vendedor actualizado exitosamente.");
      }
    } catch (error) {
      console.error("Error al actualizar el estado del vendedor:", error);
      toast.error("Error al actualizar el estado del vendedor.");
    }
  };

  const handleEdit = (vendedor) => {
    setIsModalOpen(true);
    setSelectedVendedor(vendedor);
  };

  const handleDelete = async (vendedorId) => {
    try {
      if (
        window.confirm("¿Estás seguro de que deseas eliminar este vendedor?")
      ) {
        const token = localStorage.getItem("access_token");
        const response = await axios.delete(
          `${apiUrl}/api/v1/vendedores/${vendedorId}`,
          {
            headers: {
              Authorization: `Bearer ${token}`,
            },
          }
        );

        if (response.status === 200) {
          setVendedores(
            vendedores.filter((vendedor) => vendedor.id !== vendedorId)
          );
          toast.success("Vendedor eliminado exitosamente");
        }
      }
    } catch (error) {
      toast.error("Error al eliminar el vendedor");
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    const token = localStorage.getItem("access_token");

    const payload = {
      nombre: selectedVendedor.nombre,
      cedula: selectedVendedor.cedula,
      direccion: selectedVendedor.direccion,
      telefono: selectedVendedor.telefono,
      email: selectedVendedor.email,
      supervisor_id: selectedVendedor.supervisor_id || null, // Enviar supervisor_id como user_id para el backend
    };

    try {
      const response = await axios.put(
        `${apiUrl}/api/v1/vendedores/${selectedVendedor.id}`,
        payload,
        {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        }
      );
      if (response.status === 200) {
        setVendedores(
          vendedores.map((vendedor) =>
            vendedor.id === selectedVendedor.id ? response.data : vendedor
          )
        );
        toast.success("Vendedor actualizado exitosamente");
      }
      setIsModalOpen(false);
    } catch (error) {
      const msg =
        error.response?.data?.mensaje || "Error al guardar el vendedor";
      const errDetail = error.response?.data?.error || "";
      toast.error(`${msg}${errDetail ? ": " + errDetail : ""}`);
    }
  };

  return (
    <Box
      component="div"
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Box
        sx={{
          display: "flex",
          alignItems: "center",
          mb: 2,
        }}
      >
        <PeopleIcon sx={{ color: "#0A4575", fontSize: 30, mr: 1 }} />{" "}
        {/* Ícono de usuarios */}
        <Typography
          sx={{
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
          variant="h5"
        >
          VENDEDORES
        </Typography>
      </Box>

      <TableContainer component={Paper} elevation={3}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}>
            <TableRow>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Nombre
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Cédula
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Dirección
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Teléfono
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Email
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Supervisor
              </TableCell>
              <TableCell
                align="right"
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Editar
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {vendedores.length > 0 ? (
              vendedores.map((vendedor) => (
                <TableRow key={vendedor.id}>
                  <TableCell>{vendedor.nombre}</TableCell>
                  <TableCell>{vendedor.cedula}</TableCell>
                  <TableCell>{vendedor.direccion}</TableCell>
                  <TableCell>{vendedor.telefono}</TableCell>
                  <TableCell>{vendedor.email}</TableCell>
                  <TableCell>
                    {vendedor.supervisor_id
                      ? supervisores.find(
                          (sup) => sup.id === vendedor.supervisor_id
                        )?.username || "Sin supervisor"
                      : "Sin supervisor"}
                  </TableCell>

                  <TableCell align="right">
                    <IconButton onClick={() => handleEdit(vendedor)}>
                      <EditIcon style={{ color: "green" }} />
                    </IconButton>
                  </TableCell>
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={8} align="center">
                  No hay vendedores disponibles.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </TableContainer>
      <Button
        variant="contained"
        component={Link}
        to="/administracion"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>

      <Modal
        open={isModalOpen}
        onClose={() => setIsModalOpen(false)}
        closeAfterTransition
        slots={{ backdrop: Backdrop }}
        slotProps={{
          backdrop: {
            timeout: 1500,
          },
        }}
      >
        <Fade in={isModalOpen}>
          <Box
            sx={{
              backgroundColor: "white",
              padding: 3,
              margin: "auto",
              width: 320,
              position: "absolute",
              top: "50%",
              left: "63%",
              transform: "translate(-50%, -50%)",
              boxShadow: 24,
              borderRadius: 2,
            }}
          >
            <Typography variant="h6" component="h2">
              Editar Vendedor
            </Typography>
            <form onSubmit={handleSubmit}>
              <TextField
                fullWidth
                label="Nombre"
                value={selectedVendedor.nombre}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    nombre: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Cédula"
                value={selectedVendedor.cedula || ""}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    cedula: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Dirección"
                value={selectedVendedor.direccion || ""}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    direccion: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Teléfono"
                value={selectedVendedor.telefono || ""}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    telefono: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                label="Email"
                value={selectedVendedor.email}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    email: e.target.value,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              />
              <TextField
                fullWidth
                select
                label="Supervisor"
                value={selectedVendedor.supervisor_id || ""}
                size="small"
                onChange={(e) =>
                  setSelectedVendedor({
                    ...selectedVendedor,
                    supervisor_id: e.target.value
                      ? parseInt(e.target.value)
                      : null,
                  })
                }
                margin="normal"
                sx={{ boxShadow: 3 }}
              >
                <MenuItem value="">
                  <em>Sin supervisor</em>
                </MenuItem>
                {supervisores.map((supervisor) => (
                  <MenuItem key={supervisor.id} value={supervisor.id}>
                    {supervisor.username}
                  </MenuItem>
                ))}
              </TextField>
              <Box sx={{ textAlign: "right", mt: 2 }}>
                <Button
                  fullWidth
                  type="submit"
                  variant="contained"
                  color="primary"
                >
                  Guardar Cambios
                </Button>
              </Box>
            </form>
          </Box>
        </Fade>
      </Modal>
    </Box>
  );
};

export default GestionarVendedores;
