import React, { useState, useEffect } from "react";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Typography,
  Box,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  TextField,
  Button,
} from "@mui/material";
import { Inbox as InboxIcon } from "@mui/icons-material";
import axios from "axios";

const HistorialPagos = () => {
  const [contratos, setContratos] = useState([]);
  const [vendedores, setVendedores] = useState([]);
  const [contratosFiltrados, setContratosFiltrados] = useState([]);
  const [filtro, setFiltro] = useState("Todos");
  const [fechaInicio, setFechaInicio] = useState("");
  const [fechaFin, setFechaFin] = useState("");
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const apiUrl = process.env.REACT_APP_API_URL;

  useEffect(() => {
    const fetchContratos = async () => {
      try {
        const response = await axios.get(
          `${apiUrl}/api/v1/admin/historial-ingresos`,
          {
            headers: {
              Authorization: `Bearer ${localStorage.getItem("access_token")}`,
            },
          }
        );

        setContratos(response.data.contratos);
        setVendedores(response.data.vendedores);
        setLoading(false);
      } catch (err) {
        setError("Error al cargar los contratos");
        setLoading(false);
      }
    };

    fetchContratos();
  }, []);

  useEffect(() => {
    let filtrados = [...contratos];

    if (filtro !== "Todos") {
      filtrados = filtrados.filter(
        (contrato) => contrato.detalles[0]?.modalidadPago === filtro
      );
    }

    if (fechaInicio) {
      filtrados = filtrados.filter(
        (contrato) => new Date(contrato.fecha_creacion) >= new Date(fechaInicio)
      );
    }

    if (fechaFin) {
      filtrados = filtrados.filter(
        (contrato) => new Date(contrato.fecha_creacion) <= new Date(fechaFin)
      );
    }

    setContratosFiltrados(filtrados);
  }, [filtro, fechaInicio, fechaFin, contratos]);

  const totalContado = contratosFiltrados
    .filter((c) => c.detalles[0]?.modalidadPago === "De Contado")
    .reduce((sum, c) => sum + (c.monto_total || 0), 0);

  const totalFinanciado = contratosFiltrados
    .filter((c) => c.detalles[0]?.modalidadPago === "Financiado")
    .reduce((sum, c) => sum + (c.detalles[0]?.pagoInicial || 0), 0);

  const totalMonto = totalContado + totalFinanciado;
  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("es-ES", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
    });
  };

  if (loading) return <Typography>Cargando...</Typography>;
  if (error) return <Typography color="error">{error}</Typography>;

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          marginBottom: 2,
          gap: 2,
          flexWrap: "wrap",
        }}
      >
        <Typography
          variant="h6"
          sx={{
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
        >
          HISTORIAL DE PAGOS
        </Typography>
        <Box
          sx={{
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            gap: 2,
            flexWrap: "wrap",
          }}
        >
          <FormControl variant="outlined" size="small" sx={{ minWidth: 180 }}>
            <InputLabel id="filtro-label">Modalidad</InputLabel>
            <Select
              value={filtro}
              onChange={(e) => setFiltro(e.target.value)}
              label="Modalidad"
            >
              <MenuItem value="Todos">Todos</MenuItem>
              <MenuItem value="De Contado">De Contado</MenuItem>
              <MenuItem value="Financiado">Financiado</MenuItem>
            </Select>
          </FormControl>

          <TextField
            label="Fecha Inicio"
            type="date"
            size="small"
            InputLabelProps={{ shrink: true }}
            value={fechaInicio}
            onChange={(e) => setFechaInicio(e.target.value)}
          />

          <TextField
            label="Fecha Fin"
            type="date"
            size="small"
            InputLabelProps={{ shrink: true }}
            value={fechaFin}
            onChange={(e) => setFechaFin(e.target.value)}
          />
        </Box>
      </Box>
      <TableContainer component={Paper} elevation={3}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>No. de Contrato</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
              <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
              <TableCell sx={{ color: "#fff" }}>
                Fecha Inicio del Contrato
              </TableCell>
              <TableCell sx={{ color: "#fff" }}>Modalidad de Pago</TableCell>
              <TableCell sx={{ color: "#fff" }}>
                Pago Inicial - Plan Financiado
              </TableCell>
              <TableCell sx={{ color: "#fff" }}>Pago Plan de Contado</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {contratosFiltrados.length === 0 ? (
              <TableRow>
                <TableCell colSpan={7}>
                  <Box
                    sx={{
                      display: "flex",
                      flexDirection: "column",
                      alignItems: "center",
                      justifyContent: "center",
                      py: 2,
                      backgroundColor: "#f5f5f5",
                      borderRadius: 2,
                    }}
                  >
                    <InboxIcon sx={{ fontSize: 32, color: "#0A4575", mb: 1 }} />
                    <Typography variant="subtitle1" sx={{ color: "#000000" }}>
                      No hay historial de pagos disponibles.
                    </Typography>
                  </Box>
                </TableCell>
              </TableRow>
            ) : (
              contratosFiltrados.map((contrato) => (
                <TableRow key={contrato.id}>
                  <TableCell>{contrato.numero_contrato}</TableCell>
                  <TableCell>{contrato.cliente.nombre}</TableCell>
                  <TableCell>{contrato.tipo_servicio}</TableCell>
                  <TableCell>{formatDate(contrato.fecha_creacion)}</TableCell>
                  <TableCell>
                    <Button
                      variant="contained"
                      size="small"
                      sx={{
                        backgroundColor:
                          contrato.detalles[0]?.modalidadPago === "De Contado"
                            ? "#4CAF50"
                            : "#0A4575",
                        color: "#fff",
                        "&:hover": {
                          backgroundColor:
                            contrato.detalles[0]?.modalidadPago === "De Contado"
                              ? "darkgreen"
                              : "#1066acff",
                        },
                        textTransform: "none",
                      }}
                    >
                      {contrato.detalles[0]?.modalidadPago || "N/A"}
                    </Button>
                  </TableCell>
                  <TableCell>
                    ${contrato.detalles[0]?.pagoInicial?.toFixed(2) || "0.00"}
                  </TableCell>
                  <TableCell>
                    {contrato.detalles[0]?.modalidadPago === "De Contado"
                      ? `$${contrato.monto_total.toFixed(2)}`
                      : "$0.00"}
                  </TableCell>
                </TableRow>
              ))
            )}
          </TableBody>
        </Table>
      </TableContainer>

      {/* Totales */}
      <Box
        sx={{
          mt: 3,
          display: "flex",
          justifyContent: "end",
          gap: 6,
          flexWrap: "wrap",
        }}
      >
        <Typography sx={{ color: "#000" }}>
          Total De Contado:{" "}
          <Typography component="span" sx={{ fontWeight: "bold" }}>
            ${totalContado.toFixed(2)}
          </Typography>
        </Typography>
        <Typography sx={{ color: "#000" }}>
          Total Financiado:{" "}
          <Typography component="span" sx={{ fontWeight: "bold" }}>
            ${totalFinanciado.toFixed(2)}
          </Typography>
        </Typography>
        <Typography sx={{ color: "#000" }}>
          Suma total de Pagos:{" "}
          <Typography component="span" sx={{ fontWeight: "bold" }}>
            ${totalMonto.toFixed(2)}
          </Typography>
        </Typography>
      </Box>
    </Box>
  );
};

export default HistorialPagos;
