import React, { useEffect, useState } from "react";
import axios from "axios";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Typography,
  Box,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  TextField,
  Button,
} from "@mui/material";
import { Inbox as InboxIcon } from "@mui/icons-material";

export default function HistorialIngresos() {
  const [vendedores, setVendedores] = useState([]);
  const [contratos, setContratos] = useState([]);
  const [contratosFiltrados, setContratosFiltrados] = useState([]);
  const [supervisorSeleccionado, setSupervisorSeleccionado] = useState("");
  const [vendedorSeleccionado, setVendedorSeleccionado] = useState("");
  const [filtro, setFiltro] = useState("Todos");
  const [fechaInicio, setFechaInicio] = useState("");
  const [fechaFin, setFechaFin] = useState("");
  const [loading, setLoading] = useState(false);
  const apiUrl = process.env.REACT_APP_API_URL;

  useEffect(() => {
    fetchData();
  }, []);

  // Calcula resumen por vendedor a partir de contratos y lista de vendedores
  const calcularResumenVendedores = (vendedores, contratos) => {
    return vendedores.map((vendedor) => {
      // Filtrar contratos de este vendedor
      const contratosVendedor = contratos.filter(
        (c) => c.vendedor_id === vendedor.vendedor_id
      );

      // Totales
      const totalContratos = contratosVendedor.length;

      const totalFinanciados = contratosVendedor
        .filter((c) => c?.modalidadPago === "Financiado")
        .reduce((sum, c) => sum + (c.detalles[0]?.pagoInicial || 0), 0);

      const totalContados = contratosVendedor
        .filter((c) => c?.modalidadPago === "De Contado")
        .reduce((sum, c) => sum + (c.precioPlan || 0), 0);

      const totalIngresos = totalFinanciados + totalContados;

      return {
        ...vendedor,
        total_contratos: totalContratos,
        total_financiados: totalFinanciados,
        total_contados: totalContados,
        total_ingresos: totalIngresos,
      };
    });
  };

  const fetchData = async () => {
    setLoading(true);
    try {
      const { data } = await axios.get(
        `${apiUrl}/api/v1/admin/historial-ingresos-resumen`,
        {
          headers: {
            Authorization: `Bearer ${localStorage.getItem("access_token")}`,
          },
        }
      );
      setVendedores(data.vendedores || []);
      setContratos(data.contratos || []);
      setContratosFiltrados(data.contratos || []);
    } catch (error) {
      console.error("Error cargando datos", error);
    }
    setLoading(false);
  };

  // Resumen vendedores actualizado para la tabla
  const vendedoresResumen = calcularResumenVendedores(
    vendedores,
    contratosFiltrados
  );

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("es-ES", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
    });
  };

  const supervisoresUnicos = Array.from(
    new Set(vendedores.map((v) => v.supervisor_id).filter(Boolean))
  );

  // Lista de supervisores con datos (filtrando vendedores que sean supervisores)
  const supervisores = supervisoresUnicos
    .map((supId) => vendedores.find((v) => v.vendedor_id === supId))
    .filter(Boolean);

  // Filtrar vendedores que dependan del supervisor seleccionado
  const vendedoresFiltrados = supervisorSeleccionado
    ? vendedores.filter((v) => v.supervisor_id === supervisorSeleccionado)
    : vendedores;

  // Modificar el efecto de filtrado para incluir supervisor y vendedor
  useEffect(() => {
    let filtrados = [...contratos];

    if (filtro !== "Todos") {
      filtrados = filtrados.filter(
        (contrato) => contrato.detalles[0]?.modalidadPago === filtro
      );
    }

    if (fechaInicio) {
      filtrados = filtrados.filter(
        (contrato) => new Date(contrato.fecha_creacion) >= new Date(fechaInicio)
      );
    }

    if (fechaFin) {
      filtrados = filtrados.filter(
        (contrato) => new Date(contrato.fecha_creacion) <= new Date(fechaFin)
      );
    }

    if (supervisorSeleccionado) {
      // Filtrar contratos cuyo vendedor tiene ese supervisor
      const vendedoresSupervisor = vendedores
        .filter((v) => v.supervisor_id === supervisorSeleccionado)
        .map((v) => v.vendedor_id);

      filtrados = filtrados.filter((c) =>
        vendedoresSupervisor.includes(c.vendedor_id)
      );
    }

    if (vendedorSeleccionado) {
      filtrados = filtrados.filter(
        (c) => c.vendedor_id === vendedorSeleccionado
      );
    }

    setContratosFiltrados(filtrados);
  }, [
    filtro,
    fechaInicio,
    fechaFin,
    supervisorSeleccionado,
    vendedorSeleccionado,
    contratos,
    vendedores,
  ]);

  const totalContado = contratosFiltrados
    .filter((c) => c.detalles[0]?.modalidadPago === "De Contado")
    .reduce((sum, c) => sum + (c.monto_total || 0), 0);

  const totalFinanciado = contratosFiltrados
    .filter((c) => c.detalles[0]?.modalidadPago === "Financiado")
    .reduce((sum, c) => sum + (c.detalles[0]?.pagoInicial || 0), 0);

  const totalMonto = totalContado + totalFinanciado;

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          marginBottom: 2,
          gap: 2,
          flexWrap: "wrap",
        }}
      >
        <Typography
          variant="h6"
          sx={{
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
        >
          HISTORIAL DE INGRESOS
        </Typography>
        <Box
          sx={{
            display: "flex",
            justifyContent: "space-between",
            alignItems: "center",
            gap: 2,
            flexWrap: "wrap",
          }}
        >
          <FormControl variant="outlined" size="small" sx={{ minWidth: 180 }}>
            <InputLabel id="supervisor-label">Supervisor</InputLabel>
            <Select
              labelId="supervisor-label"
              value={supervisorSeleccionado}
              onChange={(e) => {
                setSupervisorSeleccionado(e.target.value);
                setVendedorSeleccionado(""); // reset vendedor al cambiar supervisor
              }}
              label="Supervisor"
            >
              <MenuItem value="">Todos</MenuItem>
              {supervisores.map((sup) => (
                <MenuItem key={sup.vendedor_id} value={sup.vendedor_id}>
                  {sup.nombre}
                </MenuItem>
              ))}
            </Select>
          </FormControl>

          <FormControl variant="outlined" size="small" sx={{ minWidth: 180 }}>
            <InputLabel id="vendedor-label">Vendedor</InputLabel>
            <Select
              labelId="vendedor-label"
              value={vendedorSeleccionado}
              onChange={(e) => setVendedorSeleccionado(e.target.value)}
              label="Vendedor"
              disabled={!supervisorSeleccionado}
            >
              <MenuItem value="">Todos</MenuItem>
              {vendedoresFiltrados.map((v) => (
                <MenuItem key={v.vendedor_id} value={v.vendedor_id}>
                  {v.nombre}
                </MenuItem>
              ))}
            </Select>
          </FormControl>

          <FormControl variant="outlined" size="small" sx={{ minWidth: 180 }}>
            <InputLabel id="filtro-label">Modalidad</InputLabel>
            <Select
              value={filtro}
              onChange={(e) => setFiltro(e.target.value)}
              label="Modalidad"
            >
              <MenuItem value="Todos">Todos</MenuItem>
              <MenuItem value="De Contado">De Contado</MenuItem>
              <MenuItem value="Financiado">Financiado</MenuItem>
            </Select>
          </FormControl>

          <TextField
            label="Fecha Inicio"
            type="date"
            size="small"
            InputLabelProps={{ shrink: true }}
            value={fechaInicio}
            onChange={(e) => setFechaInicio(e.target.value)}
          />

          <TextField
            label="Fecha Fin"
            type="date"
            size="small"
            InputLabelProps={{ shrink: true }}
            value={fechaFin}
            onChange={(e) => setFechaFin(e.target.value)}
          />
        </Box>
      </Box>

      {/* Tabla resumen vendedores */}
      <TableContainer component={Paper} sx={{ mb: 3 }}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>
                Supervisor / Vendedor
              </TableCell>
              <TableCell sx={{ color: "#fff" }}>Total Contratos</TableCell>
              <TableCell sx={{ color: "#fff" }}>Total Financiados</TableCell>
              <TableCell sx={{ color: "#fff" }}>Total Contados</TableCell>
              <TableCell sx={{ color: "#fff" }}>Total Ingresos</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {vendedoresResumen.map((vendedor) => (
              <TableRow key={vendedor.id}>
                <TableCell>{vendedor.nombre}</TableCell>
                <TableCell>{vendedor.total_contratos}</TableCell>
                <TableCell>${vendedor.total_financiados.toFixed(2)}</TableCell>
                <TableCell>${vendedor.total_contados.toFixed(2)}</TableCell>
                <TableCell>${vendedor.total_ingresos.toFixed(2)}</TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>

      {/* Tabla contratos */}
      <Typography
        variant="h6"
        sx={{
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
      >
        CONTRATOS
      </Typography>
      <TableContainer component={Paper}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>
                Supervisor / Vendedor
              </TableCell>
              <TableCell sx={{ color: "#fff" }}>N° Contrato</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
              <TableCell sx={{ color: "#fff" }}>Tipo Servicio</TableCell>
              <TableCell sx={{ color: "#fff" }}>Fecha</TableCell>
              <TableCell sx={{ color: "#fff" }}>Modalidad</TableCell>
              <TableCell sx={{ color: "#fff" }}>Monto Total</TableCell>
              <TableCell sx={{ color: "#fff" }}>Pago Inicial</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {contratosFiltrados.map((contrato) => (
              <TableRow key={contrato.id}>
                <TableCell>{contrato.vendedor_nombre}</TableCell>
                <TableCell>{contrato.numero_contrato}</TableCell>
                <TableCell>{contrato.cliente?.nombre}</TableCell>
                <TableCell>{contrato.tipo_servicio}</TableCell>
                <TableCell>{formatDate(contrato.fecha_creacion)}</TableCell>
                <TableCell>{contrato.detalles[0]?.modalidadPago}</TableCell>
                <TableCell>${contrato.monto_total}</TableCell>
                <TableCell>${contrato.detalles[0]?.pagoInicial}</TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>
      {/* Totales */}
      <Box
        sx={{
          mt: 3,
          display: "flex",
          justifyContent: "end",
          gap: 6,
          flexWrap: "wrap",
        }}
      >
        <Typography sx={{ color: "#000" }}>
          Total De Contado:{" "}
          <Typography component="span" sx={{ fontWeight: "bold" }}>
            ${totalContado.toFixed(2)}
          </Typography>
        </Typography>
        <Typography sx={{ color: "#000" }}>
          Total Financiado:{" "}
          <Typography component="span" sx={{ fontWeight: "bold" }}>
            ${totalFinanciado.toFixed(2)}
          </Typography>
        </Typography>
        <Typography sx={{ color: "#000" }}>
          Suma total de Pagos:{" "}
          <Typography component="span" sx={{ fontWeight: "bold" }}>
            ${totalMonto.toFixed(2)}
          </Typography>
        </Typography>
      </Box>
    </Box>
  );
}
