import React, { useState, useEffect } from "react";
import axios from "axios";
import {
  Table,
  Box,
  Grid,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  TextField,
  Typography,
  TablePagination,
  Button,
} from "@mui/material";
import { ArrowBack, Inbox as InboxIcon } from "@mui/icons-material";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";

const ParcelasTable = () => {
  const [parcelas, setParcelas] = useState([]);
  const [parcelaImg, setParcelaImg] = useState(null);
  const [file, setFile] = useState(null);
  const [editingCell, setEditingCell] = useState(null);
  const [editValue, setEditValue] = useState("");
  const [page, setPage] = useState(0); // Página actual (0-based para Material-UI)
  const [rowsPerPage] = useState(10); // Elementos por página, fijo en 10
  const [total, setTotal] = useState(0); // Total de registros
  const apiUrl = process.env.REACT_APP_API_URL; // Fallback a localhost si no está definido

  useEffect(() => {
    fetchParcelas(page + 1); // +1 porque el backend espera page 1-based
  }, [page]);

  const fetchParcelas = async (pageNum) => {
    try {
      const response = await axios.get(
        `${apiUrl}/parcelas?page=${pageNum}&per_page=${rowsPerPage}`
      );
      setParcelas(response.data.items || []); // Usar items y fallback a arreglo vacío
      setTotal(response.data.total || 0); // Establecer el total de registros
    } catch (error) {
      console.error("Error fetching parcelas:", error);
      setParcelas([]); // En caso de error, establecer un arreglo vacío
      setTotal(0);
    }
  };

  const handleEditClick = (parcelaId, field, value) => {
    setEditingCell({ parcelaId, field });
    setEditValue(value);
  };

  const handleInputChange = (e) => {
    setEditValue(e.target.value);
  };

  const handleSave = async (parcelaId) => {
    const field = editingCell.field;
    const updatedParcela = parcelas.find((p) => p.id === parcelaId);
    if (!updatedParcela) return;

    updatedParcela[field] =
      field.includes("precio") || field.includes("puestos")
        ? parseFloat(editValue) || 0
        : editValue;

    try {
      await axios.put(
        `${apiUrl}/parcelas/${updatedParcela.id}`,
        updatedParcela
      );
      setParcelas((prev) =>
        prev.map((p) => (p.id === parcelaId ? updatedParcela : p))
      );
      setEditingCell(null);
    } catch (error) {
      console.error("Error updating parcela:", error);
    }
  };

  const handleKeyPress = (e, parcelaId) => {
    if (e.key === "Enter") {
      handleSave(parcelaId);
    }
  };

  const handleChangePage = (event, newPage) => {
    setPage(newPage);
  };

  const handleFileChange = (event, setFileType) => {
    const file = event.target.files[0];
    if (file) {
      if (
        (setFileType === setParcelaImg && !file.type.startsWith("image/")) ||
        (setFileType === setFile && file.type !== "text/csv")
      ) {
        toast.info(
          `Por favor, selecciona un archivo ${
            setFileType === setParcelaImg ? "de imagen" : "CSV"
          } válido.`
        );
        return;
      }
      setFileType(file);
    } else {
      toast.info("Por favor, selecciona un archivo válido.");
    }
  };

  const handleUpload = async () => {
    if (!file || !parcelaImg) {
      toast.info("Debes subir tanto la imagen como el archivo CSV.");
      return;
    }

    const formDataImg = new FormData();
    formDataImg.append("parcela", parcelaImg);

    const formDataCsv = new FormData();
    formDataCsv.append("file", file);

    try {
      const token = localStorage.getItem("access_token");
      const config = {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "multipart/form-data",
        },
      };

      // Enviar imagen al endpoint /api/upload/inventario
      await axios.post(`${apiUrl}/api/upload/inventario`, formDataImg, config);

      // Enviar CSV al endpoint /api/admin/upload-parcelas
      await axios.post(
        `${apiUrl}/api/admin/upload-parcelas`,
        formDataCsv,
        config
      );

      toast.success("Parcelas actualizadas exitosamente.");
      setFile(null);
      setParcelaImg(null);
      fetchParcelas(page + 1); // Recargar datos después de la actualización
    } catch (error) {
      console.error("Error al subir los archivos:", error);
      alert("Error al actualizar las parcelas.");
    }
  };

  return (
    <Box
      sx={{
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        sx={{
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          marginBottom: 2,
        }}
        variant="h6"
        gutterBottom
      >
        INVENTARIO DE PARCELAS
      </Typography>

      <Grid container sx={{ marginBottom: 3, spacing: 1 }}>
        <Grid item xs={12} sm={2}>
          <Button variant="contained" component="label" color="primary">
            Anexar Imagen Parcelas
            <input
              type="file"
              hidden
              accept="image/*"
              onChange={(e) => handleFileChange(e, setParcelaImg)}
            />
          </Button>
        </Grid>
        <Grid item xs={12} sm={3} sx={{ ml: -2 }}>
          {" "}
          {/* Reducir margen izquierdo para acercar texto */}
          {parcelaImg && (
            <Typography sx={{ fontWeight: "bold" }}>
              Imagen: {parcelaImg.name}
            </Typography>
          )}
        </Grid>
        <Grid item xs={12} sm={3} sx={{ ml: -4 }}>
          <Button variant="contained" component="label" color="primary">
            Anexar Archivo CSV
            <input
              type="file"
              hidden
              accept=".csv"
              onChange={(e) => handleFileChange(e, setFile)}
            />
          </Button>
        </Grid>
        <Grid item xs={2} sx={{ ml: -4 }}>
          {" "}
          {/* Reducir margen izquierdo para acercar texto */}
          {file && (
            <Typography sx={{ fontWeight: "bold" }}>{file.name}</Typography>
          )}
        </Grid>
        <Grid item xs={2}>
          <Button
            variant="contained"
            onClick={handleUpload}
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Subir Archivos
          </Button>
        </Grid>
      </Grid>

      <TableContainer component={Paper} elevation={3}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}>
            <TableRow>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Código
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Disponibles
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Reservadas
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Ocupadas
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Vendidas
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {parcelas.map((parcela) => (
              <TableRow key={parcela.id} hover>
                <TableCell>{parcela.codigo}</TableCell>
                {["disponible", "reservadas", "ocupadas", "vendidas"].map(
                  (field) => (
                    <TableCell key={field}>
                      {editingCell?.parcelaId === parcela.id &&
                      editingCell?.field === field ? (
                        <TextField
                          type={
                            field.includes("precio") ||
                            field.includes("puestos")
                              ? "number"
                              : "text"
                          }
                          value={editValue}
                          onChange={handleInputChange}
                          onKeyPress={(e) => handleKeyPress(e, parcela.id)}
                          onBlur={() => handleSave(parcela.id)}
                          size="small"
                          autoFocus
                          fullWidth
                        />
                      ) : (
                        <span
                          onClick={() =>
                            handleEditClick(parcela.id, field, parcela[field])
                          }
                          style={{ cursor: "pointer" }}
                        >
                          {parcela[field]}
                        </span>
                      )}
                    </TableCell>
                  )
                )}
              </TableRow>
            ))}
          </TableBody>
        </Table>
        <TablePagination
          component="div"
          count={total} // Usar el total de registros
          page={page}
          onPageChange={handleChangePage}
          rowsPerPage={rowsPerPage}
          rowsPerPageOptions={[10]} // Solo 10 elementos por página
          labelRowsPerPage="Filas por página:"
        />
      </TableContainer>
      <Button
        variant="contained"
        component={Link}
        to="/administracion"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
    </Box>
  );
};

export default ParcelasTable;
