import React, { useState, useEffect } from "react";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Typography,
  Box,
} from "@mui/material";
import axios from "axios";

const PagosDeContado = () => {
  const [contratos, setContratos] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const apiUrl = process.env.REACT_APP_API_URL;

  useEffect(() => {
    const fetchContratos = async () => {
      try {
        const response = await axios.get(
          `${apiUrl}/api/v1/admin/contratos-de-contado`,
          {
            headers: {
              Authorization: `Bearer ${localStorage.getItem("token")}`,
            },
          }
        );
        // Filtramos solo contratos con modalidadPago "De Contado"
        const contratosContado = response.data.filter(
          (contrato) => contrato.detalles[0]?.modalidadPago === "De Contado"
        );
        setContratos(contratosContado);
        setLoading(false);
      } catch (err) {
        setError("Error al cargar los contratos");
        setLoading(false);
      }
    };

    fetchContratos();
  }, []);

  // Calcular la suma total de los montos
  const totalMonto = contratos.reduce(
    (sum, contrato) => sum + (contrato.monto_total || 0),
    0
  );

  if (loading) return <Typography>Cargando...</Typography>;
  if (error) return <Typography color="error">{error}</Typography>;

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("es-ES", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
    });
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        variant="h6"
        align="center"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        CONTRATOS DE CONTADO
      </Typography>
      <TableContainer component={Paper} sx={{ boxShadow: 3 }}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>No. de Contrato</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
              <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
              <TableCell sx={{ color: "#fff" }}>
                Fecha Inicio del Contrato
              </TableCell>
              <TableCell sx={{ color: "#fff" }}>Monto Total</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {contratos.map((contrato) => (
              <TableRow key={contrato.id}>
                <TableCell>{contrato.numero_contrato}</TableCell>
                <TableCell>{contrato.cliente.nombre}</TableCell>
                <TableCell>{contrato.tipo_servicio}</TableCell>
                <TableCell>{formatDate(contrato.fecha_creacion)}</TableCell>
                <TableCell>${contrato.monto_total.toFixed(2)}</TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>
      <Box sx={{ mt: 3 }}>
        <TableContainer component={Paper} sx={{ boxShadow: 3 }}>
          <Table size="small">
            <TableHead>
              <TableRow>
                <TableCell sx={{ textAlign: "right" }}>
                  Suma total de Pagos
                  <Typography
                    component="span"
                    sx={{ pl: 2, fontWeight: "bold" }}
                  >
                    ${totalMonto.toFixed(2)}
                  </Typography>
                </TableCell>
              </TableRow>
            </TableHead>
          </Table>
        </TableContainer>
      </Box>
    </Box>
  );
};

export default PagosDeContado;
