import React, { useState, useEffect } from "react";
import {
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  TextField,
  Paper,
  CircularProgress,
  Alert,
  Box,
  Button,
} from "@mui/material";
import { ArrowBack } from "@mui/icons-material";
import axios from "axios";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";

const PlanesFunerariosTable = () => {
  const [planes, setPlanes] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [editingCell, setEditingCell] = useState(null);
  const [editValue, setEditValue] = useState("");

  const apiUrl = process.env.REACT_APP_API_URL || "http://localhost:8000";

  // Cargar planes al montar el componente
  useEffect(() => {
    fetchPlanes();
  }, []);

  const fetchPlanes = async () => {
    try {
      const token = localStorage.getItem("access_token");
      const config = token
        ? { headers: { Authorization: `Bearer ${token}` } }
        : {};
      const response = await axios.get(
        `${apiUrl}/api/v1/admin/planes/funerarios`,
        config
      );
      console.log("Respuesta del backend:", response.data);
      setPlanes(Array.isArray(response.data) ? response.data : []);
      setLoading(false);
    } catch (err) {
      console.error("Error al cargar planes:", err);
      setError(`Error al cargar los planes: ${err.message}`);
      setLoading(false);
      toast.error(`Error al cargar los planes: ${err.message}`);
    }
  };

  const handleEditClick = (planId, field, value) => {
    setEditingCell({ planId, field });
    setEditValue(value || "");
  };

  const handleInputChange = (e) => {
    setEditValue(e.target.value);
  };

  const handleSave = async (planId) => {
    if (!editingCell) return;

    const field = editingCell.field;
    const updatedPlan = planes.find((p) => p.id === planId);
    if (!updatedPlan) return;

    const parsedValue = [
      "puestos",
      "de_contado",
      "financiado",
      "contado_dobles",
      "financiado_dobles",
    ].includes(field)
      ? parseFloat(editValue) || null
      : editValue;

    updatedPlan[field] = parsedValue;

    try {
      const token = localStorage.getItem("access_token");
      const config = token
        ? { headers: { Authorization: `Bearer ${token}` } }
        : {};
      await axios.put(
        `${apiUrl}/api/v1/admin/planes/funerarios/${planId}`,
        { [field]: parsedValue },
        config
      );
      setPlanes((prev) => prev.map((p) => (p.id === planId ? updatedPlan : p)));
      setEditingCell(null);
      toast.success("Plan actualizado correctamente");
    } catch (error) {
      console.error("Error updating plan:", error);
      toast.error("Error al actualizar el plan");
    }
  };

  const handleKeyPress = (e, planId) => {
    if (e.key === "Enter") {
      handleSave(planId);
    }
  };

  if (loading) return <CircularProgress />;
  if (error) return <Alert severity="error">{error}</Alert>;

  console.log("Planes en el estado:", planes);

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
        maxWidth: "1200px",
      }}
    >
      <Typography
        variant="h6"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        PLANES FUNERARIOS
      </Typography>
      {planes.length === 0 && (
        <Alert severity="info">No hay planes disponibles</Alert>
      )}
      <TableContainer
        component={Paper}
        sx={{ overflowX: "auto" }} // Habilitar scroll horizontal
      >
        <Table sx={{ minWidth: "1200px" }}>
          {" "}
          {/* Establecer ancho mínimo para la tabla */}
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              {[
                { label: "Tipo de Capilla", field: "tipo_capilla" },
                { label: "Tiempo de Velación", field: "tiempo_velacion" },
                { label: "Precio Velación", field: "precio_velacion" },
                {
                  label: "Preparación del Difunto Sencilla",
                  field: "preparacion_difunto_sencilla",
                },
                {
                  label: "Preparación del Difunto Especial",
                  field: "preparacion_difunto_especial",
                },
                {
                  label: "Preparación del Difunto Embalsamiento",
                  field: "preparacion_difunto_embalsamiento",
                },
                {
                  label: "Alquiler de Ataúd Sencillo",
                  field: "alquiler_ataud_sencillo",
                },
                {
                  label: "Alquiler de Ataúd Especial",
                  field: "alquiler_ataud_mejorado",
                },
                {
                  label: "Alquiler de Ataúd de Lujo",
                  field: "alquiler_ataud_lujo",
                },
                { label: "Precio Traslado", field: "precio_traslado" },
              ].map(({ label, field }) => (
                <TableCell
                  key={field}
                  sx={{
                    color: "#fff",
                    fontSize: "16px",
                    whiteSpace: "nowrap", // Evitar salto de línea
                    minWidth: 150, // Ancho mínimo para cada columna
                    padding: "8px 16px", // Ajustar padding para mejor legibilidad
                  }}
                >
                  {label}
                </TableCell>
              ))}
            </TableRow>
          </TableHead>
          <TableBody
            sx={{
              color: "#fff",
              fontSize: "16px",
              whiteSpace: "nowrap", // Evitar salto de línea
              minWidth: 150, // Ancho mínimo para cada columna
              padding: "8px 16px", // Ajustar padding para mejor legibilidad
            }}
          >
            {planes.map((plan) => (
              <TableRow key={plan.id} hover>
                {[
                  "tipo_capilla",
                  "tiempo_velacion",
                  "precio_velacion",
                  "preparacion_difunto_sencilla",
                  "preparacion_difunto_especial",
                  "preparacion_difunto_embalsamiento",
                  "alquiler_ataud_sencillo",
                  "alquiler_ataud_mejorado",
                  "alquiler_ataud_lujo",
                  "precio_traslado",
                ].map((field) => (
                  <TableCell key={field} sx={{ padding: "8px 16px" }}>
                    {editingCell?.planId === plan.id &&
                    editingCell?.field === field ? (
                      <TextField
                        type={
                          [
                            "precio_velacion",
                            "preparacion_difunto_sencilla",
                            "preparacion_difunto_especial",
                            "preparacion_difunto_embalsamiento",
                            "alquiler_ataud_sencillo",
                            "alquiler_ataud_mejorado",
                            "alquiler_ataud_lujo",
                            "precio_traslado",
                          ].includes(field)
                            ? "number"
                            : "text"
                        }
                        value={editValue}
                        onChange={handleInputChange}
                        onKeyPress={(e) => handleKeyPress(e, plan.id)}
                        onBlur={() => handleSave(plan.id)}
                        size="small"
                        autoFocus
                        fullWidth
                      />
                    ) : (
                      <span
                        onClick={() =>
                          handleEditClick(plan.id, field, plan[field])
                        }
                        style={{ cursor: "pointer" }}
                      >
                        {plan[field] || ""}
                      </span>
                    )}
                  </TableCell>
                ))}
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>
      <Button
        variant="contained"
        component={Link}
        to="/administracion"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
    </Box>
  );
};

export default PlanesFunerariosTable;
