import React, { useState, useEffect } from "react";
import {
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  TextField,
  Paper,
  CircularProgress,
  Alert,
  Box,
  Button,
  Switch,
} from "@mui/material";
import { ArrowBack } from "@mui/icons-material";
import axios from "axios";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";

const PlanesTable = () => {
  const [planes, setPlanes] = useState([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [editingCell, setEditingCell] = useState(null);
  const [editValue, setEditValue] = useState("");

  const apiUrl = process.env.REACT_APP_API_URL || "http://localhost:8000";

  // Cargar planes al montar el componente
  useEffect(() => {
    fetchPlanes();
  }, []);

  const fetchPlanes = async () => {
    try {
      const token = localStorage.getItem("access_token");
      const config = token
        ? { headers: { Authorization: `Bearer ${token}` } }
        : {};
      const response = await axios.get(`${apiUrl}/api/v1/admin/planes`, config);
      //console.log("Respuesta del backend:", response.data);
      setPlanes(Array.isArray(response.data) ? response.data : []);
      setLoading(false);
    } catch (err) {
      console.error("Error al cargar planes:", err);
      setError(`Error al cargar los planes: ${err.message}`);
      setLoading(false);
      toast.error(`Error al cargar los planes: ${err.message}`);
    }
  };

  const handleEditClick = (planId, field, value) => {
    setEditingCell({ planId, field });
    setEditValue(value || "");
  };

  const handleInputChange = (e) => {
    setEditValue(e.target.value);
  };

  const handleSwitchChange = (planId, value) => {
    // Actualiza el estado local para reflejar el cambio inmediatamente
    setPlanes((prevPlanes) =>
      prevPlanes.map((plan) =>
        plan.id === planId ? { ...plan, is_active: value } : plan
      )
    );

    // Actualiza el backend
    const token = localStorage.getItem("access_token");
    axios
      .patch(
        `${apiUrl}/api/v1/admin/planes/${planId}/estado`,
        { activo: value },
        { headers: { Authorization: `Bearer ${token}` } }
      )
      .then(() => {
        toast.success("Estado actualizado correctamente.");
      })
      .catch((err) => {
        toast.error("Error al actualizar el estado.");
        console.error(err);
        // Revertir cambio local en caso de error
        setPlanes((prevPlanes) =>
          prevPlanes.map((plan) =>
            plan.id === planId ? { ...plan, is_active: !value } : plan
          )
        );
      });
  };

  const handleSave = async (planId) => {
    if (!editingCell) return;

    const field = editingCell.field;
    const updatedPlan = planes.find((p) => p.id === planId);
    if (!updatedPlan) return;

    const parsedValue = [
      "puestos",
      "de_contado_bcv",
      "de_contado",
      "financiado",
      "cuota_1",
      "cuota_2",
      "cuota_3",
      "cuota_4",
      "cuota_5",
      "cuota_6",
    ].includes(field)
      ? parseFloat(editValue) || null
      : editValue;

    updatedPlan[field] = parsedValue;

    try {
      const token = localStorage.getItem("access_token");
      const config = token
        ? { headers: { Authorization: `Bearer ${token}` } }
        : {};
      await axios.put(
        `${apiUrl}/api/v1/admin/planes/${planId}`,
        { [field]: parsedValue },
        config
      );
      setPlanes((prev) => prev.map((p) => (p.id === planId ? updatedPlan : p)));
      setEditingCell(null);
      toast.success("Plan actualizado correctamente");
    } catch (error) {
      console.error("Error updating plan:", error);
      toast.error("Error al actualizar el plan");
    }
  };

  const handleKeyPress = (e, planId) => {
    if (e.key === "Enter") {
      handleSave(planId);
    }
  };

  if (loading) return <CircularProgress />;
  if (error) return <Alert severity="error">{error}</Alert>;

  //console.log("Planes en el estado:", planes);

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
        maxWidth: "1200px",
      }}
    >
      <Typography
        variant="h6"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        PLANES
      </Typography>
      {planes.length === 0 && (
        <Alert severity="info">No hay planes disponibles</Alert>
      )}
      <TableContainer
        component={Paper}
        sx={{
          overflowX: "auto", // scroll horizontal
          maxWidth: "100%",
        }}
      >
        <Table size="small" stickyHeader sx={{ tableLayout: "auto" }}>
          <TableHead
            sx={{
              "& .MuiTableCell-head": {
                backgroundColor: "#0A4575",
                color: "white",
                whiteSpace: "nowrap",
                overflow: "hidden",
                textOverflow: "ellipsis",
              },
            }}
          >
            <TableRow>
              {[
                "Tipo de Servicio",
                "Descripción",
                "Puestos",
                "De Contado a BCV",
                "De Contado",
                "Financiado",
                "1era. Cuota",
                "2da. Cuota",
                "3era. Cuota",
                "4ta. Cuota",
                "5ta. Cuota",
                "6ta. Cuota",
                "Activo",
              ].map((title) => (
                <TableCell
                  key={title}
                  sx={{
                    color: "#fff",
                    fontSize: "16px",
                    backgroundColor: "#0A4575",
                    whiteSpace: "nowrap",
                  }}
                >
                  {title}
                </TableCell>
              ))}
            </TableRow>
          </TableHead>

          <TableBody sx={{ whiteSpace: "nowrap" }}>
            {planes.map((plan) => (
              <TableRow key={plan.id} hover sx={{ whiteSpace: "nowrap" }}>
                {[
                  "tipo_servicio",
                  "descripcion",
                  "puestos",
                  "de_contado_bcv",
                  "de_contado",
                  "financiado",
                  "cuota_1",
                  "cuota_2",
                  "cuota_3",
                  "cuota_4",
                  "cuota_5",
                  "cuota_6",
                ].map((field) => (
                  <TableCell key={field} sx={{ whiteSpace: "nowrap" }}>
                    {editingCell?.planId === plan.id &&
                    editingCell?.field === field ? (
                      <TextField
                        type={
                          [
                            "puestos",
                            "de_contado_bcv",
                            "de_contado",
                            "financiado",
                            "cuota_1",
                            "cuota_2",
                            "cuota_3",
                            "cuota_4",
                            "cuota_5",
                            "cuota_6",
                          ].includes(field)
                            ? "number"
                            : "text"
                        }
                        value={editValue}
                        onChange={handleInputChange}
                        onKeyPress={(e) => handleKeyPress(e, plan.id)}
                        onBlur={() => handleSave(plan.id)}
                        size="small"
                        autoFocus
                        fullWidth
                      />
                    ) : (
                      <span
                        onClick={() =>
                          handleEditClick(plan.id, field, plan[field])
                        }
                        style={{ cursor: "pointer" }}
                      >
                        {plan[field] || ""}
                      </span>
                    )}
                  </TableCell>
                ))}

                {/* Switch de is_active */}
                <TableCell>
                  <Switch
                    checked={plan.is_active}
                    onChange={(e) =>
                      handleSwitchChange(plan.id, e.target.checked)
                    }
                  />
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>

      <Button
        variant="contained"
        component={Link}
        to="/administracion"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
    </Box>
  );
};

export default PlanesTable;
