import React, { useEffect, useState } from "react";
import {
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Typography,
  Box,
  Button,
  IconButton,
  Collapse,
  TablePagination,
  CircularProgress,
} from "@mui/material";
import { ArrowBack, ExpandMore, ExpandLess } from "@mui/icons-material";
import axios from "axios";
import { Link } from "react-router-dom";
import { toast } from "react-toastify";
import { jwtDecode } from "jwt-decode";

const apiUrl = process.env.REACT_APP_API_URL || "http://localhost:8000";

const SolicitudesCaducadasTable = () => {
  const [solicitudes, setSolicitudes] = useState([]);
  const [loading, setLoading] = useState(true);
  const [openRows, setOpenRows] = useState({});
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [totalCount, setTotalCount] = useState(0);
  const [message, setMessage] = useState("");

  const fetchSolicitudesCaducadas = async () => {
    try {
      const token = localStorage.getItem("access_token");
      if (!token) {
        toast.error("No se encontró el token de autenticación.");
        setLoading(false);
        return;
      }

      const decoded = jwtDecode(token);
      const userId = decoded.sub ? Number(decoded.sub) : null;

      if (!userId) {
        toast.error("No se pudo obtener el ID del usuario.");
        setLoading(false);
        return;
      }

      const config = {
        headers: { Authorization: `Bearer ${token}` },
        params: { skip: page * rowsPerPage, limit: rowsPerPage },
      };

      const response = await axios.get(
        `${apiUrl}/api/v1/admin/solicitudes/caducadas`,
        config
      );

      if (!response.data.data) {
        throw new Error("La respuesta de la API no contiene datos válidos.");
      }

      setSolicitudes(response.data.data);
      setTotalCount(
        response.data.total_solicitudes || response.data.data.length
      );
      setMessage(response.data.message || "Datos cargados correctamente");
    } catch (error) {
      console.error("Error al obtener solicitudes caducadas:", error);
      toast.error(
        error.response?.data?.detail || "Error al cargar solicitudes caducadas."
      );
      setMessage("Error al cargar solicitudes caducadas.");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchSolicitudesCaducadas();
  }, [page, rowsPerPage]);

  const handleDeleteSolicitud = async (solicitudId) => {
    if (
      !window.confirm(
        "¿Estás seguro de que deseas eliminar esta solicitud caducada?"
      )
    ) {
      return;
    }

    try {
      setLoading(true);
      const token = localStorage.getItem("access_token");
      if (!token) {
        toast.error("No se encontró el token de autenticación.");
        return;
      }

      const config = {
        headers: { Authorization: `Bearer ${token}` },
      };

      await axios.delete(
        `${apiUrl}/api/v1/admin/solicitudes/caducadas/${solicitudId}`,
        config
      );

      toast.success("Solicitud caducada eliminada con éxito.");
      // Refrescar la lista de solicitudes
      await fetchSolicitudesCaducadas();
    } catch (error) {
      console.error("Error al eliminar solicitud caducada:", error);
      toast.error(
        error.response?.data?.detail ||
          "Error al eliminar la solicitud caducada."
      );
    } finally {
      setLoading(false);
    }
  };

  const handleToggleRow = (id) => {
    setOpenRows((prev) => ({ ...prev, [id]: !prev[id] }));
  };

  const handleChangePage = (event, newPage) => {
    setPage(newPage);
  };

  const handleChangeRowsPerPage = (event) => {
    setRowsPerPage(parseInt(event.target.value, 10));
    setPage(0);
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 2,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        variant="h6"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        SOLICITUDES CADUCADAS
      </Typography>
      <Typography
        variant="body1"
        align="center"
        gutterBottom
        sx={{ color: "#0A4575" }}
      ></Typography>
      {loading ? (
        <Box sx={{ display: "flex", justifyContent: "center", mt: 4 }}>
          <CircularProgress />
        </Box>
      ) : solicitudes.length === 0 ? (
        <Typography variant="body1" color="text.secondary">
          No hay solicitudes caducadas disponibles.
        </Typography>
      ) : (
        <>
          <TableContainer component={Paper}>
            <Table size="small">
              <TableHead sx={{ backgroundColor: "#0A4575" }}>
                <TableRow>
                  <TableCell sx={{ color: "#fff" }}>
                    Número de Solicitud
                  </TableCell>
                  <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Teléfono</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
                  <TableCell sx={{ color: "#fff" }}>
                    Fecha de Creación
                  </TableCell>
                  <TableCell sx={{ color: "#fff" }}>
                    Fecha de Vencimiento
                  </TableCell>
                  <TableCell sx={{ color: "#fff" }}>Estado</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Acción</TableCell>
                  <TableCell sx={{ color: "#fff" }}></TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {solicitudes.map((solicitud) => (
                  <React.Fragment key={solicitud.id}>
                    <TableRow>
                      <TableCell>
                        {solicitud.numero_solicitud || "N/A"}
                      </TableCell>
                      <TableCell>
                        {solicitud.cliente?.nombre || "N/A"}
                      </TableCell>
                      <TableCell>
                        {solicitud.cliente?.celular || "N/A"}
                      </TableCell>
                      <TableCell>{solicitud.tipo_servicio || "N/A"}</TableCell>
                      <TableCell>
                        {solicitud.fecha_creacion
                          ? new Date(
                              solicitud.fecha_creacion
                            ).toLocaleDateString()
                          : "N/A"}
                      </TableCell>
                      <TableCell>
                        {solicitud.fecha_vencimiento
                          ? new Date(
                              solicitud.fecha_vencimiento
                            ).toLocaleDateString()
                          : "N/A"}
                      </TableCell>
                      <TableCell>
                        <Button
                          variant="contained"
                          sx={{ backgroundColor: "#f4d03f", color: "#000000" }}
                        >
                          {solicitud.estado || "N/A"}
                        </Button>
                      </TableCell>
                      <TableCell>
                        <Button
                          variant="contained"
                          sx={{ backgroundColor: "#d32f2f", color: "#FFFFFF" }}
                          onClick={() => handleDeleteSolicitud(solicitud.id)}
                        >
                          ELIMINAR
                        </Button>
                      </TableCell>
                      <TableCell>
                        <IconButton
                          onClick={() => handleToggleRow(solicitud.id)}
                        >
                          {openRows[solicitud.id] ? (
                            <ExpandLess />
                          ) : (
                            <ExpandMore />
                          )}
                        </IconButton>
                      </TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell
                        colSpan={9}
                        style={{ paddingBottom: 0, paddingTop: 0 }}
                      >
                        <Collapse
                          in={openRows[solicitud.id]}
                          timeout="auto"
                          unmountOnExit
                        >
                          <Box sx={{ margin: 1 }}>
                            <Typography
                              variant="h6"
                              gutterBottom
                              sx={{
                                fontFamily: "Bakbak One, sans-serif",
                                color: "#0A4575",
                              }}
                            >
                              Detalles de la Solicitud
                            </Typography>
                            {solicitud.detalles &&
                            solicitud.detalles.length > 0 ? (
                              <Table size="small">
                                <TableHead sx={{ backgroundColor: "#0A4575" }}>
                                  <TableRow>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Descripción
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Modalidad de Pago
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Pago Inicial
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Cuotas
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Monto por Cuota
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Precio Total
                                    </TableCell>
                                  </TableRow>
                                </TableHead>
                                <TableBody>
                                  {solicitud.detalles.map((detalle) => (
                                    <TableRow key={detalle.id}>
                                      <TableCell>
                                        {detalle.descripcion || "N/A"}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.modalidadPago || "N/A"}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.pagoInicial !== null
                                          ? `$${detalle.pagoInicial.toFixed(2)}`
                                          : "N/A"}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.cuotas || 0}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.montoPorCuota !== null
                                          ? `$${detalle.montoPorCuota.toFixed(
                                              2
                                            )}`
                                          : "N/A"}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.precioTotal !== null
                                          ? `$${detalle.precioTotal.toFixed(2)}`
                                          : "N/A"}
                                      </TableCell>
                                    </TableRow>
                                  ))}
                                </TableBody>
                              </Table>
                            ) : (
                              <Typography
                                variant="body2"
                                color="text.secondary"
                                textAlign={"center"}
                              >
                                No hay detalles disponibles para esta solicitud.
                              </Typography>
                            )}
                          </Box>
                        </Collapse>
                      </TableCell>
                    </TableRow>
                  </React.Fragment>
                ))}
              </TableBody>
            </Table>
          </TableContainer>
          <TablePagination
            rowsPerPageOptions={[5, 10, 25]}
            component="div"
            count={totalCount}
            rowsPerPage={rowsPerPage}
            page={page}
            onPageChange={handleChangePage}
            onRowsPerPageChange={handleChangeRowsPerPage}
            labelRowsPerPage="Filas por página:"
            labelDisplayedRows={({ from, to, count }) =>
              `${from}–${to} de ${count}`
            }
          />
          <Button
            variant="contained"
            component={Link}
            to="/administracion"
            startIcon={<ArrowBack />}
            sx={{
              marginTop: "15px",
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Volver
          </Button>
        </>
      )}
    </Box>
  );
};

export default SolicitudesCaducadasTable;
