import React, { useEffect, useState, useCallback } from "react";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Collapse,
  IconButton,
  Paper,
  TablePagination,
  Modal,
  Grid,
  TextField,
  Backdrop,
} from "@mui/material";
import {
  ArrowBack,
  ExpandMore as ExpandMoreIcon,
  ExpandLess as ExpandLessIcon,
  Delete as DeleteIcon,
  Add as AddIcon,
} from "@mui/icons-material";
import axios from "axios";
import { jwtDecode } from "jwt-decode";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";
import { v4 as uuidv4 } from "uuid";

const apiUrl = process.env.REACT_APP_API_URL;

// Componente AutorizadosModal definido fuera del componente principal
const AutorizadosModal = React.memo(({ open, onClose, contrato, onSubmit }) => {
  const [autorizados, setAutorizados] = useState([
    {
      id: uuidv4(), // ID único para cada autorizado
      nombre: "",
      cedula: "",
      direccion: "",
      fecha_nacimiento: "",
      edad: "",
      telefono: "",
      celular: "",
      email: "",
      nombre_documento: "",
      documento_identidad: null,
    },
  ]);
  const [emailValids, setEmailValids] = useState([true]);
  const [dateValids, setDateValids] = useState([true]);

  const handleAutorizadoChange = useCallback(
    (id, e) => {
      const { name, value } = e.target;
      setAutorizados((prev) =>
        prev.map((autorizado) =>
          autorizado.id === id ? { ...autorizado, [name]: value } : autorizado
        )
      );

      if (name === "fecha_nacimiento" && value) {
        const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
        if (dateRegex.test(value)) {
          const birthDate = new Date(value);
          if (!isNaN(birthDate.getTime())) {
            const today = new Date();
            let age = today.getFullYear() - birthDate.getFullYear();
            const monthDiff = today.getMonth() - birthDate.getMonth();
            if (
              monthDiff < 0 ||
              (monthDiff === 0 && today.getDate() < birthDate.getDate())
            ) {
              age--;
            }
            setAutorizados((prev) =>
              prev.map((autorizado) =>
                autorizado.id === id ? { ...autorizado, edad: age } : autorizado
              )
            );
            setDateValids((prev) =>
              prev.map((valid, i) => (autorizados[i].id === id ? true : valid))
            );
          } else {
            setDateValids((prev) =>
              prev.map((valid, i) => (autorizados[i].id === id ? false : valid))
            );
          }
        } else {
          setDateValids((prev) =>
            prev.map((valid, i) => (autorizados[i].id === id ? false : valid))
          );
        }
      }
    },
    [autorizados]
  );

  const handleBlur = useCallback(
    (id, e) => {
      const { name, value } = e.target;
      if (name === "email") {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        setEmailValids((prev) =>
          prev.map((valid, i) =>
            autorizados[i].id === id ? emailRegex.test(value) : valid
          )
        );
      } else if (name === "fecha_nacimiento") {
        const dateRegex = /^\d{4}-\d{2}-\d{2}$/;
        setDateValids((prev) =>
          prev.map((valid, i) =>
            autorizados[i].id === id
              ? dateRegex.test(value) && !isNaN(new Date(value).getTime())
              : valid
          )
        );
      }
    },
    [autorizados]
  );

  const handleFileChange = useCallback((id, e) => {
    const file = e.target.files[0];
    if (file) {
      setAutorizados((prev) =>
        prev.map((autorizado) =>
          autorizado.id === id
            ? {
                ...autorizado,
                documento_identidad: file,
                nombre_documento: file.name,
              }
            : autorizado
        )
      );
    }
  }, []);

  const handleAddAutorizado = () => {
    if (autorizados.length < 2) {
      setAutorizados((prev) => [
        ...prev,
        {
          id: uuidv4(),
          nombre: "",
          cedula: "",
          direccion: "",
          fecha_nacimiento: "",
          edad: "",
          telefono: "",
          celular: "",
          email: "",
          nombre_documento: "",
          documento_identidad: null,
        },
      ]);
      setEmailValids((prev) => [...prev, true]);
      setDateValids((prev) => [...prev, true]);
    }
  };

  const handleRemoveAutorizado = (id) => {
    setAutorizados((prev) => prev.filter((autorizado) => autorizado.id !== id));
    setEmailValids((prev) => prev.filter((_, i) => autorizados[i].id !== id));
    setDateValids((prev) => prev.filter((_, i) => autorizados[i].id !== id));
  };

  const isFormValid = () => {
    return autorizados.every(
      (autorizado, index) =>
        (!autorizado.nombre && !autorizado.cedula) ||
        (autorizado.nombre &&
          autorizado.cedula &&
          autorizado.fecha_nacimiento &&
          autorizado.edad &&
          autorizado.telefono &&
          autorizado.direccion &&
          autorizado.email &&
          emailValids[index] &&
          dateValids[index])
    );
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    await onSubmit(autorizados);
  };

  return (
    <Modal
      open={open}
      onClose={onClose}
      closeAfterTransition
      BackdropComponent={Backdrop}
      BackdropProps={{ timeout: 500 }}
    >
      <Box
        sx={{
          position: "absolute",
          top: "50%",
          left: "50%",
          transform: "translate(-50%, -50%)",
          width: "70%",
          bgcolor: "background.paper",
          boxShadow: 24,
          p: 4,
          borderRadius: 2,
          maxHeight: "90vh",
          overflowY: "auto",
        }}
      >
        <Typography
          sx={{ fontFamily: "Bakbak One", color: "#0A4575" }}
          variant="h5"
        >
          AGREGAR PERSONAS AUTORIZADAS -{" "}
          <span style={{ color: "#000000" }}>#{contrato?.numero_contrato}</span>
        </Typography>
        <form onSubmit={handleSubmit}>
          {autorizados.map((autorizado, index) => (
            <Box
              key={autorizado.id}
              sx={{ mt: 3, borderBottom: "1px solid #ccc" }}
            >
              <Box
                sx={{
                  display: "flex",
                  justifyContent: "space-between",
                  alignItems: "center",
                  mb: 2,
                }}
              >
                <Typography sx={{ fontFamily: "Bakbak One", color: "#0A4575" }}>
                  Persona Autorizada {index + 1}
                </Typography>
                {index === 1 && (
                  <IconButton
                    onClick={() => handleRemoveAutorizado(autorizado.id)}
                    sx={{ color: "#d32f2f" }}
                  >
                    <DeleteIcon />
                  </IconButton>
                )}
              </Box>
              <Grid container spacing={3}>
                <Grid item xs={12} sm={4}>
                  <TextField
                    fullWidth
                    label="Nombre cliente"
                    variant="outlined"
                    size="small"
                    name="nombre"
                    value={autorizado.nombre}
                    onChange={(e) => handleAutorizadoChange(autorizado.id, e)}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <TextField
                    fullWidth
                    label="Cédula o Rif"
                    variant="outlined"
                    size="small"
                    name="cedula"
                    value={autorizado.cedula}
                    onChange={(e) => handleAutorizadoChange(autorizado.id, e)}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <TextField
                    fullWidth
                    label="Fecha de Nacimiento (YYYY-MM-DD)"
                    variant="outlined"
                    type="date"
                    size="small"
                    name="fecha_nacimiento"
                    value={autorizado.fecha_nacimiento || ""}
                    onChange={(e) => handleAutorizadoChange(autorizado.id, e)}
                    onBlur={(e) => handleBlur(autorizado.id, e)}
                    required
                    sx={{ boxShadow: 3 }}
                    error={!dateValids[index]}
                    helperText={
                      !dateValids[index]
                        ? "Ingresa una fecha válida (YYYY-MM-DD)"
                        : ""
                    }
                    placeholder="YYYY-MM-DD"
                  />
                </Grid>
                <Grid item xs={12} sm={2}>
                  <TextField
                    fullWidth
                    label="Edad"
                    variant="outlined"
                    size="small"
                    name="edad"
                    value={autorizado.edad}
                    required
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <TextField
                    fullWidth
                    label="Teléfono"
                    variant="outlined"
                    size="small"
                    name="telefono"
                    inputProps={{ maxLength: 10 }}
                    value={autorizado.telefono}
                    onChange={(e) => handleAutorizadoChange(autorizado.id, e)}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <TextField
                    fullWidth
                    label="Celular"
                    variant="outlined"
                    size="small"
                    name="celular"
                    inputProps={{ maxLength: 11 }}
                    value={autorizado.celular}
                    onChange={(e) => handleAutorizadoChange(autorizado.id, e)}
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={6}>
                  <TextField
                    fullWidth
                    label="Email"
                    variant="outlined"
                    size="small"
                    name="email"
                    type="email"
                    value={autorizado.email}
                    onChange={(e) => handleAutorizadoChange(autorizado.id, e)}
                    onBlur={(e) => handleBlur(autorizado.id, e)}
                    required
                    sx={{ boxShadow: 3 }}
                    error={!emailValids[index]}
                    helperText={
                      !emailValids[index]
                        ? "Ingresa un email válido (ejemplo@gmail.com)"
                        : ""
                    }
                  />
                </Grid>
                <Grid item xs={12} sm={5}>
                  <TextField
                    fullWidth
                    label="Dirección"
                    variant="outlined"
                    size="small"
                    name="direccion"
                    value={autorizado.direccion}
                    onChange={(e) => handleAutorizadoChange(autorizado.id, e)}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={4}>
                  <input
                    accept="image/*"
                    style={{ display: "none" }}
                    id={`file-upload-${autorizado.id}`}
                    type="file"
                    onChange={(e) => handleFileChange(autorizado.id, e)}
                  />
                  <label htmlFor={`file-upload-${autorizado.id}`}>
                    <Button
                      variant="contained"
                      component="span"
                      sx={{
                        backgroundColor: "#0A4575",
                        color: "#FFFFFF",
                        "&:hover": { backgroundColor: "#083a5e" },
                      }}
                    >
                      Subir Documento de Identidad
                    </Button>
                  </label>
                </Grid>
                <Grid item xs={12} sm={2}>
                  {autorizado.nombre_documento && (
                    <Typography sx={{ marginTop: "10px", fontFamily: "Abel" }}>
                      <b>{autorizado.nombre_documento}</b>
                    </Typography>
                  )}
                </Grid>
              </Grid>
            </Box>
          ))}
          {autorizados.length < 2 && (
            <Button
              variant="contained"
              startIcon={<AddIcon />}
              onClick={handleAddAutorizado}
              sx={{
                mt: 2,
                backgroundColor: "#0A4575",
                color: "#FFFFFF",
                "&:hover": { backgroundColor: "#083a5e" },
              }}
            >
              Agregar Otro Autorizado
            </Button>
          )}
          <Box
            sx={{ display: "flex", justifyContent: "flex-end", marginTop: 3 }}
          >
            <Button variant="outlined" onClick={onClose} sx={{ mr: 2 }}>
              Cancelar
            </Button>
            <Button
              type="submit"
              variant="contained"
              disabled={!isFormValid()}
              sx={{
                backgroundColor: "#0A4575",
                color: "#FFFFFF",
                "&:hover": { backgroundColor: "#083a5e" },
              }}
            >
              Guardar Autorizados
            </Button>
          </Box>
        </form>
      </Box>
    </Modal>
  );
});

const AutorizadosPage = () => {
  const [contratos, setContratos] = useState({ items: [], total: 0 });
  const [expandedContrato, setExpandedContrato] = useState(null);
  const [openAutorizadosModal, setOpenAutorizadosModal] = useState(false);
  const [selectedContrato, setSelectedContrato] = useState(null);
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [autorizadosPorContrato, setAutorizadosPorContrato] = useState({});

  useEffect(() => {
    const obtenerContratosYAutorizados = async () => {
      try {
        const token = localStorage.getItem("access_token");
        if (!token) {
          toast.error("No se encontró el token de autenticación.");
          return;
        }
        const decoded = jwtDecode(token);
        const vendedorId = decoded.sub ? Number(decoded.sub) : null;

        if (!vendedorId) {
          toast.error("No se pudo obtener el ID del vendedor.");
          return;
        }

        const config = {
          headers: { Authorization: `Bearer ${token}` },
        };

        // Obtener contratos
        const response = await axios.get(
          `${apiUrl}/api/v1/contratos/lista/${vendedorId}?page=${
            page + 1
          }&per_page=${rowsPerPage}`,
          config
        );
        setContratos(response.data);

        // Obtener autorizados para cada contrato
        const contratosIds = response.data.items.flatMap((cliente) =>
          cliente.contratos.map((contrato) => contrato.id)
        );
        const fetchPromises = contratosIds.map(async (contratoId) => {
          try {
            const autorizadosResponse = await axios.get(
              `${apiUrl}/api/v1/autorizados/${contratoId}`,
              config
            );
            setAutorizadosPorContrato((prev) => ({
              ...prev,
              [contratoId]: autorizadosResponse.data,
            }));
          } catch (error) {
            console.error(
              `Error al obtener autorizados para contrato ${contratoId}:`,
              error
            );
            setAutorizadosPorContrato((prev) => ({
              ...prev,
              [contratoId]: [],
            }));
          }
        });

        await Promise.all(fetchPromises);
      } catch (error) {
        toast.error("Error al obtener la lista de contratos.");
        console.error("Error al obtener contratos:", error);
        setContratos({ items: [], total: 0 });
      }
    };

    obtenerContratosYAutorizados();
  }, [page, rowsPerPage]);

  const fetchAutorizados = async (contratoId) => {
    try {
      const token = localStorage.getItem("access_token");
      const config = {
        headers: { Authorization: `Bearer ${token}` },
      };
      const response = await axios.get(
        `${apiUrl}/api/v1/autorizados/${contratoId}`,
        config
      );
      setAutorizadosPorContrato((prev) => ({
        ...prev,
        [contratoId]: response.data,
      }));
    } catch (error) {
      console.error("Error al obtener autorizados:", error);
      toast.error("Error al obtener los autorizados.");
      setAutorizadosPorContrato((prev) => ({
        ...prev,
        [contratoId]: [],
      }));
    }
  };

  const handleToggleContrato = (contratoId) => {
    if (expandedContrato === contratoId) {
      setExpandedContrato(null);
    } else {
      setExpandedContrato(contratoId);
      // No es necesario volver a llamar a fetchAutorizados aquí, ya que los datos ya están cargados
    }
  };

  const handleChangePage = (event, newPage) => {
    setPage(newPage);
  };

  const handleChangeRowsPerPage = (event) => {
    setRowsPerPage(parseInt(event.target.value, 10));
    setPage(0);
  };

  const handleOpenAutorizadosModal = (contrato) => {
    setSelectedContrato(contrato);
    setOpenAutorizadosModal(true);
  };

  const handleCloseAutorizadosModal = () => {
    setOpenAutorizadosModal(false);
    setSelectedContrato(null);
  };

  const handleSubmit = async (autorizados) => {
    const token = localStorage.getItem("access_token");
    const decoded = jwtDecode(token);
    const vendedorId = decoded.sub ? Number(decoded.sub) : null;

    if (!vendedorId) {
      toast.error("No se pudo identificar al vendedor.");
      return;
    }

    try {
      const formData = new FormData();
      formData.append("contrato_id", selectedContrato.id);
      formData.append("vendedor_id", vendedorId);
      const autorizadosFiltrados = autorizados
        .filter((a) => a.nombre && a.cedula)
        .map((a) => ({
          nombre: a.nombre,
          cedula: a.cedula,
          direccion: a.direccion || "",
          fecha_nacimiento: a.fecha_nacimiento || "",
          edad: a.edad || "",
          telefono: a.telefono || "",
          celular: a.celular || "",
          email: a.email || "",
          nombre_documento: a.nombre_documento || "",
        }));

      formData.append("autorizados_str", JSON.stringify(autorizadosFiltrados));

      autorizadosFiltrados.forEach((autorizado, index) => {
        if (autorizados[index].documento_identidad) {
          formData.append("files", autorizados[index].documento_identidad);
        }
      });

      const config = {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "multipart/form-data",
        },
      };

      await axios.post(`${apiUrl}/api/v1/autorizados`, formData, config);
      toast.success("Personas autorizadas registradas con éxito");
      // Actualizar los contratos y autorizados
      const contratosResponse = await axios.get(
        `${apiUrl}/api/v1/contratos/lista/${vendedorId}?page=${
          page + 1
        }&per_page=${rowsPerPage}`,
        config
      );
      setContratos(contratosResponse.data);
      fetchAutorizados(selectedContrato.id);
      handleCloseAutorizadosModal();
    } catch (error) {
      console.error("Error al registrar autorizados:", error);
      toast.error(
        `Error al registrar: ${error.response?.data?.detail || error.message}`
      );
    }
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        variant="h6"
        align="center"
        gutterBottom
        sx={{
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          marginBottom: 2,
        }}
      >
        GESTIÓN DE PERSONAS AUTORIZADAS
      </Typography>
      <TableContainer component={Paper} sx={{ boxShadow: 3 }}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Nº Contrato</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
              <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
              <TableCell sx={{ color: "#fff" }}>Fecha de Creación</TableCell>
              <TableCell sx={{ color: "#fff" }}>Autorizados</TableCell>
              <TableCell sx={{ color: "#fff" }}>Acción</TableCell>
              <TableCell sx={{ color: "#fff" }}>Detalles</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {contratos.items.length > 0 ? (
              contratos.items.flatMap((cliente) =>
                cliente.contratos.map((contrato) => (
                  <React.Fragment key={contrato.id}>
                    <TableRow>
                      <TableCell>{contrato.numero_contrato}</TableCell>
                      <TableCell>{cliente.nombre}</TableCell>
                      <TableCell>{contrato.tipo_servicio}</TableCell>
                      <TableCell>
                        {new Date(contrato.fecha_creacion).toLocaleDateString(
                          "es-ES",
                          { day: "2-digit", month: "2-digit", year: "numeric" }
                        )}
                      </TableCell>
                      <TableCell>
                        {autorizadosPorContrato[contrato.id]?.length || 0}/2
                      </TableCell>
                      <TableCell>
                        <Button
                          onClick={() => handleOpenAutorizadosModal(contrato)}
                          disabled={
                            autorizadosPorContrato[contrato.id]?.length >= 2
                          }
                          sx={{
                            backgroundColor:
                              autorizadosPorContrato[contrato.id]?.length >= 2
                                ? "#B0BEC5"
                                : "#0A4575",
                            color: "#FFFFFF",
                            "&:hover": {
                              backgroundColor:
                                autorizadosPorContrato[contrato.id]?.length >= 2
                                  ? "#B0BEC5"
                                  : "#083a5e",
                            },
                          }}
                        >
                          Agregar Autorizados
                        </Button>
                      </TableCell>
                      <TableCell>
                        <IconButton
                          onClick={() => handleToggleContrato(contrato.id)}
                        >
                          {expandedContrato === contrato.id ? (
                            <ExpandLessIcon />
                          ) : (
                            <ExpandMoreIcon />
                          )}
                        </IconButton>
                      </TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell colSpan={8} sx={{ padding: 0, width: "100%" }}>
                        <Collapse
                          in={expandedContrato === contrato.id}
                          timeout="auto"
                          unmountOnExit
                        >
                          <Box sx={{ padding: 2 }}>
                            <Typography
                              variant="h6"
                              gutterBottom
                              sx={{
                                fontFamily: "Bakbak One, sans-serif",
                                color: "#0A4575",
                              }}
                            >
                              DETALLES DE AUTORIZADOS
                            </Typography>
                            <TableContainer
                              component={Paper}
                              sx={{ boxShadow: 3 }}
                            >
                              <Table size="small">
                                <TableHead sx={{ backgroundColor: "#0A4575" }}>
                                  <TableRow>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Nombre
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Cédula
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Teléfono
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Email
                                    </TableCell>
                                  </TableRow>
                                </TableHead>
                                <TableBody>
                                  {autorizadosPorContrato[contrato.id]?.length >
                                  0 ? (
                                    autorizadosPorContrato[contrato.id].map(
                                      (autorizado) => (
                                        <TableRow key={autorizado.id}>
                                          <TableCell>
                                            {autorizado.nombre}
                                          </TableCell>
                                          <TableCell>
                                            {autorizado.cedula}
                                          </TableCell>
                                          <TableCell>
                                            {autorizado.telefono}
                                          </TableCell>
                                          <TableCell>
                                            {autorizado.email}
                                          </TableCell>
                                        </TableRow>
                                      )
                                    )
                                  ) : (
                                    <TableRow>
                                      <TableCell colSpan={4} align="center">
                                        No hay personas autorizadas registradas.
                                      </TableCell>
                                    </TableRow>
                                  )}
                                </TableBody>
                              </Table>
                            </TableContainer>
                          </Box>
                        </Collapse>
                      </TableCell>
                    </TableRow>
                  </React.Fragment>
                ))
              )
            ) : (
              <TableRow>
                <TableCell colSpan={8} align="center">
                  No hay contratos disponibles.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
        <TablePagination
          rowsPerPageOptions={[5, 10, 25]}
          component="div"
          count={contratos.total}
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={handleChangePage}
          onRowsPerPageChange={handleChangeRowsPerPage}
          labelRowsPerPage="Filas por página:"
          labelDisplayedRows={({ from, to, count }) =>
            `${from}-${to} de ${count}`
          }
        />
      </TableContainer>
      <Button
        variant="contained"
        component={Link}
        to="/ventas"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
      <AutorizadosModal
        open={openAutorizadosModal}
        onClose={handleCloseAutorizadosModal}
        contrato={selectedContrato}
        onSubmit={handleSubmit}
      />
    </Box>
  );
};

export default AutorizadosPage;
