import React, { useState, useEffect } from "react";
import {
  Grid,
  Box,
  TextField,
  Typography,
  Select,
  MenuItem,
  FormControl,
  InputLabel,
  Button,
  FormControlLabel,
  Checkbox,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";
import { CrearItemServicio } from "../utils/CrearItemServicio";

const apiUrl = process.env.REACT_APP_API_URL;

const HonraExequias = ({ agregarAlPresupuesto, resetTrigger }) => {
  const [parcelaData, setParcelaData] = useState(null);
  const [planes, setPlanes] = useState([]);
  const [planSeleccionado, setPlanSeleccionado] = useState(null);
  const [codigo, setCodigo] = useState("");
  const [errorCodigo, setErrorCodigo] = useState(false);
  const [inicial, setInicial] = useState(0);
  const [cuotas, setCuotas] = useState(1);
  const [montoFinanciar, setMontoFinanciar] = useState(0);
  const [montoCuotas, setMontoCuotas] = useState(0);
  const [modalidad, setModalidad] = useState("");

  // NUEVO: estados de descuentos
  const [pagoTotalEnDivisas, setPagoTotalEnDivisas] = useState(false);
  const [pagoInicialEnDivisas, setPagoInicialEnDivisas] = useState(false);

  const disponible = parcelaData ? parcelaData.disponible || 0 : 0;
  const precioContado = planSeleccionado ? planSeleccionado.de_contado || 0 : 0;
  const puestos = planSeleccionado
    ? parseInt(planSeleccionado.puestos) || 0
    : 0;

  useEffect(() => {
    const fetchPlanes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(
          `${apiUrl}/api/planes/solicitudes?tipo_servicio==honra-exequias`,
          { headers: { Authorization: `Bearer ${token}` } }
        );
        setPlanes(response.data);
      } catch (error) {
        console.error("Error al obtener los planes:", error);
        toast.error("Error al cargar las honras de exequias.");
      }
    };
    fetchPlanes();
  }, []);

  useEffect(() => {
    setParcelaData(null);
    setPlanSeleccionado(null);
    setCodigo("");
    setErrorCodigo(false);
    setModalidad("");
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  }, [resetTrigger]);

  const handlePlanChange = (e) => {
    const planId = e.target.value;
    const plan = planes.find((p) => p.id === planId);
    setPlanSeleccionado(plan);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  };

  const handleModalidadChange = (e) => {
    setModalidad(e.target.value);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  };

  const handleCodigoChange = (e) => {
    const input = e.target.value.toUpperCase();
    setCodigo(input);
    setErrorCodigo(false);
    setParcelaData(null);
  };

  // Calcular montos
  useEffect(() => {
    if (!parcelaData || !modalidad || !planSeleccionado) {
      setMontoFinanciar(0);
      setMontoCuotas(0);
      return;
    }

    if (modalidad === "De Contado") {
      let precio = precioContado;
      if (pagoTotalEnDivisas) {
        precio = precio - precio * 0.2; // 20% descuento
      }
      setMontoFinanciar(precio);
    }

    if (modalidad === "Financiado") {
      const cuotaKey = `cuota_${cuotas}`;
      let total =
        planSeleccionado[cuotaKey] || planSeleccionado.financiado || 0;

      // Descuento si pago inicial en divisas
      let montoInicialValido = parseFloat(inicial) || 0;
      if (pagoInicialEnDivisas) {
        montoInicialValido = montoInicialValido * 1.2;
      }

      const saldo = total - montoInicialValido;
      setMontoFinanciar(saldo > 0 ? saldo : 0);

      if (cuotas === 1) {
        setMontoCuotas(saldo > 0 ? saldo : 0);
      } else {
        const restante = saldo / (cuotas - 1);
        setMontoCuotas(restante > 0 ? restante : 0);
      }
    }
  }, [
    parcelaData,
    modalidad,
    inicial,
    cuotas,
    planSeleccionado,
    pagoTotalEnDivisas,
    pagoInicialEnDivisas,
  ]);

  const handleAgregar = () => {
    if (!codigo || !parcelaData || !planSeleccionado || !modalidad) {
      toast.error("Por favor, complete todos los campos requeridos.");
      return;
    }

    if (disponible === 0 || puestos > disponible) {
      toast.error(
        "No se puede agregar una parcela que no esté disponible o exceda los puestos."
      );
      return;
    }

    if (modalidad === "Financiado" && inicial > montoFinanciar + inicial) {
      toast.error(
        "El monto inicial no puede exceder el precio total financiado."
      );
      return;
    }

    const item = CrearItemServicio({
      tipo_servicio: "parcelas",
      tipo_contrato: "parcelas",
      codigo,
      descripcion: planSeleccionado.descripcion,
      modalidadPago: modalidad,
      precioTotal: modalidad === "De Contado" ? montoFinanciar : montoFinanciar,
      pagoInicial: modalidad === "Financiado" ? inicial : 0,
      cuotas: modalidad === "Financiado" ? cuotas : 0,
      montoPorCuota: modalidad === "Financiado" ? montoCuotas : 0,
      detalles_adicionales: { codigo, puestos, disponible },
    });

    agregarAlPresupuesto(item);
  };

  return (
    <>
      <Typography
        sx={{
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          textAlign: "center",
        }}
        variant="h5"
      >
        SOLICITUD DE HONRA DE EXEQUIAS
      </Typography>

      <Box>
        <Typography
          sx={{
            marginTop: "15px",
            marginBottom: "10px",
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
          variant="h6"
        >
          PLAN A SOLICITAR
        </Typography>
      </Box>

      <Grid container spacing={2} sx={{ mt: 2 }}>
        {disponible > 0 && (
          <Grid item xs={3}>
            <FormControl
              fullWidth
              size="small"
              variant="outlined"
              sx={{ boxShadow: 3 }}
            >
              <InputLabel id="puestos-label">Plan</InputLabel>
              <Select
                labelId="puestos-label"
                id="puestos"
                label="Puestos"
                value={planSeleccionado ? planSeleccionado.id : ""}
                onChange={handlePlanChange}
                autoComplete="off"
              >
                <MenuItem value="" disabled>
                  Seleccione
                </MenuItem>
                {planes
                  .filter(
                    (plan) =>
                      parseInt(plan.puestos) <= disponible &&
                      (plan.is_active === true || plan.is_active === 1)
                  )
                  .map((plan) => (
                    <MenuItem key={plan.id} value={plan.id}>
                      {plan.descripcion}
                    </MenuItem>
                  ))}
              </Select>
            </FormControl>
          </Grid>
        )}

        {disponible > 0 && planSeleccionado && (
          <Grid item xs={2}>
            <FormControl
              fullWidth
              size="small"
              variant="outlined"
              sx={{ boxShadow: 3 }}
            >
              <InputLabel id="modalidad-label">Modalidad</InputLabel>
              <Select
                labelId="modalidad-label"
                id="modalidad"
                label="Modalidad"
                value={modalidad}
                onChange={handleModalidadChange}
                autoComplete="off"
              >
                <MenuItem value="" disabled>
                  Seleccione
                </MenuItem>
                <MenuItem value="De Contado">De Contado</MenuItem>
                <MenuItem value="Financiado">Financiado</MenuItem>
              </Select>
            </FormControl>
          </Grid>
        )}

        {disponible > 0 && planSeleccionado && modalidad && (
          <>
            {modalidad === "De Contado" && (
              <>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio al Contado"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={3}>
                  <FormControlLabel
                    control={
                      <Checkbox
                        checked={pagoTotalEnDivisas}
                        onChange={(e) =>
                          setPagoTotalEnDivisas(e.target.checked)
                        }
                        color="primary"
                      />
                    }
                    label="Pago en Divisas (20% Descuento)"
                  />
                </Grid>
                <Grid item xs={2}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    disabled={disponible === 0 || puestos > disponible}
                    fullWidth
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}

            {modalidad === "Financiado" && (
              <>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Monto a Financiar"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Monto Inicial ($)"
                    variant="outlined"
                    size="small"
                    type="number"
                    value={inicial}
                    onChange={(e) =>
                      setInicial(parseFloat(e.target.value) || 0)
                    }
                    sx={{ boxShadow: 3 }}
                    inputProps={{ min: 0 }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <FormControl
                    fullWidth
                    size="small"
                    variant="outlined"
                    sx={{ boxShadow: 3 }}
                  >
                    <InputLabel id="cuotas-label">Cuotas</InputLabel>
                    <Select
                      labelId="cuotas-label"
                      id="cuotas"
                      label="Cuotas"
                      value={cuotas}
                      onChange={(e) => setCuotas(parseInt(e.target.value) || 1)}
                      autoComplete="off"
                    >
                      {[1, 2, 3, 4, 5, 6].map((num) => (
                        <MenuItem key={num} value={num}>
                          {num}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Monto por Cuota"
                    variant="outlined"
                    size="small"
                    value={`$${montoCuotas.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={3}>
                  <FormControlLabel
                    control={
                      <Checkbox
                        checked={pagoInicialEnDivisas}
                        onChange={(e) =>
                          setPagoInicialEnDivisas(e.target.checked)
                        }
                        color="primary"
                      />
                    }
                    label="Inicial en Divisas (20% Descuento)"
                  />
                </Grid>
                <Grid item xs={3}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    disabled={disponible === 0 || puestos > disponible}
                    fullWidth
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}
          </>
        )}
      </Grid>
    </>
  );
};

export default HonraExequias;
