import React, { useState, useEffect } from "react";
import {
  Grid,
  Box,
  TextField,
  Typography,
  Select,
  MenuItem,
  FormControl,
  FormControlLabel,
  InputLabel,
  Button,
  Dialog,
  DialogTitle,
  DialogContent,
  IconButton,
  Checkbox,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";
import HelpOutlineIcon from "@mui/icons-material/HelpOutline";

const apiUrl = process.env.REACT_APP_API_URL;

const Columbarios = ({ agregarAlPresupuesto, resetTrigger }) => {
  const [codigo, setCodigo] = useState("");
  const [columbario, setColumbario] = useState({});
  const [modalidadPago, setModalidadPago] = useState("");
  const [inicial, setInicial] = useState("");
  const [cuotas, setCuotas] = useState(1);
  const [montoCuotas, setMontoCuotas] = useState(0);
  const [precioPlan, setPrecioPlan] = useState(0);
  const [isOpen, setIsOpen] = useState(false);
  const [montoFinanciar, setMontoFinanciar] = useState(0);
  const [metodoPago, setMetodoPago] = useState("");
  const [descuento, setDescuento] = useState(0);
  const [precioDescuento, setPrecioDescuento] = useState(0);
  const [pagoTotalEnDivisas, setPagoTotalEnDivisas] = useState(false);
  const [pagoInicialEnDivisas, setPagoInicialEnDivisas] = useState(false);

  const columbarios = [
    {
      nombre: "Columbario Izquierdo - Externo",
      url: `${apiUrl}/uploads/mapas/Nichos - Columbarios -I-E.jpg`,
    },
    {
      nombre: "Columbario Izquierdo - Interno",
      url: `${apiUrl}/uploads/mapas/Nichos - Columbarios -I-I.jpg`,
    },
    {
      nombre: "Columbario Derecho - Interno",
      url: `${apiUrl}/uploads/mapas/Nichos - Columbarios -D-E.jpg`,
    },
    {
      nombre: "Columbario Derecho - Externo",
      url: `${apiUrl}/uploads/mapas/Nichos - Columbarios -D-I.jpg`,
    },
  ];

  // Manejar cambios en el código
  const handleCodigoChange = (e) => {
    setCodigo(e.target.value);
  };

  const parseNumber = (value) => {
    if (!value) return 0;
    return parseFloat(value.replace(",", "."));
  };

  // Consultar datos del columbario por código
  const handleCodigo = async () => {
    if (!codigo) return;

    try {
      const response = await axios.get(
        `${apiUrl}/api/consulta-columbario/${codigo}`
      );
      const data = response.data;

      if (data.estado.toLowerCase() === "ocupado") {
        toast.error(
          "El nicho seleccionado está ocupado. Por favor, elija otro."
        );
        setColumbario({});
        return;
      }

      if (parseInt(data.cantidad_nichos) === 0) {
        toast.error("El nicho seleccionado no tiene disponibilidad.");
        setColumbario({});
        return;
      }

      setColumbario({
        codigo: data.codigo,
        plan: data.plan || `Nicho ${data.codigo}`,
        descripcion: data.descripcion || `Nicho ${data.codigo}`,
        mto_contado: parseNumber(data.mto_contado) || 1000,
        mto_financiado: parseNumber(data.mto_financiado) || 1200,
        capilla: data.capilla || "",
        lado: data.lado || "",
        cantidad_nichos: parseInt(data.cantidad_nichos) || 1,
        fila: data.fila || "",
        columna: data.columna || "",
      });
      setPrecioPlan(0);
    } catch (error) {
      //console.error("Error al consultar los datos:", error);
      if (
        error.response?.status === 400 &&
        error.response?.data?.detail.includes("está reservado")
      ) {
        toast.error(error.response.data.detail);
      } else if (error.response?.status === 404) {
        toast.error("No se encontraron datos para el código ingresado.");
      } else {
        toast.error("El nicho está ocupado o no está disponible.");
      }
      setColumbario({});
    }
  };

  // Calcular monto por cuota
  useEffect(() => {
    if (!columbario?.codigo || !modalidadPago) {
      setPrecioPlan(0);
      setMontoFinanciar(0);
      setMontoCuotas(0);
      return;
    }

    if (modalidadPago === "De Contado") {
      let precio = columbario.mto_contado || 0;
      let metodo = "USD";
      if (pagoTotalEnDivisas) {
        precio = precio * 0.2; // aplica 20% descuento
      }

      setPrecioPlan(precio);
      setMontoFinanciar(precio);
      setMetodoPago(metodo);
      setMontoCuotas(0);
    }

    if (modalidadPago === "Financiado") {
      let total = columbario.mto_financiado || 0;
      setPrecioPlan(total);

      let metodo = "USD";
      setMetodoPago(metodo);

      // Descuento si pago inicial en divisas
      let montoInicialValido = parseFloat(inicial) || 0;
      if (pagoInicialEnDivisas) {
        montoInicialValido = montoInicialValido * 1.2;
      }

      let saldo = 0;

      if (montoInicialValido > 0) {
        saldo = total - montoInicialValido;
      }
      setMontoFinanciar(saldo > 0 ? saldo : 0);

      if (cuotas === 1) {
        setMontoCuotas(saldo > 0 ? saldo : 0);
      } else {
        const restante = saldo / cuotas;
        setMontoCuotas(restante > 0 ? restante : 0);
      }
    }
  }, [
    columbario,
    modalidadPago,
    inicial,
    cuotas,
    pagoTotalEnDivisas,
    pagoInicialEnDivisas,
  ]);

  const handleAgregar = () => {
    if (!modalidadPago || !columbario.plan) {
      toast.error(
        "Por favor, seleccione una modalidad de pago y un columbario válido."
      );
      return;
    }

    if (
      modalidadPago === "Financiado" &&
      (parseFloat(inicial) || 0) >= precioPlan
    ) {
      toast.error(
        "El monto inicial no puede ser igual o mayor al precio total."
      );
      return;
    }

    const item = {
      tipo_servicio: "nichos-columbarios",
      descripcion: columbario.descripcion,
      modalidadPago,
      codigo: codigo,
      metodoPago: metodoPago,
      pagoInicial:
        modalidadPago === "Financiado" ? parseFloat(inicial) || 0 : 0,
      cuotas: modalidadPago === "Financiado" ? cuotas : 0,
      montoPorCuotas: modalidadPago === "Financiado" ? montoCuotas : 0,
      precioFinanciado:
        modalidadPago === "De Contado" ? montoFinanciar : montoFinanciar,
      descuento: metodoPago === "USD" ? descuento : 0,
      precioConDescuento: metodoPago === "USD" ? precioDescuento : 0,
      precioPlan: precioPlan,
      detalles_adicionales: {
        codigo: columbario.codigo,
        capilla: columbario.capilla,
        lado: columbario.lado,
        cantidad_nichos: columbario.cantidad_nichos,
        fila: columbario.fila,
        columna: columbario.columna,
        estado: "reservado",
      },
      items_adicionales: {
        metodoPago: metodoPago,
        descuento: metodoPago === "USD" ? descuento : 0,
        precioConDescuento: metodoPago === "USD" ? precioDescuento : 0,
      },
    };

    agregarAlPresupuesto(item);
    //toast.success("Servicio agregado al contrato con éxito.");
    //resetFormulario();
  };

  useEffect(() => {
    if (resetTrigger) {
      resetFormulario();
    }
  }, [resetTrigger]);

  const resetFormulario = () => {
    setCodigo("");
    setColumbario({});
    setModalidadPago("");
    setInicial("");
    setCuotas(1);
    setMontoCuotas(0);
  };

  const openImageInNewTab = (url) => {
    const img = new Image();
    img.src = url;
    img.onload = () => window.open(url, "_blank");
    img.onerror = () =>
      toast.error(`No se pudo cargar la imagen: ${url.split("/").pop()}`);
  };

  return (
    <>
      <Typography
        sx={{
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          textAlign: "center",
        }}
        variant="h5"
      >
        SOLICITUD DE CONTRATO NICHOS DE COLUMBARIOS
      </Typography>
      <Box>
        <Typography
          sx={{
            marginTop: "15px",
            marginBottom: "10px",
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
          variant="h6"
        >
          PLAN A SOLICITAR
        </Typography>
      </Box>

      <Grid container spacing={2}>
        <Grid item xs={2}>
          <TextField
            fullWidth
            label="Código"
            variant="outlined"
            size="small"
            name="codigo_columbario"
            value={codigo || ""}
            onChange={handleCodigoChange}
            onBlur={handleCodigo}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item>
          <IconButton onClick={() => setIsOpen(true)}>
            <HelpOutlineIcon />
          </IconButton>
        </Grid>
        <Dialog
          open={isOpen}
          onClose={() => setIsOpen(false)}
          maxWidth="sm"
          fullWidth
        >
          <DialogTitle
            sx={{
              fontFamily: "Bakbak One, sans-serif",
              color: "#0A4575",
              textAlign: "center",
            }}
          >
            Códigos Nichos de Columbarios
          </DialogTitle>
          <DialogContent>
            <Grid container spacing={2} justifyContent="center">
              {columbarios.map((item, index) => (
                <Grid item key={index}>
                  <Button
                    variant="outlined"
                    onClick={() => openImageInNewTab(item.url)}
                  >
                    {item.nombre}
                  </Button>
                </Grid>
              ))}
            </Grid>
          </DialogContent>
        </Dialog>

        {columbario.plan && (
          <>
            <Grid item xs={1}>
              <TextField
                fullWidth
                label="Capilla"
                variant="outlined"
                size="small"
                value={columbario.capilla || ""}
                InputProps={{ readOnly: true }}
                InputLabelProps={{ shrink: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={1}>
              <TextField
                fullWidth
                label="Lado"
                variant="outlined"
                size="small"
                value={columbario.lado || ""}
                InputProps={{ readOnly: true }}
                InputLabelProps={{ shrink: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={1}>
              <TextField
                fullWidth
                label="Nichos"
                variant="outlined"
                size="small"
                value={columbario.cantidad_nichos || ""}
                InputProps={{ readOnly: true }}
                InputLabelProps={{ shrink: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={1}>
              <TextField
                fullWidth
                label="Fila"
                variant="outlined"
                size="small"
                value={columbario.fila || ""}
                InputProps={{ readOnly: true }}
                InputLabelProps={{ shrink: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={1}>
              <TextField
                fullWidth
                label="Columna"
                variant="outlined"
                size="small"
                value={columbario.columna || ""}
                InputProps={{ readOnly: true }}
                InputLabelProps={{ shrink: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
          </>
        )}
      </Grid>
      <Grid container spacing={2} sx={{ mt: 1 }}>
        <Grid item xs={2}>
          <TextField
            fullWidth
            label="Plan"
            variant="outlined"
            size="small"
            value={columbario.plan || ""}
            InputProps={{ readOnly: true }}
            InputLabelProps={{ shrink: true }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={3}>
          <TextField
            fullWidth
            label="Descripción"
            variant="outlined"
            size="small"
            value={columbario.descripcion || ""}
            InputProps={{ readOnly: true }}
            InputLabelProps={{ shrink: true }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>

        <Grid item xs={2}>
          <FormControl
            fullWidth
            variant="outlined"
            size="small"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel id="modalidad de Pago">Modalidad de Pago</InputLabel>
            <Select
              value={modalidadPago}
              onChange={(e) => setModalidadPago(e.target.value)}
              label="Modalidad de Pago"
            >
              <MenuItem value="" disabled>
                Seleccione una opción
              </MenuItem>
              <MenuItem value="De Contado">De Contado</MenuItem>
              <MenuItem value="Financiado">Financiado</MenuItem>
            </Select>
          </FormControl>
        </Grid>

        {modalidadPago === "De Contado" && (
          <>
            <Grid item xs={2}>
              <TextField
                fullWidth
                label="Monto Contado"
                variant="outlined"
                size="small"
                value={precioPlan.toFixed(2)}
                InputProps={{ readOnly: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={3}>
              <Button
                variant="contained"
                color="primary"
                onClick={handleAgregar}
                disabled={!modalidadPago}
                sx={{
                  backgroundColor: "#0A4575",
                  "&:hover": { backgroundColor: "#083a5e" },
                }}
              >
                Crear Solicitud
              </Button>
            </Grid>
          </>
        )}

        {modalidadPago === "Financiado" && (
          <>
            <Grid item xs={2}>
              <TextField
                fullWidth
                label="Precio del Plan"
                variant="outlined"
                size="small"
                value={`$${precioPlan.toFixed(2)}`}
                sx={{ boxShadow: 3 }}
                InputProps={{ readOnly: true }}
              />
            </Grid>
            <Grid item xs={1}>
              <TextField
                fullWidth
                label="Cuotas"
                variant="outlined"
                size="small"
                type="number"
                value={cuotas}
                onChange={(e) => {
                  const value = parseInt(e.target.value) || 1;
                  if (value >= 1 && value <= 24) {
                    setCuotas(value);
                  }
                }}
                inputProps={{ min: 1, max: 6 }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={2}>
              <TextField
                fullWidth
                label="Pago Inicial"
                variant="outlined"
                size="small"
                type="number"
                value={inicial}
                onChange={(e) => setInicial(e.target.value)}
                inputProps={{ step: "0.01", min: "0" }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={2}>
              <TextField
                fullWidth
                label="Monto Financiado"
                variant="outlined"
                size="small"
                value={montoFinanciar.toFixed(2)}
                InputProps={{ readOnly: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>

            <Grid item xs={2}>
              <TextField
                fullWidth
                label="Monto por Cuota"
                variant="outlined"
                size="small"
                value={montoCuotas.toFixed(2)}
                InputProps={{ readOnly: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={2}>
              <Button
                variant="contained"
                color="primary"
                onClick={handleAgregar}
                disabled={!modalidadPago}
                sx={{
                  backgroundColor: "#0A4575",
                  "&:hover": { backgroundColor: "#083a5e" },
                }}
              >
                Crear Solicitud
              </Button>
            </Grid>
          </>
        )}
      </Grid>
    </>
  );
};

export default Columbarios;
