import React, { useState, useEffect } from "react";
import { useParams } from "react-router-dom";
import axios from "axios";
import { Link } from "react-router-dom";
import { jwtDecode } from "jwt-decode";
import {
  Box,
  Button,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Grid,
  Checkbox,
  TextField,
} from "@mui/material";
import { ArrowBack } from "@mui/icons-material";
import { toast } from "react-toastify";
import ItemServicios from "../utils/ItemServicios";

const apiUrl = process.env.REACT_APP_API_URL;

const VentasContrato = () => {
  const { producto } = useParams();
  const [items, setItems] = useState([]);
  const [itemsSeleccionados, setItemsSeleccionados] = useState([]);
  const [totalContrato, setTotalContrato] = useState(0);
  const [modalidadPago, setModalidadPago] = useState(null);
  const [clienteSeleccionado, setClienteSeleccionado] = useState(null);
  const [resetCampos, setResetCampos] = useState(0);
  const [cliente, setCliente] = useState({
    nombre: "",
    cedula: "",
    fecha_nacimiento: "",
    edad: "",
    telefono: "",
    celular: "",
    email: "",
    direccion: "",
    nombre_documento: "",
    documento_identidad: null,
  });

  const [emailValid, setEmailValid] = useState(true);
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;

  const buscarCliente = async (cedula) => {
    if (!cedula.trim()) return;

    try {
      const token = localStorage.getItem("access_token");
      if (!token) {
        toast.error("No hay token de sesión");
        return;
      }

      // PASAMOS vendedor_id COMO QUERY PARAM (informativo)
      const response = await axios.get(`${apiUrl}/api/v1/clientes/${cedula}`, {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      });

      const clienteEncontrado = response.data;
      toast.success("El cliente ya está registrado. Se asociará el contrato.");

      const fechaFormateada = clienteEncontrado.fecha_nacimiento
        ? new Date(clienteEncontrado.fecha_nacimiento)
            .toISOString()
            .split("T")[0]
        : "";

      setClienteSeleccionado(clienteEncontrado);
      setCliente({
        nombre: clienteEncontrado.nombre,
        cedula: clienteEncontrado.cedula,
        fecha_nacimiento: fechaFormateada,
        edad: clienteEncontrado.edad,
        telefono: clienteEncontrado.telefono,
        celular: clienteEncontrado.celular,
        email: clienteEncontrado.email,
        direccion: clienteEncontrado.direccion,
        documento_identidad: null,
        nombre_documento: "",
      });
    } catch (error) {
      // Si el backend devuelve 404 => cliente nuevo
      if (error.response?.status === 404) {
        toast.info(
          "El Cliente no existe. Se creará el cliente para esta solicitud."
        );
        setClienteSeleccionado(null);
        return;
      }

      // Si devuelve 403 => cliente pertenece a otro vendedor
      if (error.response?.status === 403) {
        const msg =
          error.response?.data?.detail ||
          "Este cliente se encuentra asociado a otro vendedor. Por favor comuníquese con administración.";
        toast.error(msg);
        setClienteSeleccionado(null);
        return;
      }

      console.error("Error al buscar cliente:", error);
      toast.error("Error al buscar cliente.");
    }
  };

  const handleClienteChange = (e) => {
    const { name, value } = e.target;
    let updatedFields = { [name]: value };

    if (name === "fecha_nacimiento") {
      const edadCalculada = calcularEdad(value);
      updatedFields.edad = edadCalculada.toString();
    }

    setCliente((prev) => ({
      ...prev,
      ...updatedFields,
    }));
  };

  const agregarAlPresupuesto = (item) => {
    setItems((prevItems) => [...prevItems, item]);
  };

  const handleEliminarItems = () => {
    const nuevosItems = items.filter(
      (_, index) => !itemsSeleccionados.includes(index)
    );
    setItems(nuevosItems);
    setItemsSeleccionados([]);
  };

  const handleSeleccionarItem = (index) => {
    const seleccionado = itemsSeleccionados.includes(index);
    if (seleccionado) {
      setItemsSeleccionados(itemsSeleccionados.filter((i) => i !== index));
    } else {
      setItemsSeleccionados([...itemsSeleccionados, index]);
    }
  };

  const calcularEdad = (fechaNacimiento) => {
    const hoy = new Date();
    const fecha_nacimiento = new Date(fechaNacimiento);
    let edad = hoy.getFullYear() - fecha_nacimiento.getFullYear();
    const mes = hoy.getMonth() - fecha_nacimiento.getMonth();

    if (mes < 0 || (mes === 0 && hoy.getDate() < fecha_nacimiento.getDate())) {
      edad--;
    }
    return edad;
  };

  const handleFileChange = (event) => {
    const file = event.target.files[0];
    if (file) {
      setCliente((prev) => ({
        ...prev,
        documento_identidad: file,
        nombre_documento: file.name,
      }));
    }
  };

  useEffect(() => {
    const total = items.reduce((sum, item) => sum + (item.precioPlan || 0), 0);
    const modalidadPago = items.length > 0 ? items[0].modalidadPago : null;
    setTotalContrato(total);
    setModalidadPago(modalidadPago);
  }, [items]);

  const handleSubmit = async (e) => {
    e.preventDefault();
    const token = localStorage.getItem("access_token");
    let vendedorId = null;
    let supervisorId = null;

    if (token) {
      try {
        const decoded = jwtDecode(token);
        vendedorId = decoded.sub ? Number(decoded.sub) : null;
        supervisorId = decoded.supervisor_id
          ? Number(decoded.supervisor_id)
          : null;
      } catch (error) {
        console.error("Error al decodificar el token:", error);
      }
    }

    if (!vendedorId) {
      toast.error(
        "No se pudo identificar al vendedor. Por favor, inicia sesión nuevamente."
      );
      return;
    }

    let clienteId;
    try {
      if (clienteSeleccionado) {
        clienteId = clienteSeleccionado.id;
      } else {
        if (
          !cliente.nombre ||
          !cliente.cedula ||
          !cliente.fecha_nacimiento ||
          !cliente.telefono ||
          !cliente.email ||
          !cliente.direccion
        ) {
          toast.error(
            "Por favor, complete todos los campos obligatorios del cliente."
          );
          return;
        }

        const formData = new FormData();
        formData.append("vendedor_id", vendedorId);
        formData.append("nombre", cliente.nombre);
        formData.append("cedula", cliente.cedula);
        formData.append("fecha_nacimiento", cliente.fecha_nacimiento);
        formData.append("edad", cliente.edad);
        formData.append("telefono", cliente.telefono);
        formData.append("celular", cliente.celular || "");
        formData.append("email", cliente.email);
        formData.append("direccion", cliente.direccion);
        formData.append("nombre_documento", cliente.nombre_documento || "");

        if (cliente.documento_identidad instanceof File) {
          formData.append("documento_identidad", cliente.documento_identidad);
        }

        const clienteResponse = await axios.post(
          `${apiUrl}/api/v1/clientes/`,
          formData,
          {
            headers: {
              Authorization: `Bearer ${token}`,
              "Content-Type": "multipart/form-data",
            },
          }
        );
        clienteId = clienteResponse.data.id;
      }

      // Preparar array de solicitudes
      const solicitudesData = items.map((item) => ({
        cliente_id: clienteId,
        vendedor_id: vendedorId,
        supervisor_id: supervisorId,
        tipo_servicio: item.tipo_servicio,
        descripcion: item.descripcion,
        modalidadPago: item.modalidadPago,
        fecha_creacion: new Date().toISOString(),
        fecha_vencimiento: new Date(
          new Date().getTime() + 3 * 24 * 60 * 60 * 1000
        ).toISOString(), // 3 días después
        estado: "solicitud",
        precioPlan: item.precioPlan,
        items_adicionales: {
          metodoPago: item.metodoPago,
          descuento: item.descuento,
          precioConDescuento: item.precioConDescuento,
        },
        detalles: [item],
      }));

      // Enviar todas las solicitudes al backend
      await axios.post(`${apiUrl}/api/v1/solicitudes/`, solicitudesData, {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
      });

      toast.success("Solicitudes registradas con éxito.");
      setResetCampos((prev) => prev + 1);
      setCliente({
        nombre: "",
        cedula: "",
        fecha_nacimiento: "",
        edad: "",
        telefono: "",
        celular: "",
        email: "",
        direccion: "",
        documento_identidad: null,
        nombre_documento: "",
      });
      setItems([]);
      setClienteSeleccionado(null);
      setItemsSeleccionados([]);
    } catch (error) {
      console.error("Error:", error.response?.data?.detail || error.message);
      toast.error(
        `Error al registrar solicitudes: ${
          error.response?.data?.detail || error.message
        }`
      );
    }
  };

  const validateEmail = (email) => {
    return emailRegex.test(email);
  };

  const handleBlur = (e) => {
    const { name, value } = e.target;
    if (name === "email") {
      setEmailValid(validateEmail(value));
    }
  };

  return (
    <Box
      component="form"
      onSubmit={handleSubmit}
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <ItemServicios
        tipoServicio={producto}
        agregarAlPresupuesto={agregarAlPresupuesto}
        resetTrigger={resetCampos}
      />

      <Typography
        sx={{
          marginTop: "20px",
          marginBottom: "5px",
          fontFamily: "Bakbak One",
          color: "#0A4575",
        }}
        variant="h6"
      >
        DATOS DEL CLIENTE
      </Typography>
      <Grid container spacing={3}>
        <Grid item xs={12} sm={4}>
          <TextField
            fullWidth
            label="Nombre cliente"
            variant="outlined"
            size="small"
            name="nombre"
            value={cliente.nombre}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Cédula o Rif"
            variant="outlined"
            size="small"
            name="cedula"
            value={cliente.cedula}
            onChange={handleClienteChange}
            onBlur={(e) => buscarCliente(e.target.value)}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Fecha de Nacimiento"
            variant="outlined"
            type="date"
            size="small"
            name="fecha_nacimiento"
            value={cliente.fecha_nacimiento || ""}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
            InputLabelProps={{ shrink: true }}
          />
        </Grid>
        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Edad"
            variant="outlined"
            size="small"
            name="edad"
            value={cliente.edad}
            required
            sx={{ boxShadow: 3 }}
            InputProps={{ readOnly: true }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Teléfono"
            variant="outlined"
            size="small"
            name="telefono"
            inputProps={{ maxLength: 10 }}
            value={cliente.telefono}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Celular"
            variant="outlined"
            size="small"
            name="celular"
            inputProps={{ maxLength: 11 }}
            value={cliente.celular}
            onChange={handleClienteChange}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={6}>
          <TextField
            fullWidth
            label="Email"
            variant="outlined"
            size="small"
            name="email"
            type="email"
            value={cliente.email}
            onChange={handleClienteChange}
            onBlur={handleBlur}
            required
            sx={{ boxShadow: 3 }}
            error={!emailValid}
            helperText={
              !emailValid ? "Ingresa un email válido (ejemplo@gmail.com)" : ""
            }
          />
        </Grid>
        <Grid item xs={12} sm={5}>
          <TextField
            fullWidth
            label="Dirección"
            variant="outlined"
            size="small"
            name="direccion"
            value={cliente.direccion}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={4}>
          <input
            accept="image/*"
            style={{ display: "none" }}
            id="file-upload"
            type="file"
            onChange={handleFileChange}
          />
          <label htmlFor="file-upload">
            <Button
              variant="contained"
              component="span"
              sx={{
                backgroundColor: "#0A4575",
                color: "#FFFFFF",
                "&:hover": { backgroundColor: "#083a5e" },
              }}
            >
              Subir Documento de Identidad
            </Button>
          </label>
        </Grid>
        <Grid item xs={12} sm={2}>
          {cliente.nombre_documento && (
            <Typography sx={{ marginTop: "10px", fontFamily: "Abel" }}>
              <b>{cliente.nombre_documento}</b>
            </Typography>
          )}
        </Grid>
      </Grid>
      <Typography
        sx={{
          textAlign: "center",
          marginTop: "25px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
        gutterBottom
      >
        RESUMEN DE SOLICITUD DEL CONTRATO
      </Typography>
      <TableContainer component={Paper}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}>
            <TableRow>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Código
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Descripción
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Modalidad de Pago
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Inicial
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Precio Financiado
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Monto Cuota
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Precio con Descuento
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Precio del Plan
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Seleccionar
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {items.map((item, index) => (
              <TableRow key={index}>
                <TableCell>{item.codigo}</TableCell>
                <TableCell>{item.descripcion}</TableCell>
                <TableCell>{item.modalidadPago}</TableCell>
                <TableCell>
                  {item.pagoInicial !== ""
                    ? `$${item.pagoInicial.toFixed(2)}`
                    : 0}
                </TableCell>
                <TableCell>
                  {item.modalidadPago === "Financiado"
                    ? `$${item.precioFinanciado.toFixed(2)}`
                    : "$0.00"}
                </TableCell>
                <TableCell>
                  {item.montoPorCuota !== ""
                    ? `$${parseFloat(item.montoPorCuotas).toFixed(2)}`
                    : 0}
                </TableCell>
                <TableCell>{`$${item.precioConDescuento.toFixed(
                  2
                )}`}</TableCell>
                <TableCell>{`$${item.precioPlan.toFixed(2)}`}</TableCell>
                <TableCell>
                  <Checkbox
                    checked={itemsSeleccionados.includes(index)}
                    onChange={() => handleSeleccionarItem(index)}
                    sx={{
                      color: "#0A4575",
                      "&.Mui-checked": { color: "#0A4575" },
                    }}
                  />
                </TableCell>
              </TableRow>
            ))}
            <TableRow>
              <TableCell colSpan={7} align="right">
                <Typography
                  sx={{
                    color: "#0A4575",
                    fontWeight: "bold",
                    fontSize: "16px",
                  }}
                >
                  Total:
                </Typography>
              </TableCell>
              <TableCell>
                <Typography sx={{ fontWeight: "bold", color: "#000000" }}>
                  ${totalContrato.toFixed(2)}
                </Typography>
              </TableCell>
              <TableCell />
            </TableRow>
          </TableBody>
        </Table>
      </TableContainer>
      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          marginTop: 3,
        }}
      >
        <Button
          variant="contained"
          component={Link}
          to="/ventas"
          startIcon={<ArrowBack />}
          sx={{
            backgroundColor: "#0A4575",
            color: "#FFFFFF",
            "&:hover": { backgroundColor: "#083a5e" },
          }}
        >
          Volver
        </Button>
        <Box>
          <Button
            variant="outlined"
            color="error"
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF !important",
              borderColor: "#0A4575",
              "&:hover": { backgroundColor: "#083a5e !important" },
              marginRight: 2,
            }}
            onClick={handleEliminarItems}
            disabled={itemsSeleccionados.length === 0}
          >
            Eliminar Ítems
          </Button>
          <Button
            type="submit"
            variant="contained"
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#000000" },
            }}
          >
            Guardar Solicitud
          </Button>
        </Box>
      </Box>
    </Box>
  );
};

export default VentasContrato;
