import React, { useState, useEffect } from "react";
import axios from "axios";
import {
  Box,
  Button,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  IconButton,
  Collapse,
  Paper,
} from "@mui/material";
import {
  ArrowBack,
  ExpandMore as ExpandMoreIcon,
  ExpandLess as ExpandLessIcon,
} from "@mui/icons-material";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";

const DocumentosFamiliares = () => {
  const [contratos, setContratos] = useState([]);
  const [familiaresPorContrato, setFamiliaresPorContrato] = useState({});
  const [expandedContrato, setExpandedContrato] = useState(null);
  const apiUrl = process.env.REACT_APP_API_URL;

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("es-ES", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
    });
  };
  // Cargar contratos previsivos al montar el componente
  useEffect(() => {
    const fetchContratosPrevisivos = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(
          `${apiUrl}/api/v1/previsivos/tipo_servicio=previsivo`,
          {
            headers: { Authorization: `Bearer ${token}` },
          }
        );
        const contratosData = response.data;
        setContratos(contratosData);

        // Consultar familiares para cada contrato
        const familiaresPromises = contratosData.map(async (contrato) => {
          const familiaresResponse = await axios.get(
            `${apiUrl}/api/v1/previsivos/familiares?contrato_id=${contrato.id}`,
            { headers: { Authorization: `Bearer ${token}` } }
          );
          return {
            contratoId: contrato.id,
            familiares: familiaresResponse.data,
          };
        });

        const familiaresResults = await Promise.all(familiaresPromises);
        const familiaresMap = familiaresResults.reduce((acc, result) => {
          acc[result.contratoId] = result.familiares;
          return acc;
        }, {});
        setFamiliaresPorContrato(familiaresMap);
      } catch (error) {
        console.error(
          "Error al cargar contratos previsivos o familiares:",
          error
        );
        toast.error("Error al cargar los datos.");
      }
    };
    fetchContratosPrevisivos();
  }, [apiUrl]);

  // Manejar la subida de documento de identidad
  const handleFileChange = async (familiarId, event) => {
    const file = event.target.files[0];
    if (!file) return;

    try {
      const token = localStorage.getItem("access_token");
      const formData = new FormData();
      formData.append("documento_identidad", file);
      formData.append("nombre_documento", file.name);

      await axios.patch(
        `${apiUrl}/api/v1/previsivos/familiares/${familiarId}/documento`,
        formData,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "multipart/form-data",
          },
        }
      );

      // Actualizar el estado local para reflejar el documento subido
      setFamiliaresPorContrato((prev) => {
        const updated = { ...prev };
        Object.keys(updated).forEach((contratoId) => {
          updated[contratoId] = updated[contratoId].map((familiar) =>
            familiar.id === familiarId
              ? { ...familiar, nombre_documento: file.name }
              : familiar
          );
        });
        return updated;
      });

      toast.success("Documento de identidad subido exitosamente.");
    } catch (error) {
      console.error("Error al subir documento:", error);
      toast.error(`Error: ${error.response?.data?.detail || error.message}`);
    }
  };

  // Manejar la expansión/colapso de contratos
  const handleToggleContrato = (contratoId) => {
    setExpandedContrato(expandedContrato === contratoId ? null : contratoId);
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        variant="h6"
        align="center"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575", mb: 2 }}
      >
        SUBIR DOCUMENTOS DE IDENTIDAD DE LOS FAMILIARES PREVISIVOS
      </Typography>

      <TableContainer component={Paper} sx={{ boxShadow: 3 }}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Número de Contrato</TableCell>
              <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
              <TableCell sx={{ color: "#fff" }}>Fecha de Creación</TableCell>
              <TableCell sx={{ color: "#fff" }}>Monto Total</TableCell>
              <TableCell sx={{ color: "#fff" }}></TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {contratos.length > 0 ? (
              contratos.map((contrato) => (
                <React.Fragment key={contrato.id}>
                  <TableRow>
                    <TableCell>{contrato.numero_contrato}</TableCell>
                    <TableCell>{contrato.tipo_servicio}</TableCell>
                    <TableCell>{formatDate(contrato.fecha_creacion)}</TableCell>
                    <TableCell>
                      ${parseFloat(contrato.precioPlan).toFixed(2)}
                    </TableCell>
                    <TableCell>
                      <IconButton
                        onClick={() => handleToggleContrato(contrato.id)}
                      >
                        {expandedContrato === contrato.id ? (
                          <ExpandLessIcon />
                        ) : (
                          <ExpandMoreIcon />
                        )}
                      </IconButton>
                    </TableCell>
                  </TableRow>
                  <TableRow>
                    <TableCell colSpan={5}>
                      <Collapse
                        in={expandedContrato === contrato.id}
                        timeout="auto"
                        unmountOnExit
                      >
                        <Box sx={{ p: 1, backgroundColor: "#fafafa" }}>
                          <Typography
                            variant="h6"
                            gutterBottom
                            sx={{
                              fontFamily: "Bakbak One, sans-serif",
                              color: "#0A4575",
                            }}
                          >
                            FAMILIARES
                          </Typography>
                          <TableContainer
                            component={Paper}
                            sx={{ boxShadow: 3 }}
                          >
                            <Table size="small">
                              <TableHead sx={{ backgroundColor: "#0A4575" }}>
                                <TableRow>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Nombre
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Cédula
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Parentesco
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Documento de Identidad
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Acción
                                  </TableCell>
                                </TableRow>
                              </TableHead>
                              <TableBody>
                                {(familiaresPorContrato[contrato.id] || [])
                                  .length === 0 ? (
                                  <TableRow>
                                    <TableCell colSpan={5} align="center">
                                      No hay familiares asociados a este
                                      contrato.
                                    </TableCell>
                                  </TableRow>
                                ) : (
                                  familiaresPorContrato[contrato.id].map(
                                    (familiar) => (
                                      <TableRow key={familiar.id}>
                                        <TableCell>
                                          {familiar.nombre || "N/A"}
                                        </TableCell>
                                        <TableCell>
                                          {familiar.cedula || "N/A"}
                                        </TableCell>
                                        <TableCell>
                                          {familiar.parentesco || "N/A"}
                                        </TableCell>
                                        <TableCell>
                                          {familiar.nombre_documento ||
                                            "No subido"}
                                        </TableCell>
                                        <TableCell>
                                          {familiar.nombre_documento ? (
                                            <Typography
                                              sx={{
                                                fontFamily: "Abel",
                                                color: "#0A4575",
                                              }}
                                            >
                                              <Button
                                                sx={{
                                                  backgroundColor: "#4CAF50",
                                                  color: "#FFFFFF",
                                                }}
                                              >
                                                DOCUMENTO SUBIDO
                                              </Button>
                                            </Typography>
                                          ) : (
                                            <>
                                              <input
                                                accept="image/*,application/pdf"
                                                id={`file-upload-${familiar.id}`}
                                                type="file"
                                                style={{ display: "none" }}
                                                onChange={(e) =>
                                                  handleFileChange(
                                                    familiar.id,
                                                    e
                                                  )
                                                }
                                              />
                                              <label
                                                htmlFor={`file-upload-${familiar.id}`}
                                              >
                                                <Button
                                                  variant="contained"
                                                  component="span"
                                                  sx={{
                                                    backgroundColor: "#0A4575",
                                                    color: "#FFFFFF",
                                                    "&:hover": {
                                                      backgroundColor:
                                                        "#083a5e",
                                                    },
                                                  }}
                                                >
                                                  Subir Documento
                                                </Button>
                                              </label>
                                            </>
                                          )}
                                        </TableCell>
                                      </TableRow>
                                    )
                                  )
                                )}
                              </TableBody>
                            </Table>
                          </TableContainer>
                        </Box>
                      </Collapse>
                    </TableCell>
                  </TableRow>
                </React.Fragment>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={4} align="center">
                  No hay contratos previsivos disponibles.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </TableContainer>

      <Box sx={{ display: "flex", justifyContent: "flex-start", marginTop: 3 }}>
        <Button
          variant="contained"
          component={Link}
          to="/ventas"
          startIcon={<ArrowBack />}
          sx={{
            backgroundColor: "#0A4575",
            color: "#FFFFFF",
            "&:hover": { backgroundColor: "#083a5e" },
          }}
        >
          Volver
        </Button>
      </Box>
    </Box>
  );
};

export default DocumentosFamiliares;
