import React, { useState } from "react";
import {
  Modal,
  Box,
  Typography,
  TextField,
  Button,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";

const apiUrl = process.env.REACT_APP_API_URL;

const style = {
  position: "absolute",
  top: "50%",
  left: "50%",
  transform: "translate(-50%, -50%)",
  width: 400,
  bgcolor: "background.paper",
  boxShadow: 24,
  p: 4,
  borderRadius: 2,
};

const EditarContratoModal = ({
  open,
  onClose,
  contrato,
  onContratoActualizado,
}) => {
  const [modalidadPago, setModalidadPago] = useState(
    contrato?.modalidad_pago || "Contado"
  );
  const [pagoInicial, setPagoInicial] = useState("");
  const [numeroCuotas, setNumeroCuotas] = useState(1);
  const [montoCuota, setMontoCuota] = useState("");
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e) => {
    e.preventDefault();
    setLoading(true);

    try {
      const token = localStorage.getItem("access_token");
      const config = {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      };

      const precioTotal = contrato?.monto || 0;
      const montoRestante = precioTotal - (pagoInicial || 0);
      if (modalidadPago === "Financiado" && montoRestante <= 0) {
        toast.error("El pago inicial debe ser menor al precio total.");
        setLoading(false);
        return;
      }

      const data = {
        modalidad_pago: modalidadPago,
        pago_inicial:
          modalidadPago === "Financiado" ? Number(pagoInicial) : precioTotal,
        numero_cuotas:
          modalidadPago === "Financiado" ? Number(numeroCuotas) : 0,
        monto_cuota: modalidadPago === "Financiado" ? Number(montoCuota) : 0,
        precio_restante: modalidadPago === "Financiado" ? montoRestante : 0,
      };

      await axios.put(
        `${apiUrl}/api/contratos/${contrato.numero_contrato}/detalles`,
        data,
        config
      );

      toast.success("Contrato actualizado exitosamente.");
      onContratoActualizado();
      onClose();
    } catch (error) {
      console.error("Error al actualizar el contrato:", error);
      toast.error("No se pudo actualizar el contrato.");
    } finally {
      setLoading(false);
    }
  };

  return (
    <Modal open={open} onClose={onClose}>
      <Box sx={style}>
        <Typography variant="h6" component="h2" gutterBottom>
          Editar Contrato #{contrato?.numero_contrato}
        </Typography>
        <form onSubmit={handleSubmit}>
          <FormControl fullWidth sx={{ mb: 2 }}>
            <InputLabel>Modalidad de Pago</InputLabel>
            <Select
              value={modalidadPago}
              onChange={(e) => setModalidadPago(e.target.value)}
              label="Modalidad de Pago"
            >
              <MenuItem value="Contado">Contado</MenuItem>
              <MenuItem value="Financiado">Financiado</MenuItem>
            </Select>
          </FormControl>

          {modalidadPago === "Financiado" && (
            <>
              <TextField
                fullWidth
                label="Pago Inicial"
                type="number"
                value={pagoInicial}
                onChange={(e) => setPagoInicial(e.target.value)}
                sx={{ mb: 2 }}
                required
              />
              <FormControl fullWidth sx={{ mb: 2 }}>
                <InputLabel>Número de Cuotas</InputLabel>
                <Select
                  value={numeroCuotas}
                  onChange={(e) => setNumeroCuotas(e.target.value)}
                  label="Número de Cuotas"
                >
                  <MenuItem value={1}>1</MenuItem>
                  <MenuItem value={2}>2</MenuItem>
                  <MenuItem value={3}>3</MenuItem>
                </Select>
              </FormControl>
              <TextField
                fullWidth
                label="Monto por Cuota"
                type="number"
                value={montoCuota}
                onChange={(e) => setMontoCuota(e.target.value)}
                sx={{ mb: 2 }}
                required
              />
            </>
          )}

          <Box sx={{ display: "flex", justifyContent: "flex-end", gap: 2 }}>
            <Button onClick={onClose} disabled={loading}>
              Cancelar
            </Button>
            <Button
              type="submit"
              variant="contained"
              color="primary"
              disabled={loading}
            >
              {loading ? "Guardando..." : "Guardar Cambios"}
            </Button>
          </Box>
        </form>
      </Box>
    </Modal>
  );
};

export default EditarContratoModal;
