import React, { useEffect, useState } from "react";
import {
  Box,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Typography,
  Button,
  Dialog,
  DialogTitle,
  DialogContent,
} from "@mui/material";
import axios from "axios";
import { jwtDecode } from "jwt-decode";
import { TablasEmail } from "./EmailTablas";
import { toast } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";
import { Link } from "react-router-dom";

const ListaEmail = () => {
  const [solicitud, setSolicitud] = useState([]);
  const [detallesSolicitud, setDetallesSolicitud] = useState([]);
  const [reportePagos, setReportePagos] = useState([]); // Nuevo estado para reportes de pago
  const [openDialog, setOpenDialog] = useState(false);
  const [tipoServicio, setTipoServicio] = useState("");
  const [selectedSolicitud, setSelectedSolicitud] = useState(null);
  const [clienteSeleccionado, setClienteSeleccionado] = useState(null);
  const [difuntoSeleccionado, setDifunto] = useState(null);
  const apiUrl = process.env.REACT_APP_API_URL;

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("es-ES", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
    });
  };

  useEffect(() => {
    const obtenerSolicituds = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const decoded = jwtDecode(token);
        const vendedorId = decoded.sub ? Number(decoded.sub) : null;

        if (!vendedorId) {
          console.error("Error: vendedorId es null");
          return;
        }

        const config = {
          headers: {
            Authorization: `Bearer ${token}`,
          },
        };
        const response = await axios.get(
          `${apiUrl}/api/v1/vendedor/solicitudes/${vendedorId}`,
          config
        );
        setSolicitud(response.data);
      } catch (error) {
        console.error("Error al obtener los Solicituds:", error);
      }
    };

    obtenerSolicituds();
  }, [apiUrl]);

  const mostrarDetallesSolicitudes = async (
    clienteId,
    SolicitudId,
    tipo_servicio
  ) => {
    try {
      const token = localStorage.getItem("access_token");
      const config = {
        headers: { Authorization: `Bearer ${token}` },
      };

      const response = await axios.get(
        `${apiUrl}/api/v1/vendedor/detalles/${clienteId}/${SolicitudId}`,
        config
      );

      setClienteSeleccionado(response.data.cliente);
      setDetallesSolicitud(response.data.detalles_solicitud);
      setSelectedSolicitud(response.data.solicitud);
      setReportePagos(response.data.reporte_pagos || []); // Guardar reportes de pago

      if (response.data.difunto) {
        setDifunto(response.data.difunto);
      } else {
        setDifunto(null);
      }

      setTipoServicio(tipo_servicio);
      setOpenDialog(true);
    } catch (error) {
      console.error("Error al obtener los detalles del Solicitud:", error);
      toast.error("Error al obtener los detalles del Solicitud.");
      setDetallesSolicitud([]);
      setReportePagos([]); // Limpiar reportes en caso de error
      setDifunto(null);
      setOpenDialog(true);
    }
  };

  const handleCloseDialog = () => {
    setOpenDialog(false);
    setDetallesSolicitud([]);
    setReportePagos([]); // Limpiar reportes al cerrar
  };

  const enviarCorreo = async () => {
    try {
      const token = localStorage.getItem("access_token");
      const decoded = jwtDecode(token);
      const user_email = decoded.email;
      const config = {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      };

      const difuntoList = difuntoSeleccionado ? [difuntoSeleccionado] : [];

      const correoData = {
        Solicitud: selectedSolicitud,
        detalles: detallesSolicitud,
        cliente: clienteSeleccionado,
        tipoServicio: tipoServicio,
        difunto: difuntoList,
        reportePagos: reportePagos, // Incluir reportes en el correo
        user_email: user_email,
      };

      await axios.post(`${apiUrl}/api/enviar_correo`, correoData, config);
      toast.success("Correo enviado exitosamente");
    } catch (error) {
      console.error("Error al enviar el correo electrónico:", error);
      alert("Ocurrió un error al enviar el correo electrónico");
    }
  };

  return (
    <Box
      component="form"
      sx={{
        maxWidth: "950px",
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        sx={{
          textAlign: "center",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
        gutterBottom
      >
        LISTA DE SOLICITUDES DE SolicitudS
      </Typography>

      <TableContainer>
        <Table>
          <TableHead>
            <TableRow sx={{ backgroundColor: "#0A4575" }}>
              <TableCell
                sx={{
                  textAlign: "center",
                  color: "#FFFFFF",
                  paddingTop: "8px",
                  paddingBottom: "8px",
                  fontSize: "16px",
                }}
              >
                No. de Transacción
              </TableCell>
              <TableCell
                sx={{
                  textAlign: "center",
                  color: "#FFFFFF",
                  paddingTop: "8px",
                  paddingBottom: "8px",
                  fontSize: "16px",
                }}
              >
                Tipo de Servicio
              </TableCell>
              <TableCell
                sx={{
                  textAlign: "center",
                  color: "#FFFFFF",
                  paddingTop: "8px",
                  paddingBottom: "8px",
                  fontSize: "16px",
                }}
              >
                Fecha Creación
              </TableCell>
              <TableCell
                sx={{
                  textAlign: "center",
                  color: "#FFFFFF",
                  paddingTop: "8px",
                  paddingBottom: "8px",
                  fontSize: "16px",
                }}
              >
                Monto Total
              </TableCell>
              <TableCell
                sx={{
                  textAlign: "center",
                  color: "#FFFFFF",
                  paddingTop: "8px",
                  paddingBottom: "8px",
                  fontSize: "16px",
                }}
              >
                Detalles
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {solicitud.length > 0 ? (
              solicitud.map((Solicitud) => (
                <TableRow key={Solicitud.id}>
                  <TableCell sx={{ textAlign: "center" }}>
                    {Solicitud.numero_solicitud}
                  </TableCell>
                  <TableCell sx={{ textAlign: "center" }}>
                    {Solicitud.tipo_servicio}
                  </TableCell>
                  <TableCell sx={{ textAlign: "center" }}>
                    {formatDate(Solicitud.fecha_creacion)}
                  </TableCell>
                  <TableCell sx={{ textAlign: "center" }}>
                    ${Solicitud.monto_total}{" "}
                  </TableCell>
                  <TableCell sx={{ textAlign: "center" }}>
                    <Button
                      variant="contained"
                      color="primary"
                      onClick={() =>
                        mostrarDetallesSolicitudes(
                          Solicitud.cliente_id,
                          Solicitud.id,
                          Solicitud.tipo_servicio
                        )
                      }
                    >
                      Ver Detalles
                    </Button>
                  </TableCell>
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={7} align="center">
                  Usted no posee solicitudes registradas.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </TableContainer>

      <Button
        variant="contained"
        component={Link}
        to="/ventas"
        sx={{
          marginTop: "10px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>

      <Dialog
        open={openDialog}
        onClose={handleCloseDialog}
        maxWidth="md"
        fullWidth
      >
        <DialogTitle
          sx={{
            textAlign: "center",
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
        >
          DETALLES DE LA SOLICITUD
        </DialogTitle>
        <DialogContent>
          <TablasEmail
            tipoServicio={tipoServicio}
            cliente={clienteSeleccionado}
            detalles={detallesSolicitud}
            difunto={difuntoSeleccionado}
            reportePagos={reportePagos} // Pasar reportes como prop
          />
          <Box
            sx={{
              display: "flex",
              justifyContent: "flex-end",
              marginTop: "20px",
            }}
          >
            <Button variant="contained" color="primary" onClick={enviarCorreo}>
              Enviar por Correo Electrónico
            </Button>
          </Box>
        </DialogContent>
      </Dialog>
    </Box>
  );
};

export default ListaEmail;
