import React, { useState, useEffect } from "react";
import {
  Grid,
  Box,
  TextField,
  Typography,
  Select,
  MenuItem,
  FormControl,
  InputLabel,
  Button,
  FormControlLabel,
  Checkbox,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";
import { CrearItemServicio } from "../utils/CrearItemServicio";

const apiUrl = process.env.REACT_APP_API_URL;

const Exequias = ({ agregarAlPresupuesto, resetTrigger }) => {
  const [planes, setPlanes] = useState([]);
  const [planSeleccionado, setPlanSeleccionado] = useState(null);
  const [inicial, setInicial] = useState(0);
  const [cuotas, setCuotas] = useState(1);
  const [montoFinanciar, setMontoFinanciar] = useState(0);
  const [montoCuotas, setMontoCuotas] = useState(0);
  const [modalidad, setModalidad] = useState("");
  const [precioPlan, setPrecioPlan] = useState(0);
  const [metodoPago, setMetodoPago] = useState("");
  const [descuento, setDescuento] = useState(0);
  const [precioDescuento, setPrecioDescuento] = useState(0);
  const [contadoChecked, setContadoChecked] = useState(false);
  const [financiadoChecked, setFinanciadoChecked] = useState(false);

  const precioContado_bcv = planSeleccionado
    ? planSeleccionado.de_contado_bcv || 0
    : 0;
  const precioContado = planSeleccionado ? planSeleccionado.de_contado || 0 : 0;

  useEffect(() => {
    const fetchPlanes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(
          `${apiUrl}/api/planes/solicitudes?tipo_servicio=honra-exequias`,
          { headers: { Authorization: `Bearer ${token}` } }
        );
        setPlanes(response.data);
      } catch (error) {
        console.error("Error al obtener los planes:", error);
        toast.error("Error al cargar los planes de honra de exequias.");
      }
    };
    fetchPlanes();
  }, []);

  useEffect(() => {
    setPlanSeleccionado(null);
    setModalidad("");
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setMetodoPago("");
    setDescuento(0);
    setPrecioDescuento(0);
    setPrecioPlan("");
  }, [resetTrigger]);

  const handlePlanChange = (e) => {
    const planId = e.target.value;
    const plan = planes.find((p) => p.id === planId);
    setPlanSeleccionado(plan);
    setPrecioPlan(0);
    setModalidad("");
    setDescuento(0);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
  };

  const handleModalidadChange = (e) => {
    setModalidad(e.target.value);
    setDescuento(0);
    setPrecioPlan(0);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
  };

  // Calcular montos
  useEffect(() => {
    if (!planSeleccionado || !modalidad) {
      setPrecioPlan(0);
      setDescuento(0);
      setMontoFinanciar(0);
      setMontoCuotas(0);
      return;
    }

    if (modalidad === "De Contado") {
      // 1️⃣ Elegir precio base según el checkbox
      let precio = contadoChecked ? precioContado : precioContado_bcv;

      // 2️⃣ Aplicar 20% de descuento si está marcado "Pago en Divisas"
      let metodo = "VES";
      let dscto = 0;
      let precioConDescuento = 0;

      if (contadoChecked) {
        metodo = "USD";
        dscto = precio * 0.2;

        precioConDescuento = precio - dscto; // precio con descuento aplicado
      }

      // 3️⃣ Actualizar estados
      setMetodoPago(metodo);
      setDescuento(dscto);
      setPrecioDescuento(precioConDescuento);
      setPrecioPlan(precio);
      setMontoFinanciar(precio);
      setMontoCuotas(0);
    }

    // Función que devuelve todos los estados listos para setear
    function calcularEstadosFinanciado({
      totalPlan,
      inicial,
      cuotas,
      financiadoChecked,
    }) {
      const montoInicialValido = parseFloat(inicial) || 0;
      const financiadoActivo = financiadoChecked && montoInicialValido > 0;

      const descuento = financiadoActivo ? montoInicialValido * 0.2 : 0;
      const metodo = financiadoActivo ? "USD" : "VES";
      const precioConDescuento = financiadoActivo ? totalPlan - descuento : 0;

      const saldoFinal = financiadoActivo
        ? totalPlan - descuento - montoInicialValido
        : montoInicialValido > 0
        ? totalPlan - montoInicialValido
        : totalPlan;

      const montoPorCuota =
        cuotas === 1 ? saldoFinal : saldoFinal / (cuotas - 1);

      return {
        montoCuotas: montoPorCuota > 0 ? montoPorCuota : 0,
        metodoPago: metodo,
        descuento,
        precioDescuento: precioConDescuento,
        precioPlan: totalPlan > 0 ? totalPlan : 0,
        montoFinanciar: saldoFinal > 0 ? saldoFinal : 0,
      };
    }

    // -------------------------------
    // Bloque principal ultra limpio
    if (modalidad === "Financiado") {
      const cuotaKey = `cuota_${cuotas}`;
      const totalPlan =
        planSeleccionado[cuotaKey] || planSeleccionado.financiado || 0;

      const estados = calcularEstadosFinanciado({
        totalPlan,
        inicial,
        cuotas,
        financiadoChecked,
      });

      // Aplicar todos los estados
      setMontoCuotas(estados.montoCuotas);
      setMetodoPago(estados.metodoPago);
      setDescuento(estados.descuento);
      setPrecioDescuento(estados.precioDescuento);
      setPrecioPlan(estados.precioPlan);
      setMontoFinanciar(estados.montoFinanciar);
    }
  }, [
    modalidad,
    inicial,
    cuotas,
    planSeleccionado,
    contadoChecked, // 🔹 importante para recalcular al marcar/desmarcar
    financiadoChecked,
  ]);

  const handleAgregar = () => {
    if (!planSeleccionado || !modalidad) {
      toast.error("Por favor, seleccione un plan y una modalidad.");
      return;
    }

    if (modalidad === "Financiado" && inicial >= montoFinanciar + inicial) {
      toast.error(
        "El monto inicial no puede exceder el precio total financiado."
      );
      return;
    }

    const item = CrearItemServicio({
      tipo_servicio: "honra-exequias",
      descripcion: planSeleccionado.descripcion,
      modalidadPago: modalidad,
      codigo: "EXE000001",
      metodoPago: metodoPago,
      pagoInicial: modalidad === "Financiado" ? inicial : 0,
      cuotas: modalidad === "Financiado" ? cuotas : 0,
      montoPorCuotas: modalidad === "Financiado" ? montoCuotas : 0,
      precioFinanciado: modalidad === "Financiado" ? montoFinanciar : 0,
      descuento: metodoPago === "USD" ? descuento : 0,
      precioConDescuento: metodoPago === "USD" ? precioDescuento : 0,
      precioPlan: precioPlan,
      detalles_adicionales: {
        tipo_servicio: "honra-exequias",
        descripcion: planSeleccionado.descripcion,
        modalidadPago: modalidad,
      },
      items_adicionales: {
        metodoPago: metodoPago,
        descuento: metodoPago === "USD" ? descuento : 0,
        precioConDescuento: metodoPago === "USD" ? precioDescuento : 0,
      },
    });

    agregarAlPresupuesto(item);
    //toast.success("Solicitud de honra de exequias agregada al presupuesto.");
  };

  return (
    <>
      <Typography
        sx={{
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          textAlign: "center",
        }}
        variant="h5"
      >
        SOLICITUD DE CONTRATO DE HONRA DE EXEQUIAS
      </Typography>
      <Box>
        <Typography
          sx={{
            marginTop: "15px",
            marginBottom: "10px",
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
          variant="h6"
        >
          PLAN A SOLICITAR
        </Typography>
      </Box>
      <Grid container spacing={2}>
        <Grid item xs={3}>
          <FormControl
            fullWidth
            size="small"
            variant="outlined"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel>Seleccione el Plan</InputLabel>
            <Select
              label="Seleccione el Plan"
              value={planSeleccionado ? planSeleccionado.id : ""}
              onChange={handlePlanChange}
              autoComplete="off"
            >
              <MenuItem value="" disabled>
                Seleccione
              </MenuItem>
              {planes
                .filter(
                  (plan) => plan.is_active !== true && plan.is_active !== 1
                )
                .map((plan) => (
                  <MenuItem key={plan.id} value={plan.id}>
                    {plan.descripcion}
                  </MenuItem>
                ))}
            </Select>
          </FormControl>
        </Grid>

        {planSeleccionado && (
          <Grid item xs={2}>
            <FormControl
              fullWidth
              size="small"
              variant="outlined"
              sx={{ boxShadow: 3 }}
            >
              <InputLabel id="modalidad-label">Modalidad</InputLabel>
              <Select
                labelId="modalidad-label"
                id="modalidad"
                label="Modalidad"
                value={modalidad}
                onChange={handleModalidadChange}
                autoComplete="off"
              >
                <MenuItem value="" disabled>
                  Seleccione
                </MenuItem>
                <MenuItem value="De Contado">De Contado</MenuItem>
                <MenuItem value="Financiado">Financiado</MenuItem>
              </Select>
            </FormControl>
          </Grid>
        )}

        {planSeleccionado && modalidad && (
          <>
            {modalidad === "De Contado" && (
              <>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio del Plan"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio con Descuento"
                    variant="outlined"
                    size="small"
                    value={`$${precioDescuento.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid sx={{ mt: 2, ml: 2 }}>
                  <FormControlLabel
                    control={
                      <Checkbox
                        checked={contadoChecked}
                        onChange={(e) => setContadoChecked(e.target.checked)}
                        color="primary"
                      />
                    }
                    label="Pago en Divisas"
                  />
                </Grid>
                <Grid item xs={2}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    fullWidth
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}

            {modalidad === "Financiado" && (
              <>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio del Plan"
                    variant="outlined"
                    size="small"
                    value={`$${precioPlan.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={1}>
                  <FormControl
                    fullWidth
                    size="small"
                    variant="outlined"
                    sx={{ boxShadow: 3 }}
                  >
                    <InputLabel id="cuotas-label">Cuotas</InputLabel>
                    <Select
                      labelId="cuotas-label"
                      id="cuotas"
                      label="Cuotas"
                      value={cuotas}
                      onChange={(e) => setCuotas(parseInt(e.target.value) || 1)}
                      autoComplete="off"
                    >
                      {[1, 2, 3, 4, 5, 6].map((num) => (
                        <MenuItem key={num} value={num}>
                          {num}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Pago Inicial ($)"
                    variant="outlined"
                    size="small"
                    type="number"
                    value={inicial}
                    onChange={(e) =>
                      setInicial(parseFloat(e.target.value) || 0)
                    }
                    sx={{ boxShadow: 3 }}
                    inputProps={{ min: 0 }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio Financiado"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Monto por Cuota"
                    variant="outlined"
                    size="small"
                    value={`$${montoCuotas.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio con Descuento"
                    variant="outlined"
                    size="small"
                    value={`$${precioDescuento.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={4}>
                  <FormControlLabel
                    control={
                      <Checkbox
                        checked={financiadoChecked}
                        onChange={(e) => setFinanciadoChecked(e.target.checked)}
                        color="primary"
                      />
                    }
                    label="Pago Inicial en Divisas"
                  />
                </Grid>
                <Grid item xs={2}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    fullWidth
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}
          </>
        )}
      </Grid>
    </>
  );
};

export default Exequias;
