// src/components/FamiliaresAutorizados.jsx
import React, { useState, useEffect } from "react";
import axios from "axios";
import {
  Box,
  Button,
  TextField,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  CircularProgress,
  Alert,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Grid,
} from "@mui/material";
import { useParams, Link } from "react-router-dom";
import { toast } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";

const apiUrl = process.env.REACT_APP_API_URL;

const FamiliaresAutorizados = () => {
  const { clienteId, contratoId } = useParams();
  const [cliente, setCliente] = useState(null);
  const [contrato, setContrato] = useState(null);
  const [familiares, setFamiliares] = useState([]);
  const [newFamiliares, setNewFamiliares] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);

  const parentescosMap = {
    1: "Padre/Madre",
    2: "Hijo(a)",
    3: "Esposo(a)/Conyugue",
    4: "Hermano(a)",
    5: "Abuelo(a)",
    6: "Tío(a)",
    7: "Primo(a)",
    8: "Otro(a)",
  };

  // Calcular edad automáticamente
  const calcularEdad = (fechaNacimiento) => {
    if (!fechaNacimiento) return "";
    const hoy = new Date();
    const fechaNac = new Date(fechaNacimiento);
    let edad = hoy.getFullYear() - fechaNac.getFullYear();
    const mes = hoy.getMonth() - fechaNac.getMonth();
    if (mes < 0 || (mes === 0 && hoy.getDate() < fechaNac.getDate())) {
      edad--;
    }
    return edad.toString();
  };

  // Fetch client, contract, and authorized family members
  useEffect(() => {
    const fetchData = async () => {
      setLoading(true);
      try {
        // Obtener cliente
        const clienteResponse = await axios.get(
          `${apiUrl}/api/v1/contratos/cliente/${clienteId}`,
          {
            headers: {
              Authorization: `Bearer ${localStorage.getItem("access_token")}`,
            },
          }
        );
        setCliente(clienteResponse.data);

        // Obtener contrato
        const contratoResponse = await axios.get(
          `${apiUrl}/api/contratos/${contratoId}`,
          {
            headers: {
              Authorization: `Bearer ${localStorage.getItem("access_token")}`,
            },
          }
        );
        setContrato(contratoResponse.data);

        // Obtener familiares autorizados
        const familiaresResponse = await axios.get(
          `${apiUrl}/api/familiares-autorizados/${clienteId}/${contratoId}`,
          {
            headers: {
              Authorization: `Bearer ${localStorage.getItem("access_token")}`,
            },
          }
        );
        setFamiliares(familiaresResponse.data);
      } catch (err) {
        setError(err.response?.data?.detail || "Error al cargar los datos");
        toast.error(err.response?.data?.detail || "Error al cargar los datos");
      }
      setLoading(false);
    };
    fetchData();
  }, [clienteId, contratoId]);

  // Handle form input changes for new familiares
  const handleFamiliarChange = (index, e) => {
    const { name, value } = e.target;
    const updatedFamiliares = newFamiliares.map((familiar, i) => {
      if (i === index) {
        const updatedFamiliar = { ...familiar, [name]: value };
        if (name === "fecha_nacimiento") {
          updatedFamiliar.edad = calcularEdad(value);
        }
        return updatedFamiliar;
      }
      return familiar;
    });
    setNewFamiliares(updatedFamiliares);
  };

  // Add new familiar form
  const agregarFamiliar = () => {
    if (familiares.length + newFamiliares.length >= 2) {
      setError("No se pueden agregar más de dos familiares autorizados");
      toast.error("No se pueden agregar más de dos familiares autorizados");
      return;
    }
    setNewFamiliares([
      ...newFamiliares,
      {
        nombre: "",
        cedula: "",
        fecha_nacimiento: "",
        edad: "",
        parentesco: "",
      },
    ]);
  };

  // Remove new familiar form
  const eliminarFamiliar = (index) => {
    const updatedFamiliares = newFamiliares.filter((_, i) => i !== index);
    setNewFamiliares(updatedFamiliares);
    setError(null);
  };

  // Delete existing familiar from backend
  const handleDelete = async (familiarId) => {
    try {
      await axios.delete(`${apiUrl}/api/familiares-autorizados/${familiarId}`, {
        headers: {
          Authorization: `Bearer ${localStorage.getItem("access_token")}`,
        },
      });
      setFamiliares(
        familiares.filter((familiar) => familiar.id !== familiarId)
      );
      toast.success("Familiar eliminado exitosamente");
    } catch (err) {
      setError(err.response?.data?.detail || "Error al eliminar el familiar");
      toast.error(
        err.response?.data?.detail || "Error al eliminar el familiar"
      );
    }
  };

  // Handle form submission
  const handleSubmit = async (e) => {
    e.preventDefault();
    if (newFamiliares.length === 0) {
      setError("Debe agregar al menos un familiar para guardar");
      toast.error("Debe agregar al menos un familiar para guardar");
      return;
    }
    for (const familiar of newFamiliares) {
      if (!familiar.nombre || !familiar.fecha_nacimiento) {
        setError(
          "Nombre y fecha de nacimiento son obligatorios para todos los familiares"
        );
        toast.error(
          "Nombre y fecha de nacimiento son obligatorios para todos los familiares"
        );
        return;
      }
    }
    try {
      const familiaresToSend = newFamiliares.map((familiar) => ({
        ...familiar,
        cliente_id: parseInt(clienteId),
        contrato_id: parseInt(contratoId),
      }));
      const response = await axios.post(
        `${apiUrl}/api/familiares-autorizados/`,
        familiaresToSend,
        {
          headers: {
            Authorization: `Bearer ${localStorage.getItem("access_token")}`,
          },
        }
      );
      setFamiliares([...familiares, ...response.data]);
      setNewFamiliares([]);
      setError(null);
      toast.success("Familiares guardados exitosamente");
    } catch (err) {
      setError(err.response?.data?.detail || "Error al guardar los familiares");
      toast.error(
        err.response?.data?.detail || "Error al guardar los familiares"
      );
    }
  };

  return (
    <Box
      component="form"
      onSubmit={handleSubmit}
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
        maxWidth: 950,
      }}
    >
      <Typography
        variant="h5"
        align="center"
        gutterBottom
        sx={{
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          marginBottom: 2,
        }}
      >
        Familiares Autorizados
      </Typography>

      {/* Datos del Cliente */}
      {cliente && (
        <Box sx={{ mb: 3 }}>
          <Typography variant="h6">Datos del Cliente</Typography>
          <Typography>Nombre: {cliente.nombre}</Typography>
          <Typography>Cédula: {cliente.cedula}</Typography>
          <Typography>Email: {cliente.email || "N/A"}</Typography>
        </Box>
      )}

      {/* Detalles del Contrato */}
      {contrato && (
        <>
          <Typography
            variant="h6"
            sx={{
              fontFamily: "Bakbak One, sans-serif",
              color: "#0A4575",
              marginBottom: 2,
            }}
          >
            Detalles del Contrato
          </Typography>
          <TableContainer component={Paper}>
            <Table size="small">
              <TableHead sx={{ backgroundColor: "#0A4575" }}>
                <TableRow>
                  <TableCell sx={{ color: "#fff" }}>Código</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Descripción</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
                  <TableCell sx={{ color: "#fff" }}>Estado</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                <TableRow>
                  <TableCell>{contrato.codigo}</TableCell>
                  <TableCell>{contrato.descripcion}</TableCell>
                  <TableCell>{contrato.tipo_servicio}</TableCell>
                  <TableCell>{contrato.estado}</TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </TableContainer>
        </>
      )}

      {/* Familiares Autorizados */}
      <Typography
        variant="h6"
        sx={{
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          marginTop: 3,
          marginBottom: 2,
        }}
      >
        Familiares Autorizados
      </Typography>
      <TableContainer component={Paper}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Nombre</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cédula</TableCell>
              <TableCell sx={{ color: "#fff" }}>Fecha de Nacimiento</TableCell>
              <TableCell sx={{ color: "#fff" }}>Edad</TableCell>
              <TableCell sx={{ color: "#fff" }}>Parentesco</TableCell>
              <TableCell sx={{ color: "#fff" }}>Acciones</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {familiares.map((familiar) => (
              <TableRow key={familiar.id}>
                <TableCell>{familiar.nombre}</TableCell>
                <TableCell>{familiar.cedula || "N/A"}</TableCell>
                <TableCell>
                  {familiar.fecha_nacimiento
                    ? new Date(familiar.fecha_nacimiento).toLocaleDateString(
                        "es-ES",
                        {
                          day: "2-digit",
                          month: "2-digit",
                          year: "numeric",
                        }
                      )
                    : "N/A"}
                </TableCell>
                <TableCell>{familiar.edad || "N/A"}</TableCell>
                <TableCell>
                  {parentescosMap[familiar.parentesco] || "N/A"}
                </TableCell>
                <TableCell>
                  <Button
                    variant="outlined"
                    color="error"
                    size="small"
                    onClick={() => handleDelete(familiar.id)}
                  >
                    Eliminar
                  </Button>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>

      {/* Botones para añadir/eliminar familiar */}
      <Grid container spacing={3} sx={{ marginTop: 2 }}>
        <Grid item xs={12} sm={6}>
          <Button
            variant="contained"
            onClick={agregarFamiliar}
            disabled={familiares.length + newFamiliares.length >= 2}
            sx={{
              fontFamily: "Abel, sans-serif",
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
              marginRight: 2,
            }}
          >
            Añadir Familiar
          </Button>
          <Button
            variant="contained"
            onClick={() => eliminarFamiliar(newFamiliares.length - 1)}
            disabled={newFamiliares.length === 0}
            sx={{
              fontFamily: "Abel, sans-serif",
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Eliminar Familiar
          </Button>
        </Grid>
      </Grid>

      {/* Formulario para nuevos familiares */}
      {newFamiliares.length > 0 && (
        <Grid container spacing={3} sx={{ marginTop: 2 }}>
          {newFamiliares.map((familiar, index) => (
            <React.Fragment key={index}>
              <Grid item xs={12} sm={3}>
                <TextField
                  fullWidth
                  label="Nombre"
                  variant="outlined"
                  size="small"
                  name="nombre"
                  value={familiar.nombre}
                  onChange={(e) => handleFamiliarChange(index, e)}
                  required
                  sx={{ boxShadow: 3 }}
                />
              </Grid>
              <Grid item xs={12} sm={2}>
                <TextField
                  fullWidth
                  label="Cédula"
                  variant="outlined"
                  size="small"
                  name="cedula"
                  value={familiar.cedula}
                  onChange={(e) => handleFamiliarChange(index, e)}
                  sx={{ boxShadow: 3 }}
                />
              </Grid>
              <Grid item xs={12} sm={3}>
                <TextField
                  fullWidth
                  label="Fecha de Nacimiento"
                  variant="outlined"
                  type="date"
                  size="small"
                  name="fecha_nacimiento"
                  value={familiar.fecha_nacimiento || ""}
                  onChange={(e) => handleFamiliarChange(index, e)}
                  required
                  sx={{ boxShadow: 3 }}
                  InputLabelProps={{ shrink: true }}
                />
              </Grid>
              <Grid item xs={12} sm={2}>
                <TextField
                  fullWidth
                  label="Edad"
                  variant="outlined"
                  size="small"
                  name="edad"
                  value={familiar.edad}
                  disabled
                  sx={{ boxShadow: 3 }}
                />
              </Grid>
              <Grid item xs={12} sm={2}>
                <FormControl
                  fullWidth
                  variant="outlined"
                  size="small"
                  sx={{ boxShadow: 3 }}
                >
                  <InputLabel id={`familiar_parentesco_${index}`}>
                    Parentesco
                  </InputLabel>
                  <Select
                    labelId={`familiar_parentesco_${index}`}
                    name="parentesco"
                    value={familiar.parentesco}
                    onChange={(e) => handleFamiliarChange(index, e)}
                    label="Parentesco"
                  >
                    <MenuItem value="" disabled>
                      Seleccione una opción
                    </MenuItem>
                    {Object.entries(parentescosMap).map(([id, texto]) => (
                      <MenuItem key={id} value={id}>
                        {texto}
                      </MenuItem>
                    ))}
                  </Select>
                </FormControl>
              </Grid>
            </React.Fragment>
          ))}
        </Grid>
      )}

      {loading && <CircularProgress sx={{ mt: 2 }} />}
      {error && (
        <Alert severity="error" sx={{ mt: 2 }}>
          {error}
        </Alert>
      )}

      {/* Botones de acción */}
      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          marginTop: 3,
        }}
      >
        <Button
          variant="contained"
          component={Link}
          to="/ventas"
          sx={{
            backgroundColor: "#0A4575",
            color: "#FFFFFF",
            "&:hover": { backgroundColor: "#083a5e" },
          }}
        >
          Volver
        </Button>
        <Button
          type="submit"
          variant="contained"
          sx={{
            backgroundColor: "#0A4575",
            color: "#FFFFFF",
            "&:hover": { backgroundColor: "#083a5e" },
          }}
          disabled={newFamiliares.length === 0}
        >
          Guardar
        </Button>
      </Box>
    </Box>
  );
};

export default FamiliaresAutorizados;
