import React, { useState, useEffect } from "react";
import {
  Grid,
  Box,
  TextField,
  Typography,
  Select,
  MenuItem,
  FormControl,
  InputLabel,
  Button,
  Tooltip,
  IconButton,
  FormControlLabel,
  Checkbox,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";
import { CrearItemServicio } from "../utils/CrearItemServicio";
import HelpOutlineIcon from "@mui/icons-material/HelpOutline";

const apiUrl = process.env.REACT_APP_API_URL;

const NichosColumbarios = ({ agregarAlPresupuesto, resetTrigger }) => {
  const [nichosData, setParcelaData] = useState(null);
  const [planes, setPlanes] = useState([]);
  const [planSeleccionado, setPlanSeleccionado] = useState(null);
  const [codigo, setCodigo] = useState("");
  const [errorCodigo, setErrorCodigo] = useState(false);
  const [inicial, setInicial] = useState(0);
  const [cuotas, setCuotas] = useState(1);
  const [montoFinanciar, setMontoFinanciar] = useState(0);
  const [precioPlan, setPrecioPlan] = useState(0);
  const [montoCuotas, setMontoCuotas] = useState(0);
  const [modalidad, setModalidad] = useState("");

  // NUEVO: estados de descuentos
  const [pagoTotalEnDivisas, setPagoTotalEnDivisas] = useState(false);
  const [pagoInicialEnDivisas, setPagoInicialEnDivisas] = useState(false);

  const disponible = parcelaData ? parcelaData.disponible || 0 : 0;
  const precioContado = planSeleccionado ? planSeleccionado.de_contado || 0 : 0;
  const puestos = planSeleccionado
    ? parseInt(planSeleccionado.puestos) || 0
    : 0;

  useEffect(() => {
    const fetchPlanes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(
          `${apiUrl}/api/planes/solicitudes?tipo_servicio=parcela`,
          { headers: { Authorization: `Bearer ${token}` } }
        );
        setPlanes(response.data);
      } catch (error) {
        console.error("Error al obtener los planes:", error);
        toast.error("Error al cargar los planes de parcelas.");
      }
    };
    fetchPlanes();
  }, []);

  const handleCodigoChange = (e) => {
    const input = e.target.value.toUpperCase();
    setCodigo(input);
    setErrorCodigo(false);
    setParcelaData(null);
  };

  const handleCodigoBlur = async () => {
    if (!codigo) {
      setErrorCodigo(true);
      toast.error("Por favor, ingrese el código de la parcela.");
      return;
    }

    try {
      const token = localStorage.getItem("access_token");
      const normalizedCode = codigo.replace(/\s/g, "").toUpperCase();
      let finalCode = normalizedCode;

      if (normalizedCode.match(/^A\d{2}-\d{2}$/)) {
        finalCode = `${normalizedCode}-01`;
      }

      if (!finalCode.match(/^A\d{2}-\d{2}-\d{2}$/)) {
        toast.error(
          "El código debe tener el formato AXX-XX-XX (ejemplo: A03-26-01)."
        );
        setErrorCodigo(true);
        setCodigo("");
        return;
      }

      const response = await axios.get(
        `${apiUrl}/api/consulta-parcela/${finalCode}`,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      const parcela = response.data;
      setParcelaData(parcela);

      if (parcela.disponible === 0) {
        toast.error(`La parcela ${finalCode} no tiene puestos disponibles.`);
        setCodigo("");
      } else {
        toast.success(
          `La parcela ${finalCode} tiene ${parcela.disponible} puesto(s) disponible(s).`
        );
      }
    } catch (error) {
      if (error.response?.status === 404) {
        toast.warning(
          `No se encontró la parcela con código ${codigo}. Puede crear una nueva entrada.`
        );
        setParcelaData({
          estado: "disponible",
          disponible: 3,
          precio_contado: 0,
          precio_financiado: 0,
        });
      } else {
        console.error("Error al consultar la parcela:", error);
        toast.error("Error al verificar el código de la parcela.");
      }
    }
  };

  useEffect(() => {
    setParcelaData(null);
    setPlanSeleccionado(null);
    setCodigo("");
    setErrorCodigo(false);
    setModalidad("");
    setPrecioPlan("");
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  }, [resetTrigger]);

  const handlePlanChange = (e) => {
    const planId = e.target.value;
    const plan = planes.find((p) => p.id === planId);
    setPlanSeleccionado(plan);
    setPrecioPlan(0);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  };

  const handleModalidadChange = (e) => {
    setModalidad(e.target.value);
    setPrecioPlan(0);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  };
  // Calcular montos
  useEffect(() => {
    if (!parcelaData || !modalidad || !planSeleccionado) {
      setPrecioPlan(0);
      setMontoFinanciar(0);
      setMontoCuotas(0);
      return;
    }

    if (modalidad === "De Contado") {
      let precio = precioContado;

      if (pagoTotalEnDivisas) {
        precio = precio - precio * 0.2; // 20% descuento
      }
      setMontoFinanciar(precio);
      setPrecioPlan(precio);
    }

    if (modalidad === "Financiado") {
      const cuotaKey = `cuota_${cuotas}`;
      let total =
        planSeleccionado[cuotaKey] || planSeleccionado.financiado || 0;
      setPrecioPlan(total > 0 ? total : 0);

      // Descuento si pago inicial en divisas
      let montoInicialValido = parseFloat(inicial) || 0;
      if (pagoInicialEnDivisas) {
        montoInicialValido = montoInicialValido * 1.2;
      }

      let saldo = 0;

      if (montoInicialValido > 0) {
        saldo = total - montoInicialValido;
      }
      setMontoFinanciar(saldo > 0 ? saldo : 0);

      if (cuotas === 1) {
        setMontoCuotas(saldo > 0 ? saldo : 0);
      } else {
        const restante = saldo / (cuotas - 1);
        setMontoCuotas(restante > 0 ? restante : 0);
      }
    }
  }, [
    parcelaData,
    modalidad,
    inicial,
    cuotas,
    planSeleccionado,
    pagoTotalEnDivisas,
    pagoInicialEnDivisas,
  ]);

  const handleAgregar = () => {
    if (!codigo || !parcelaData || !planSeleccionado || !modalidad) {
      toast.error("Por favor, complete todos los campos requeridos.");
      return;
    }

    if (disponible === 0 || puestos > disponible) {
      toast.error(
        "No se puede agregar una parcela que no esté disponible o exceda los puestos."
      );
      return;
    }

    if (modalidad === "Financiado" && inicial > montoFinanciar + inicial) {
      toast.error(
        "El monto inicial no puede exceder el precio total financiado."
      );
      return;
    }

    const item = CrearItemServicio({
      tipo_servicio: "parcelas",
      descripcion: planSeleccionado.descripcion,
      modalidadPago: modalidad,
      codigo,
      pagoInicial: modalidad === "Financiado" ? inicial : 0,
      cuotas: modalidad === "Financiado" ? cuotas : 0,
      montoPorCuotas: modalidad === "Financiado" ? montoCuotas : 0,
      precioFinanciado:
        modalidad === "De Contado" ? montoFinanciar : montoFinanciar,
      precioTotal: montoFinanciar,
      precioPlan: precioPlan,
      detalles_adicionales: { codigo, puestos: puestos, disponible },
    });

    agregarAlPresupuesto(item);
    //toast.success("Solicitud agregada al presupuesto con éxito.");
  };

  return (
    <>
      <Typography
        sx={{
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          textAlign: "center",
        }}
        variant="h5"
      >
        SOLICITUD DE CONTRATO DE PARCELAS
      </Typography>
      <Box>
        <Typography
          sx={{
            marginTop: "15px",
            marginBottom: "10px",
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
          variant="h6"
        >
          PLAN A SOLICITAR
        </Typography>
      </Box>
      <Grid container spacing={2}>
        <Grid item xs={2}>
          <Tooltip
            title="Por favor, ingrese el código de la parcela."
            open={errorCodigo}
            disableHoverListener
            placement="top"
            arrow
          >
            <TextField
              fullWidth
              label="Código Parcela"
              variant="outlined"
              size="small"
              value={codigo}
              onChange={handleCodigoChange}
              onBlur={handleCodigoBlur}
              sx={{ boxShadow: 3 }}
              error={errorCodigo}
            />
          </Tooltip>
        </Grid>

        <Grid sx={{ mt: 2 }}>
          <IconButton onClick={() => openImageInNewTab(mapaParcelas[0].url)}>
            <HelpOutlineIcon />
          </IconButton>
        </Grid>
      </Grid>{" "}
      <Grid container spacing={2} sx={{ mt: 2 }}>
        {disponible > 0 && (
          <Grid item xs={3}>
            <FormControl
              fullWidth
              size="small"
              variant="outlined"
              sx={{ boxShadow: 3 }}
            >
              <InputLabel>Seleccione el Plan</InputLabel>
              <Select
                label="Seleccione el Plan"
                value={planSeleccionado ? planSeleccionado.id : ""}
                onChange={handlePlanChange}
                autoComplete="off"
              >
                <MenuItem value="" disabled>
                  Seleccione
                </MenuItem>
                {planes
                  .filter(
                    (plan) => plan.is_active !== true && plan.is_active !== 1
                  )
                  .map((plan) => (
                    <MenuItem key={plan.id} value={plan.id}>
                      {plan.descripcion}
                    </MenuItem>
                  ))}
              </Select>
            </FormControl>
          </Grid>
        )}

        {disponible > 0 && planSeleccionado && (
          <Grid item xs={2}>
            <FormControl
              fullWidth
              size="small"
              variant="outlined"
              sx={{ boxShadow: 3 }}
            >
              <InputLabel id="modalidad-label">Modalidad</InputLabel>
              <Select
                labelId="modalidad-label"
                id="modalidad"
                label="Modalidad"
                value={modalidad}
                onChange={handleModalidadChange}
                autoComplete="off"
              >
                <MenuItem value="" disabled>
                  Seleccione
                </MenuItem>
                <MenuItem value="De Contado">De Contado</MenuItem>
                <MenuItem value="Financiado">Financiado</MenuItem>
              </Select>
            </FormControl>
          </Grid>
        )}

        {disponible > 0 && planSeleccionado && modalidad && (
          <>
            {modalidad === "De Contado" && (
              <>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio del Plan"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid sx={{ mt: 2, ml: 2 }}>
                  <FormControlLabel
                    control={
                      <Checkbox
                        checked={pagoTotalEnDivisas}
                        onChange={(e) =>
                          setPagoTotalEnDivisas(e.target.checked)
                        }
                        color="primary"
                      />
                    }
                    label="Pago en Divisas (20% Descuento)"
                  />
                </Grid>
                <Grid item xs={2}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    disabled={disponible === 0 || puestos > disponible}
                    fullWidth
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}

            {modalidad === "Financiado" && (
              <>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio del Plan"
                    variant="outlined"
                    size="small"
                    value={`$${precioPlan.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={1}>
                  <FormControl
                    fullWidth
                    size="small"
                    variant="outlined"
                    sx={{ boxShadow: 3 }}
                  >
                    <InputLabel id="cuotas-label">Cuotas</InputLabel>
                    <Select
                      labelId="cuotas-label"
                      id="cuotas"
                      label="Cuotas"
                      value={cuotas}
                      onChange={(e) => setCuotas(parseInt(e.target.value) || 1)}
                      autoComplete="off"
                    >
                      {[1, 2, 3, 4, 5, 6].map((num) => (
                        <MenuItem key={num} value={num}>
                          {num}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Pago Inicial ($)"
                    variant="outlined"
                    size="small"
                    type="number"
                    value={inicial}
                    onChange={(e) =>
                      setInicial(parseFloat(e.target.value) || 0)
                    }
                    sx={{ boxShadow: 3 }}
                    inputProps={{ min: 0 }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio Financiado"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Monto por Cuota"
                    variant="outlined"
                    size="small"
                    value={`$${montoCuotas.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={4}>
                  <FormControlLabel
                    control={
                      <Checkbox
                        checked={pagoInicialEnDivisas}
                        onChange={(e) =>
                          setPagoInicialEnDivisas(e.target.checked)
                        }
                        color="primary"
                      />
                    }
                    label="Pago Inicial en Divisas (20% Descuento)"
                  />
                </Grid>
                <Grid item xs={2}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    disabled={disponible === 0 || puestos > disponible}
                    fullWidth
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}
          </>
        )}
      </Grid>
    </>
  );
};

export default NichosColumbarios;
