import React, { useEffect, useState } from "react";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Collapse,
  IconButton,
  Paper,
  TablePagination,
  Dialog,
  DialogTitle,
  DialogContent,
  TextField,
  Grid,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Modal,
} from "@mui/material";
import ExpandMoreIcon from "@mui/icons-material/ExpandMore";
import ExpandLessIcon from "@mui/icons-material/ExpandLess";
import axios from "axios";
import { jwtDecode } from "jwt-decode";
import { toast } from "react-toastify";
import { useDollar } from "../../DollarPrice";

// Valores fijos para pagos en Bolívares
const BANCO_RECEPTOR = "Banco Banesco";
const NUMERO_CUENTA = "0134-0319-82-3191111990";
const TELEFONO_PAGO_MOVIL = "0414 4282678";
const RIF = "J-408124637";

const apiUrl = process.env.REACT_APP_API_URL;

const PagoSolicitudes = () => {
  const [solicitudes, setSolicitudes] = useState({ items: [], total: 0 });
  const [expandedContrato, setExpandedContrato] = useState(null);
  const [openModal, setOpenModal] = useState(false);
  const [selectedSolicitud, setSelectedSolicitud] = useState(null);
  const [openPagoModal, setOpenPagoModal] = useState(false);
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);

  useEffect(() => {
    const obtenerSolicitudes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        if (!token) {
          toast.error("No se encontró el token de autenticación.");
          return;
        }
        const decoded = jwtDecode(token);
        const vendedorId = decoded.sub ? Number(decoded.sub) : null;

        if (!vendedorId) {
          toast.error("No se pudo obtener el ID del vendedor.");
          return;
        }

        const config = {
          headers: { Authorization: `Bearer ${token}` },
        };

        const response = await axios.get(
          `${apiUrl}/api/reporte-pagos/contratos/${vendedorId}?page=${
            page + 1
          }&per_page=${rowsPerPage}`,
          config
        );
        setSolicitudes(response.data);
      } catch (error) {
        toast.error("Error al obtener la lista de solicitudes.");
        console.error("Error al obtener solicitudes:", error);
        setSolicitudes({ items: [], total: 0 });
      }
    };

    obtenerSolicitudes();
  }, [page, rowsPerPage]);

  const handleToggleContrato = (contratoId) => {
    setExpandedContrato(expandedContrato === contratoId ? null : contratoId);
  };

  const handleOpenModal = (solicitud) => {
    axios
      .get(
        `${apiUrl}/api/v1/contratos/detalles/${solicitud.cliente_id}/${solicitud.id}`
      )
      .then((response) => {
        setSelectedSolicitud({
          ...solicitud,
          detalles_contrato: response.data,
        });
        setOpenModal(true);
      })
      .catch((error) => {
        toast.error("Error al obtener los detalles de la solicitud.");
        console.error("Error al obtener detalles:", error);
      });
  };

  const handleCloseModal = () => {
    setOpenModal(false);
    setSelectedSolicitud(null);
  };

  const handleChangePage = (event, newPage) => {
    setPage(newPage);
  };

  const handleChangeRowsPerPage = (event) => {
    setRowsPerPage(parseInt(event.target.value, 10));
    setPage(0);
  };

  const getEstadoPagoTexto = (contrato) => {
    if (
      contrato.estado_pago === "pagado" ||
      contrato.estado_pago === "inicial_pagada"
    ) {
      return contrato.modalidad_pago === "Financiado"
        ? "INICIAL PAGADA"
        : "PAGADA";
    }
    return "PENDIENTE";
  };

  const handleOpenPagoModal = (contrato, detalle) => {
    setSelectedSolicitud({ ...contrato, detalle });
    setOpenPagoModal(true);
  };

  const handleClosePagoModal = () => {
    setOpenPagoModal(false);
    setSelectedSolicitud(null);
  };

  const handlePagoRegistrado = () => {
    const obtenerSolicitudes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const decoded = jwtDecode(token);
        const vendedorId = decoded.sub ? Number(decoded.sub) : null;

        if (!vendedorId) {
          toast.error("No se pudo obtener el ID del vendedor.");
          return;
        }

        const config = {
          headers: { Authorization: `Bearer ${token}` },
        };

        const response = await axios.get(
          `${apiUrl}/api/reporte-pagos/contratos/${vendedorId}?page=${
            page + 1
          }&per_page=${rowsPerPage}`,
          config
        );
        setSolicitudes(response.data);
      } catch (error) {
        toast.error("Error al actualizar la lista de solicitudes.");
        console.error("Error al actualizar solicitudes:", error);
      }
    };
    obtenerSolicitudes();
  };

  const ReportePagosModal = ({
    open,
    onClose,
    contrato,
    detalle,
    numeroContrato,
    tipoServicio,
    monto,
    contratoId,
    onPagoRegistrado,
  }) => {
    const montoTotal = parseFloat(monto || 0);
    const { dollarPrice } = useDollar();
    const [tipoPago, setTipoPago] = useState("USD");
    const [metodoPagoVes, setMetodoPagoVes] = useState("");
    const [pagoData, setPagoData] = useState({
      montoUsd: (detalle?.modalidadPago === "Financiado"
        ? 400
        : montoTotal
      ).toFixed(2),
      observacionesUsd: "",
      comprobanteUsd: null,
      montoVes: (detalle?.modalidadPago === "Financiado"
        ? 400 * dollarPrice
        : montoTotal * dollarPrice
      ).toFixed(2),
      bancoEmisorVes: "",
      numeroTransferenciaVes: "",
      comprobanteTransferenciaVes: null,
      referenciaPagoMovilVes: "",
      comprobantePagoMovilVes: null,
      numeroContrato: numeroContrato || contrato?.id || "",
      tipoServicio: tipoServicio || contrato?.tipo_servicio || "",
    });

    const banks = [
      { value: "", label: "Seleccione un Banco" },
      { value: "Banesco", label: "Banco Banesco Banco Universal, C.A." },
      { value: "100% Banco", label: "100% Banco Banco Universal, C.A." },
      { value: "Banca Amiga", label: "Banca Amiga, Banco Microfinanciero" },
      { value: "Bancaribe", label: "Bancaribe, C.A. Banco Universal" },
      { value: "Banco Activo", label: "Banco Activo" },
      { value: "Banco Bicentenario", label: "Banco Bicentenario" },
      { value: "Banco Caroní", label: "Banco Caroní, C.A. Banco Universal" },
      { value: "Banco de la FANB", label: "Banco de la FANB" },
      { value: "Banco de Venezuela", label: "Banco de Venezuela" },
      { value: "Banco del Tesoro", label: "Banco del Tesoro" },
      {
        value: "Banco Exterior",
        label: "Banco Exterior, C.A. Banco Universal",
      },
      {
        value: "Banco Internacional de Desarrollo",
        label: "Banco Internacional de Desarrollo",
      },
      { value: "Banco Mercantil", label: "Banco Mercantil" },
      { value: "BNC", label: "Banco Nacional de Crédito (BNC)" },
      { value: "Banco Plaza", label: "Banco Plaza, C.A. Banco Universal" },
      { value: "Banco Provincial", label: "Banco Provincial" },
      { value: "Banco Sofitasa", label: "Banco Sofitasa" },
      { value: "Bancrecer", label: "Bancrecer, S.A. Banco Microfinanciero" },
      { value: "Banplus", label: "Banplus Banco Universal, C.A." },
      { value: "Banco Fondo Común", label: "BFC Banco Fondo Común" },
      { value: "Mi Banco", label: "Mi Banco, Banco Microfinanciero, C.A." },
      {
        value: "Banco Venezolano de Credito",
        label: "Banco Venezolano de Credito",
      },
    ];

    const formatDateToMySQL = (date) => {
      const d = new Date(date);
      const year = d.getFullYear();
      const month = String(d.getMonth() + 1).padStart(2, "0");
      const day = String(d.getDate()).padStart(2, "0");
      const hours = String(d.getHours()).padStart(2, "0");
      const minutes = String(d.getMinutes()).padStart(2, "0");
      const seconds = String(d.getSeconds()).padStart(2, "0");
      return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
    };

    useEffect(() => {
      setTipoPago("USD");
      setPagoData((prev) => ({
        ...prev,
        montoUsd: (detalle?.modalidadPago === "Financiado"
          ? 400
          : montoTotal
        ).toFixed(2),
        montoVes: (detalle?.modalidadPago === "Financiado"
          ? 400 * dollarPrice
          : montoTotal * dollarPrice
        ).toFixed(2),
        observacionesUsd: "",
        comprobanteUsd: null,
        bancoEmisorVes: "",
        numeroTransferenciaVes: "",
        comprobanteTransferenciaVes: null,
        referenciaPagoMovilVes: "",
        comprobantePagoMovilVes: null,
      }));
      setMetodoPagoVes("");
    }, [montoTotal, dollarPrice, detalle?.modalidadPago, open]);

    const handleTipoPagoChange = (event) => {
      const newTipoPago = event.target.value;
      setTipoPago(newTipoPago);
      setPagoData((prev) => ({
        ...prev,
        montoUsd:
          newTipoPago === "USD" || newTipoPago === "Ambos"
            ? (detalle?.modalidadPago === "Financiado"
                ? 400
                : montoTotal
              ).toFixed(2)
            : "",
        montoVes:
          newTipoPago === "VES" || newTipoPago === "Ambos"
            ? (detalle?.modalidadPago === "Financiado"
                ? 400 * dollarPrice
                : montoTotal * dollarPrice
              ).toFixed(2)
            : "",
        observacionesUsd: "",
        comprobanteUsd: null,
        bancoEmisorVes: "",
        numeroTransferenciaVes: "",
        comprobanteTransferenciaVes: null,
        referenciaPagoMovilVes: "",
        comprobantePagoMovilVes: null,
      }));
      setMetodoPagoVes("");
    };

    const handleMetodoPagoVesChange = (event) => {
      setMetodoPagoVes(event.target.value);
      setPagoData((prev) => ({
        ...prev,
        bancoEmisorVes: "",
        numeroTransferenciaVes: "",
        comprobanteTransferenciaVes: null,
        referenciaPagoMovilVes: "",
        comprobantePagoMovilVes: null,
      }));
    };

    const handlePagoDataChange = (e) => {
      const { name, value } = e.target;
      setPagoData((prev) => {
        const updatedData = { ...prev, [name]: value };
        if (
          name === "montoUsd" &&
          (tipoPago === "USD" || tipoPago === "Ambos")
        ) {
          updatedData.montoVes = (parseFloat(value || 0) * dollarPrice).toFixed(
            2
          );
        }
        return updatedData;
      });
    };

    const handleComprobanteChange = (e, tipo) => {
      const file = e.target.files[0];
      if (file) {
        console.log(`Archivo seleccionado para ${tipo}:`, file);
        setPagoData((prev) => {
          const updatedData = { ...prev, [tipo]: file };
          console.log(`Estado actualizado para ${tipo}:`, updatedData);
          return updatedData;
        });
      } else {
        console.log(`No se seleccionó archivo para ${tipo}`);
      }
    };

    // Nuevo handleSubmit corregido y comentado paso a paso
    const handleSubmit = async (e) => {
      e.preventDefault();

      const token = localStorage.getItem("access_token");
      let vendedorId = null;

      if (token) {
        try {
          const decoded = jwtDecode(token);
          vendedorId = decoded.sub ? Number(decoded.sub) : null;
        } catch (error) {
          console.error("Error al decodificar el token:", error);
        }
      }

      if (!vendedorId) {
        toast.error("No se pudo identificar al vendedor.");
        return;
      }

      if (!contrato?.cliente_id) {
        toast.error("No se encontró el ID del cliente.");
        return;
      }

      try {
        const formData = new FormData();

        formData.append("numero_solicitud", contrato?.numero_solicitud || "");
        formData.append("cliente_id", contrato?.cliente_id);
        formData.append("vendedor_id", vendedorId);
        formData.append("tipo_servicio", pagoData.tipoServicio);
        formData.append(
          "monto_usd",
          tipoPago === "VES" ? 0 : parseFloat(pagoData.montoUsd || 0)
        );
        formData.append(
          "monto_ves",
          tipoPago === "USD" ? 0 : parseFloat(pagoData.montoVes || 0)
        );
        formData.append("monto_total", montoTotal);
        formData.append("metodo_pago", tipoPago);
        formData.append("observaciones", pagoData.observacionesUsd || "");
        formData.append(
          "referencia",
          pagoData.numeroTransferenciaVes ||
            pagoData.referenciaPagoMovilVes ||
            ""
        );
        formData.append("banco_emisor", pagoData.bancoEmisorVes || "");
        formData.append("banco_receptor", BANCO_RECEPTOR);
        formData.append(
          "estado_pago",
          detalle?.modalidadPago === "Financiado" ? "inicial_pagada" : "pagado"
        );
        formData.append("fecha_pago", formatDateToMySQL(new Date()));

        // 🧾 Agregar archivos binarios
        if (pagoData.comprobanteUsd instanceof File) {
          formData.append("comprobante_usd", pagoData.comprobanteUsd);
        }

        if (pagoData.comprobanteTransferenciaVes instanceof File) {
          formData.append(
            "comprobante_ves",
            pagoData.comprobanteTransferenciaVes
          );
        } else if (pagoData.comprobantePagoMovilVes instanceof File) {
          formData.append("comprobante_ves", pagoData.comprobantePagoMovilVes);
        }

        // ✅ Aquí se envía TODO junto
        const response = await axios.post(
          `${apiUrl}/api/reporte-pagos/guardar-pago`,
          formData,
          {
            headers: {
              Authorization: `Bearer ${token}`,
              "Content-Type": "multipart/form-data",
            },
          }
        );

        toast.success("Pago reportado con éxito");
        onPagoRegistrado();
        onClose();
      } catch (error) {
        console.error("Error al enviar FormData:", error);
        toast.error(
          `Error al registrar: ${error.response?.data?.detail || error.message}`
        );
      }
    };

    return (
      <Modal open={open} onClose={onClose}>
        <Box
          sx={{
            position: "absolute",
            top: "50%",
            left: "50%",
            transform: "translate(-50%, -50%)",
            width: "90%",
            maxWidth: "950px",
            bgcolor: "background.paper",
            boxShadow: 24,
            p: 4,
            borderRadius: 2,
            maxHeight: "90vh",
            overflowY: "auto",
          }}
        >
          <Box
            sx={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
            }}
          >
            <Typography
              sx={{ fontFamily: "Bakbak One", color: "#0A4575" }}
              variant="h5"
            >
              REPORTAR PAGO - #{pagoData.numeroContrato}
            </Typography>
            <Typography
              sx={{
                fontFamily: "Bakbak One",
                color: "#0A4575",
                fontSize: "20px",
              }}
            >
              Monto total: {montoTotal}$
            </Typography>
          </Box>
          <Grid container spacing={2}>
            <Grid item xs={12} sm={3}>
              <Typography
                sx={{
                  fontFamily: "Bakbak One",
                  color: "#0A4575",
                  fontSize: "20px",
                  mt: 1,
                }}
              >
                Servicio: {pagoData.tipoServicio}
              </Typography>
            </Grid>
            <Grid item xs={12} sm={3}>
              <Typography
                sx={{
                  fontFamily: "Bakbak One",
                  color: "#0A4575",
                  fontSize: "20px",
                  mt: 1,
                  mb: 2,
                }}
              >
                Modalidad: {detalle?.modalidadPago}
              </Typography>
            </Grid>
          </Grid>
          <form onSubmit={handleSubmit}>
            <Grid container spacing={2}>
              <Grid item xs={12} sm={3}>
                <FormControl
                  fullWidth
                  variant="outlined"
                  size="small"
                  sx={{ boxShadow: 3 }}
                >
                  <InputLabel>Tipo de Pago</InputLabel>
                  <Select
                    value={tipoPago}
                    onChange={handleTipoPagoChange}
                    label="Tipo de Pago"
                    name="tipoPago"
                  >
                    <MenuItem value="USD">USD</MenuItem>
                    <MenuItem value="VES">VES</MenuItem>
                    <MenuItem value="Ambos">AMBOS</MenuItem>
                  </Select>
                </FormControl>
              </Grid>

              {(tipoPago === "USD" || tipoPago === "Ambos") && (
                <>
                  <Grid item xs={12} sm={3}>
                    <TextField
                      fullWidth
                      label="Monto en USD"
                      variant="outlined"
                      size="small"
                      name="montoUsd"
                      type="number"
                      value={pagoData.montoUsd}
                      onChange={handlePagoDataChange}
                      required
                      sx={{ boxShadow: 3 }}
                    />
                  </Grid>
                  <Grid item xs={12} sm={6}>
                    <input
                      accept="image/*"
                      style={{ display: "none" }}
                      id="comprobante-usd"
                      type="file"
                      onChange={(e) =>
                        handleComprobanteChange(e, "comprobanteUsd")
                      }
                    />
                    <label htmlFor="comprobante-usd">
                      <Button
                        variant="contained"
                        component="span"
                        sx={{
                          backgroundColor: "#0A4575",
                          color: "#FFFFFF",
                          "&:hover": { backgroundColor: "#083a5e" },
                        }}
                      >
                        Subir Foto de los USD
                      </Button>
                    </label>
                    {pagoData.comprobanteUsd && (
                      <Typography
                        sx={{ marginTop: "10px", fontFamily: "Abel" }}
                      >
                        <b>{pagoData.comprobanteUsd.name}</b>
                      </Typography>
                    )}
                  </Grid>
                </>
              )}

              {(tipoPago === "VES" || tipoPago === "Ambos") && (
                <>
                  <Grid item xs={12} sm={3}>
                    <TextField
                      fullWidth
                      label="Monto en VES"
                      variant="outlined"
                      size="small"
                      name="montoVes"
                      type="number"
                      value={pagoData.montoVes}
                      onChange={handlePagoDataChange}
                      required
                      sx={{ boxShadow: 3 }}
                    />
                  </Grid>
                  <Grid item xs={12} sm={3}>
                    <FormControl
                      fullWidth
                      variant="outlined"
                      size="small"
                      sx={{ boxShadow: 3 }}
                    >
                      <InputLabel>Método de Pago VES</InputLabel>
                      <Select
                        value={metodoPagoVes}
                        onChange={handleMetodoPagoVesChange}
                        label="Método de Pago VES"
                        name="metodoPagoVes"
                      >
                        <MenuItem value="">Seleccione un método</MenuItem>
                        <MenuItem value="transferencia">
                          Transferencia Bancaria
                        </MenuItem>
                        <MenuItem value="pagoMovil">Pago Móvil</MenuItem>
                      </Select>
                    </FormControl>
                  </Grid>

                  {metodoPagoVes === "transferencia" && (
                    <>
                      <Grid item xs={12}>
                        <Grid container spacing={2}>
                          <Grid item xs={12} sm={3}>
                            <FormControl
                              fullWidth
                              variant="outlined"
                              size="small"
                              sx={{ boxShadow: 3 }}
                            >
                              <InputLabel>Banco Emisor</InputLabel>
                              <Select
                                name="bancoEmisorVes"
                                value={pagoData.bancoEmisorVes}
                                onChange={handlePagoDataChange}
                                label="Banco Emisor"
                                required
                              >
                                {banks.map((bank) => (
                                  <MenuItem key={bank.value} value={bank.value}>
                                    {bank.label}
                                  </MenuItem>
                                ))}
                              </Select>
                            </FormControl>
                          </Grid>
                          <Grid item xs={12} sm={3}>
                            <TextField
                              fullWidth
                              label="Número de Transferencia"
                              variant="outlined"
                              size="small"
                              name="numeroTransferenciaVes"
                              value={pagoData.numeroTransferenciaVes}
                              onChange={handlePagoDataChange}
                              required
                              sx={{ boxShadow: 3 }}
                            />
                          </Grid>
                          <Grid item xs={12} sm={3}>
                            <input
                              accept="image/*,application/pdf"
                              style={{ display: "none" }}
                              id="comprobante-transferencia-ves"
                              type="file"
                              onChange={(e) =>
                                handleComprobanteChange(
                                  e,
                                  "comprobanteTransferenciaVes"
                                )
                              }
                            />
                            <label htmlFor="comprobante-transferencia-ves">
                              <Button
                                variant="contained"
                                component="span"
                                sx={{
                                  backgroundColor: "#0A4575",
                                  color: "#FFFFFF",
                                  "&:hover": { backgroundColor: "#083a5e" },
                                }}
                              >
                                Subir Comprobante
                              </Button>
                            </label>
                            {pagoData.comprobanteTransferenciaVes && (
                              <Typography
                                sx={{ marginTop: "10px", fontFamily: "Abel" }}
                              >
                                <b>
                                  {pagoData.comprobanteTransferenciaVes.name}
                                </b>
                              </Typography>
                            )}
                          </Grid>
                        </Grid>
                      </Grid>
                    </>
                  )}
                </>
              )}
            </Grid>
            {metodoPagoVes === "pagoMovil" && (
              <>
                <Grid container spacing={2} sx={{ marginTop: 0 }}>
                  <Grid item xs={12} sm={3}>
                    <FormControl
                      fullWidth
                      variant="outlined"
                      size="small"
                      sx={{ boxShadow: 3 }}
                    >
                      <InputLabel>Banco Emisor</InputLabel>
                      <Select
                        name="bancoEmisorVes"
                        value={pagoData.bancoEmisorVes}
                        onChange={handlePagoDataChange}
                        label="Banco Emisor"
                        required
                      >
                        {banks.map((bank) => (
                          <MenuItem key={bank.value} value={bank.value}>
                            {bank.label}
                          </MenuItem>
                        ))}
                      </Select>
                    </FormControl>
                  </Grid>
                  <Grid item xs={12} sm={3}>
                    <TextField
                      fullWidth
                      label="Referencia"
                      variant="outlined"
                      size="small"
                      name="referenciaPagoMovilVes"
                      value={pagoData.referenciaPagoMovilVes}
                      onChange={handlePagoDataChange}
                      required
                      sx={{ boxShadow: 3 }}
                    />
                  </Grid>
                  <Grid item xs={12} sm={3}>
                    <input
                      accept="image/*,application/pdf"
                      style={{ display: "none" }}
                      id="comprobante-pago-movil-ves"
                      type="file"
                      onChange={(e) =>
                        handleComprobanteChange(e, "comprobantePagoMovilVes")
                      }
                    />
                    <label htmlFor="comprobante-pago-movil-ves">
                      <Button
                        variant="contained"
                        component="span"
                        sx={{
                          backgroundColor: "#0A4575",
                          color: "#FFFFFF",
                          "&:hover": { backgroundColor: "#083a5e" },
                        }}
                      >
                        Subir Comprobante
                      </Button>
                    </label>
                    {pagoData.comprobantePagoMovilVes && (
                      <Typography
                        sx={{ marginTop: "10px", fontFamily: "Abel" }}
                      >
                        <b>{pagoData.comprobantePagoMovilVes.name}</b>
                      </Typography>
                    )}
                  </Grid>
                </Grid>
              </>
            )}
            <Grid container spacing={2} sx={{ marginTop: 1 }}>
              <Grid item xs={12} sm={6}>
                <TextField
                  fullWidth
                  label="Observaciones"
                  variant="outlined"
                  size="small"
                  name="observacionesUsd"
                  value={pagoData.observacionesUsd}
                  onChange={handlePagoDataChange}
                  multiline
                  rows={3}
                  sx={{ boxShadow: 3 }}
                />
              </Grid>
              {metodoPagoVes === "pagoMovil" && (
                <Grid item xs={12} sm={6}>
                  <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                    Banco Receptor: {BANCO_RECEPTOR}
                  </Typography>
                  <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                    Teléfono: {TELEFONO_PAGO_MOVIL}
                  </Typography>
                  <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                    RIF: {RIF}
                  </Typography>
                </Grid>
              )}
              {metodoPagoVes === "transferencia" && (
                <Grid item xs={12} sm={6}>
                  <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                    Banco Receptor: {BANCO_RECEPTOR}
                  </Typography>
                  <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                    Número de Cuenta: {NUMERO_CUENTA}
                  </Typography>
                  <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                    RIF: {RIF}
                  </Typography>
                </Grid>
              )}
            </Grid>
            <Box
              sx={{ display: "flex", justifyContent: "flex-end", marginTop: 3 }}
            >
              <Button variant="outlined" onClick={onClose} sx={{ mr: 2 }}>
                Cancelar
              </Button>
              <Button
                type="submit"
                variant="contained"
                sx={{
                  backgroundColor: "#0A4575",
                  color: "#FFFFFF",
                  "&:hover": { backgroundColor: "#083a5e" },
                }}
              >
                Reportar Pago
              </Button>
            </Box>
          </form>
        </Box>
      </Modal>
    );
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        variant="h6"
        align="center"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        REPORTE DE PAGOS
      </Typography>
      <TableContainer component={Paper} sx={{ boxShadow: 3, width: "100%" }}>
        <Table>
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Nº Solicitud</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
              <TableCell sx={{ color: "#fff" }}>Estado</TableCell>
              <TableCell sx={{ color: "#fff" }}>Monto Total</TableCell>
              <TableCell sx={{ color: "#FFFFFF" }}>Estado de Pago</TableCell>
              <TableCell sx={{ color: "#fff" }}>Acciones</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {solicitudes.items.length > 0 ? (
              solicitudes.items.flatMap((cliente) =>
                cliente.contratos.map((contrato) => (
                  <React.Fragment key={contrato.id}>
                    <TableRow>
                      <TableCell>{contrato.numero_solicitud}</TableCell>
                      <TableCell>{cliente.nombre}</TableCell>
                      <TableCell>{contrato.estado}</TableCell>
                      <TableCell>${contrato.monto_total || 0}</TableCell>
                      <TableCell>
                        <Button
                          variant="contained"
                          sx={{
                            backgroundColor:
                              contrato.estado_pago === "pagado" ||
                              contrato.estado_pago === "inicial_pagada"
                                ? "#4CAF50"
                                : "#F44336",
                            color: "#FFFFFF",
                            "&:hover": {
                              backgroundColor:
                                contrato.estado_pago === "pagado" ||
                                contrato.estado_pago === "inicial_pagada"
                                  ? "#45A049"
                                  : "#D32F2F",
                            },
                            pointerEvents: "none",
                          }}
                        >
                          {getEstadoPagoTexto(contrato)}
                        </Button>
                      </TableCell>
                      <TableCell>
                        <IconButton
                          onClick={() => handleToggleContrato(contrato.id)}
                        >
                          {expandedContrato === contrato.id ? (
                            <ExpandLessIcon />
                          ) : (
                            <ExpandMoreIcon />
                          )}
                        </IconButton>
                      </TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell colSpan={6} sx={{ padding: 0, width: "100%" }}>
                        <Collapse
                          in={expandedContrato === contrato.id}
                          timeout="auto"
                          unmountOnExit
                        >
                          <Box sx={{ padding: 2 }}>
                            <Typography
                              variant="h6"
                              gutterBottom
                              sx={{
                                fontFamily: "Bakbak One, sans-serif",
                                color: "#0A4575",
                              }}
                            >
                              DETALLES DE SOLICITUD
                            </Typography>
                            <TableContainer
                              component={Paper}
                              sx={{
                                boxShadow: 3,
                                width: "100%",
                                margin: 0,
                                padding: 0,
                              }}
                            >
                              <Table>
                                <TableHead sx={{ backgroundColor: "#0A4575" }}>
                                  <TableRow>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Tipo Servicio
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Descripción
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Modalidad Pago
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Pago de Inicial
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Precio Total
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Estado Pago
                                    </TableCell>
                                    <TableCell sx={{ color: "#fff" }}>
                                      Acción
                                    </TableCell>
                                  </TableRow>
                                </TableHead>
                                <TableBody>
                                  {contrato.detalles.map((detalle) => (
                                    <TableRow key={detalle.id}>
                                      <TableCell>
                                        {detalle.tipo_contrato}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.descripcion}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.modalidadPago}
                                      </TableCell>
                                      <TableCell>
                                        ${detalle.pagoInicial}
                                      </TableCell>
                                      <TableCell>
                                        ${detalle.precioTotal}
                                      </TableCell>
                                      <TableCell>
                                        {detalle.estado_pago || "Pendiente"}
                                      </TableCell>
                                      <TableCell>
                                        <Button
                                          variant="contained"
                                          onClick={() =>
                                            handleOpenPagoModal(
                                              contrato,
                                              detalle
                                            )
                                          }
                                          sx={{
                                            backgroundColor: "#0A4575",
                                            color: "#FFFFFF",
                                            "&:hover": {
                                              backgroundColor: "#083a5e",
                                            },
                                          }}
                                        >
                                          Reportar Pagos
                                        </Button>
                                      </TableCell>
                                    </TableRow>
                                  ))}
                                </TableBody>
                              </Table>
                            </TableContainer>
                          </Box>
                        </Collapse>
                      </TableCell>
                    </TableRow>
                  </React.Fragment>
                ))
              )
            ) : (
              <TableRow>
                <TableCell colSpan={6} align="center">
                  No hay solicitudes disponibles.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
        <TablePagination
          rowsPerPageOptions={[5, 10, 25]}
          component="div"
          count={solicitudes.total}
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={handleChangePage}
          onRowsPerPageChange={handleChangeRowsPerPage}
          labelRowsPerPage="Filas por página:"
          labelDisplayedRows={({ from, to, count }) =>
            `${from}-${to} de ${count}`
          }
        />
      </TableContainer>

      <Dialog open={openModal} onClose={handleCloseModal}>
        <DialogTitle>
          Detalles de la Solicitud{" "}
          {selectedSolicitud?.numero_solicitud || "Sin número"}
        </DialogTitle>
        <DialogContent>
          <Table>
            <TableHead>
              <TableRow>
                <TableCell>Tipo Servicio</TableCell>
                <TableCell>Descripción</TableCell>
                <TableCell>Modalidad Pago</TableCell>
                <TableCell>Inicial</TableCell>
                <TableCell>Precio Total</TableCell>
                <TableCell>Estado Pago</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {selectedSolicitud?.detalles_contrato &&
              Array.isArray(selectedSolicitud.detalles_contrato) &&
              selectedSolicitud.detalles_contrato.length > 0 ? (
                selectedSolicitud.detalles_contrato.map((detalle) => (
                  <TableRow key={detalle.id}>
                    <TableCell>{detalle.tipo_contrato || "N/A"}</TableCell>
                    <TableCell>
                      {detalle.descripcion || "Sin descripción"}
                    </TableCell>
                    <TableCell>{detalle.modalidadPago || "N/A"}</TableCell>
                    <TableCell>{detalle.pagoInicial || "N/A"}</TableCell>
                    <TableCell>${detalle.precioTotal || 0}</TableCell>
                    <TableCell>{detalle.estado_pago || "Pendiente"}</TableCell>
                  </TableRow>
                ))
              ) : (
                <TableRow>
                  <TableCell colSpan={6} align="center">
                    No hay detalles disponibles.
                  </TableCell>
                </TableRow>
              )}
            </TableBody>
          </Table>
        </DialogContent>
      </Dialog>

      <ReportePagosModal
        open={openPagoModal}
        onClose={handleClosePagoModal}
        contrato={selectedSolicitud}
        detalle={selectedSolicitud?.detalle}
        numeroContrato={selectedSolicitud?.numero_solicitud}
        tipoServicio={selectedSolicitud?.detalle?.tipo_contrato}
        monto={
          selectedSolicitud?.detalle?.modalidadPago === "Financiado"
            ? selectedSolicitud?.detalle?.precioTotal
            : selectedSolicitud?.detalle?.precioTotal
        }
        contratoId={selectedSolicitud?.id}
        onPagoRegistrado={handlePagoRegistrado}
      />
    </Box>
  );
};

export default PagoSolicitudes;
