import React, { useState, useEffect } from "react";
import {
  Grid,
  Box,
  TextField,
  Typography,
  Select,
  MenuItem,
  FormControl,
  InputLabel,
  Button,
  Tooltip,
  IconButton,
  FormControlLabel,
  Checkbox,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";
import { CrearItemServicio } from "../utils/CrearItemServicio";
import HelpOutlineIcon from "@mui/icons-material/HelpOutline";

const apiUrl = process.env.REACT_APP_API_URL;

const Parcelas = ({ agregarAlPresupuesto, resetTrigger }) => {
  const [parcelaData, setParcelaData] = useState(null);
  const [planes, setPlanes] = useState([]);
  const [planSeleccionado, setPlanSeleccionado] = useState(null);
  const [codigo, setCodigo] = useState("");
  const [errorCodigo, setErrorCodigo] = useState(false);
  const [inicial, setInicial] = useState(0);
  const [cuotas, setCuotas] = useState(1);
  const [montoFinanciar, setMontoFinanciar] = useState(0);
  const [precioPlan, setPrecioPlan] = useState(0);
  const [montoCuotas, setMontoCuotas] = useState(0);
  const [modalidad, setModalidad] = useState("");
  const [metodoPago, setMetodoPago] = useState("");
  const [descuento, setDescuento] = useState(0);
  const [precioDescuento, setPrecioDescuento] = useState(0);
  const [contadoChecked, setContadoChecked] = useState(false);
  const [financiadoChecked, setFinanciadoChecked] = useState(false);

  const precioContado = planSeleccionado ? planSeleccionado.de_contado || 0 : 0;

  const disponible = parcelaData ? parcelaData.disponible || 0 : 0;
  const puestos = planSeleccionado
    ? parseInt(planSeleccionado.puestos) || 0
    : 0;

  const mapaParcelas = [
    { nombre: "Mapa Parcela", url: `${apiUrl}/uploads/mapas/parcelas.jpg` },
  ];

  const openImageInNewTab = (url) => {
    const img = new Image();
    img.src = url;
    img.onload = () => window.open(url, "_blank");
    img.onerror = () =>
      toast.error(`No se pudo cargar la imagen: ${url.split("/").pop()}`);
  };

  useEffect(() => {
    const fetchPlanes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(
          `${apiUrl}/api/planes/solicitudes?tipo_servicio=parcela`,
          { headers: { Authorization: `Bearer ${token}` } }
        );
        setPlanes(response.data);
      } catch (error) {
        console.error("Error al obtener los planes:", error);
        toast.error("Error al cargar los planes de parcelas.");
      }
    };
    fetchPlanes();
  }, []);

  const handleCodigoChange = (e) => {
    const input = e.target.value.toUpperCase();
    setCodigo(input);
    setErrorCodigo(false);
    setParcelaData(null);
  };

  const handleCodigoBlur = async () => {
    if (!codigo) {
      setErrorCodigo(true);
      toast.error("Por favor, ingrese el código de la parcela.");
      return;
    }

    try {
      const token = localStorage.getItem("access_token");
      const normalizedCode = codigo.replace(/\s/g, "").toUpperCase();
      let finalCode = normalizedCode;

      if (normalizedCode.match(/^A\d{2}-\d{2}$/)) {
        finalCode = `${normalizedCode}-01`;
      }

      if (!finalCode.match(/^A\d{2}-\d{2}-\d{2}$/)) {
        toast.error(
          "El código debe tener el formato AXX-XX-XX (ejemplo: A03-26-01)."
        );
        setErrorCodigo(true);
        setCodigo("");
        return;
      }

      const response = await axios.get(
        `${apiUrl}/api/consulta-parcela/${finalCode}`,
        { headers: { Authorization: `Bearer ${token}` } }
      );
      const parcela = response.data;
      setParcelaData(parcela);

      if (parcela.disponible === 0) {
        toast.error(`La parcela ${finalCode} no tiene puestos disponibles.`);
        setCodigo("");
      } else {
        toast.success(
          `La parcela ${finalCode} tiene ${parcela.disponible} puesto(s) disponible(s).`
        );
      }
    } catch (error) {
      if (error.response?.status === 404) {
        toast.warning(
          `No se encontró la parcela con código ${codigo}. Puede crear una nueva entrada.`
        );
        setParcelaData({
          estado: "disponible",
          disponible: 3,
          precio_contado: 0,
          precio_financiado: 0,
        });
      } else {
        console.error("Error al consultar la parcela:", error);
        toast.error("Error al verificar el código de la parcela.");
      }
    }
  };

  useEffect(() => {
    setPlanSeleccionado(null);
    setModalidad("");
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setMetodoPago("");
    setDescuento(0);
    setPrecioDescuento(0);
    setPrecioPlan("");
  }, [resetTrigger]);

  const handlePlanChange = (e) => {
    const planId = e.target.value;
    const plan = planes.find((p) => p.id === planId);
    setPlanSeleccionado(plan);
    setPrecioPlan(0);
    setModalidad("");
    setDescuento(0);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
  };

  const handleModalidadChange = (e) => {
    setModalidad(e.target.value);
    setDescuento(0);
    setPrecioPlan(0);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
  };

  // Calcular montos
  useEffect(() => {
    if (!planSeleccionado || !modalidad) {
      setPrecioPlan(0);
      setDescuento(0);
      setMontoFinanciar(0);
      setMontoCuotas(0);
      return;
    }

    if (modalidad === "De Contado") {
      // 1️⃣ Elegir precio base según el checkbox
      let precio = precioContado;

      // 2️⃣ Aplicar 20% de descuento si está marcado "Pago en Divisas"
      let metodo = "VES";
      let dscto = 0;
      let precioConDescuento = 0;

      if (contadoChecked) {
        metodo = "USD";
        dscto = precio * 0.2;

        precioConDescuento = precio - dscto; // precio con descuento aplicado
      }

      // 3️⃣ Actualizar estados
      setMetodoPago(metodo);
      setDescuento(dscto);
      setPrecioDescuento(precioConDescuento);
      setPrecioPlan(precio);
      setMontoFinanciar(precio);
      setMontoCuotas(0);
    }

    // Función que devuelve todos los estados listos para setear
    function calcularEstadosFinanciado({
      totalPlan,
      inicial,
      cuotas,
      financiadoChecked,
    }) {
      const montoInicialValido = parseFloat(inicial) || 0;
      const financiadoActivo = financiadoChecked && montoInicialValido > 0;

      const descuento = financiadoActivo ? montoInicialValido * 0.2 : 0;
      const metodo = financiadoActivo ? "USD" : "VES";
      const precioConDescuento = financiadoActivo ? totalPlan - descuento : 0;

      const saldoFinal = financiadoActivo
        ? totalPlan - descuento - montoInicialValido
        : montoInicialValido > 0
        ? totalPlan - montoInicialValido
        : totalPlan;

      const montoPorCuota =
        cuotas === 1 ? saldoFinal : saldoFinal / (cuotas - 1);

      return {
        montoCuotas: montoPorCuota > 0 ? montoPorCuota : 0,
        metodoPago: metodo,
        descuento,
        precioDescuento: precioConDescuento,
        precioPlan: totalPlan > 0 ? totalPlan : 0,
        montoFinanciar: saldoFinal > 0 ? saldoFinal : 0,
      };
    }

    // -------------------------------
    // Bloque principal ultra limpio
    if (modalidad === "Financiado") {
      const cuotaKey = `cuota_${cuotas}`;
      const totalPlan =
        planSeleccionado[cuotaKey] || planSeleccionado.financiado || 0;

      const estados = calcularEstadosFinanciado({
        totalPlan,
        inicial,
        cuotas,
        financiadoChecked,
      });

      // Aplicar todos los estados
      setMontoCuotas(estados.montoCuotas);
      setMetodoPago(estados.metodoPago);
      setDescuento(estados.descuento);
      setPrecioDescuento(estados.precioDescuento);
      setPrecioPlan(estados.precioPlan);
      setMontoFinanciar(estados.montoFinanciar);
    }
  }, [
    modalidad,
    inicial,
    cuotas,
    planSeleccionado,
    contadoChecked, // 🔹 importante para recalcular al marcar/desmarcar
    financiadoChecked,
  ]);

  const handleAgregar = () => {
    if (!codigo || !parcelaData || !planSeleccionado || !modalidad) {
      toast.error("Por favor, complete todos los campos requeridos.");
      return;
    }

    if (disponible === 0 || puestos > disponible) {
      toast.error(
        "No se puede agregar una parcela que no esté disponible o exceda los puestos."
      );
      return;
    }

    if (modalidad === "Financiado" && inicial > montoFinanciar + inicial) {
      toast.error(
        "El monto inicial no puede exceder el precio total financiado."
      );
      return;
    }
    const item = CrearItemServicio({
      tipo_servicio: "parcelas",
      descripcion: planSeleccionado.descripcion,
      modalidadPago: modalidad,
      codigo,
      metodoPago: metodoPago,
      pagoInicial: modalidad === "Financiado" ? inicial : 0,
      cuotas: modalidad === "Financiado" ? cuotas : 0,
      montoPorCuotas: modalidad === "Financiado" ? montoCuotas : 0,
      precioFinanciado: modalidad === "Financiado" ? montoFinanciar : 0,
      descuento: metodoPago === "USD" ? descuento : 0,
      precioConDescuento: metodoPago === "USD" ? precioDescuento : 0,
      precioTotal: precioPlan,
      precioPlan: precioPlan,
      detalles_adicionales: {
        tipo_servicio: "parcelas",
        descripcion: planSeleccionado.descripcion,
        modalidadPago: modalidad,
      },
      items_adicionales: {
        metodoPago: metodoPago,
        descuento: metodoPago === "USD" ? descuento : 0,
        precioConDescuento: metodoPago === "USD" ? precioDescuento : 0,
      },
    });

    agregarAlPresupuesto(item);
    //toast.success("Solicitud agregada al presupuesto con éxito.");
  };

  return (
    <>
      <Typography
        sx={{
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          textAlign: "center",
        }}
        variant="h5"
      >
        SOLICITUD DE CONTRATO DE PARCELAS
      </Typography>
      <Typography
        sx={{
          marginTop: "15px",
          marginBottom: "10px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
      >
        PLAN A SOLICITAR
      </Typography>
      <Grid container spacing={2}>
        <Grid item xs={2}>
          <Tooltip
            title="Por favor, ingrese el código de la parcela."
            open={errorCodigo}
            disableHoverListener
            placement="top"
            arrow
          >
            <TextField
              fullWidth
              label="Código Parcela"
              variant="outlined"
              size="small"
              value={codigo}
              onChange={handleCodigoChange}
              onBlur={handleCodigoBlur}
              sx={{ boxShadow: 3 }}
              error={errorCodigo}
            />
          </Tooltip>
        </Grid>

        <Grid sx={{ mt: 2 }}>
          <IconButton onClick={() => openImageInNewTab(mapaParcelas[0].url)}>
            <HelpOutlineIcon />
          </IconButton>
        </Grid>
      </Grid>{" "}
      <Grid container spacing={2} sx={{ mt: 2 }}>
        {disponible > 0 && (
          <Grid item xs={3}>
            <FormControl
              fullWidth
              size="small"
              variant="outlined"
              sx={{ boxShadow: 3 }}
            >
              <InputLabel>Seleccione el Plan</InputLabel>
              <Select
                label="Seleccione el Plan"
                value={planSeleccionado ? planSeleccionado.id : ""}
                onChange={handlePlanChange}
                autoComplete="off"
              >
                <MenuItem value="" disabled>
                  Seleccione
                </MenuItem>
                {planes
                  .filter(
                    (plan) => plan.is_active !== true && plan.is_active !== 1
                  )
                  .map((plan) => (
                    <MenuItem key={plan.id} value={plan.id}>
                      {plan.descripcion}
                    </MenuItem>
                  ))}
              </Select>
            </FormControl>
          </Grid>
        )}

        {disponible > 0 && planSeleccionado && (
          <Grid item xs={2}>
            <FormControl
              fullWidth
              size="small"
              variant="outlined"
              sx={{ boxShadow: 3 }}
            >
              <InputLabel id="modalidad-label">Modalidad</InputLabel>
              <Select
                labelId="modalidad-label"
                id="modalidad"
                label="Modalidad"
                value={modalidad}
                onChange={handleModalidadChange}
                autoComplete="off"
              >
                <MenuItem value="" disabled>
                  Seleccione
                </MenuItem>
                <MenuItem value="De Contado">De Contado</MenuItem>
                <MenuItem value="Financiado">Financiado</MenuItem>
              </Select>
            </FormControl>
          </Grid>
        )}

        {disponible > 0 && planSeleccionado && modalidad && (
          <>
            {modalidad === "De Contado" && (
              <>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio del Plan"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    disabled={disponible === 0 || puestos > disponible}
                    fullWidth
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}

            {modalidad === "Financiado" && (
              <>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio del Plan"
                    variant="outlined"
                    size="small"
                    value={`$${precioPlan.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <FormControl
                    fullWidth
                    size="small"
                    variant="outlined"
                    sx={{ boxShadow: 3 }}
                  >
                    <InputLabel id="cuotas-label">Cuotas</InputLabel>
                    <Select
                      labelId="cuotas-label"
                      id="cuotas"
                      label="Cuotas"
                      value={cuotas}
                      onChange={(e) => setCuotas(parseInt(e.target.value) || 1)}
                      autoComplete="off"
                    >
                      {[1, 2, 3, 4, 5, 6].map((num) => (
                        <MenuItem key={num} value={num}>
                          {num}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Pago Inicial ($)"
                    variant="outlined"
                    size="small"
                    type="number"
                    value={inicial}
                    onChange={(e) =>
                      setInicial(parseFloat(e.target.value) || 0)
                    }
                    sx={{ boxShadow: 3 }}
                    inputProps={{ min: 0 }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Precio Financiado"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <TextField
                    fullWidth
                    label="Monto por Cuota"
                    variant="outlined"
                    size="small"
                    value={`$${montoCuotas.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={2}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    disabled={disponible === 0 || puestos > disponible}
                    fullWidth
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}
          </>
        )}
      </Grid>
    </>
  );
};

export default Parcelas;
