import React, { useState, useEffect } from "react";
import {
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Button,
  Grid,
  Typography,
  Box,
  TextField,
  Checkbox,
  Table,
  TableContainer,
  TableBody,
  TableCell,
  TableHead,
  TableRow,
  Paper,
} from "@mui/material";
import { ArrowBack } from "@mui/icons-material";
import axios from "axios";
import { toast } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";
import { Link } from "react-router-dom";
import { jwtDecode } from "jwt-decode";

const Previsivos = () => {
  const [planes, setPlanes] = useState([]);
  const [items, setItems] = useState([]);
  const [itemsSeleccionados, setItemsSeleccionados] = useState([]);
  const [planSeleccionado, setPlanSeleccionado] = useState(null);
  const [inicial, setInicial] = useState("");
  const [montoFinanciar, setMontoFinanciar] = useState(0);
  const [cuotas, setCuotas] = useState(1);
  const [montoCuotas, setMontoCuotas] = useState(0);
  const [totalContrato, setTotalContrato] = useState(0);
  const [clienteSeleccionado, setClienteSeleccionado] = useState(null);
  const apiUrl = process.env.REACT_APP_API_URL;

  const [cliente, setCliente] = useState({
    nombre: "",
    cedula: "",
    fecha_nacimiento: "",
    edad: "",
    telefono: "",
    celular: "",
    email: "",
    direccion: "",
    nombre_documento: "",
    documento_identidad: null,
    cantidad_parientes: "",
    familiares: [],
  });

  const [modalidadPrevisivo, setModalidadPrevisivo] = useState({
    descripcion: "",
    fecha_actual: "",
    fecha_vencimiento: "",
    monto_adicional: 0,
    total: 0,
    tipo_servicio: "previsivo",
    tipo_contrato: "", // Corregido: valor inicial vacío para coincidir con Select
    modalidadPago: "Financiado",
    inicial: 0,
    monto_financiar: 0,
    cuotas: 1,
    montoPorCuotas: 0,
  });

  const parentescosMap = {
    1: "Padre/Madre",
    2: "Hijo(a)",
    3: "Esposo(a)/Conyugue",
    4: "Hermano(a)",
    5: "Abuelo(a)",
    6: "Tío(a)",
    7: "Primo(a)",
    8: "Otro(a)",
  };

  useEffect(() => {
    const fetchPlanes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(
          `${apiUrl}/api/planes?tipo_servicio=previsivo`,
          {
            headers: { Authorization: `Bearer ${token}` },
          }
        );
        const filteredPlanes = response.data.filter(
          (plan) => plan.financiado > 0
        );
        setPlanes(filteredPlanes);
      } catch (error) {
        console.error("Error al obtener los planes de cremación:", error);
        toast.error("Error al cargar los planes de cremación.");
      }
    };
    fetchPlanes();
  }, []);

  const handlePlanChange = (e) => {
    const planId = parseInt(e.target.value, 10);
    const plan = planes.find((p) => p.id === planId);
    setPlanSeleccionado(plan);
    setModalidadPrevisivo((prev) => ({
      ...prev,
      descripcion: plan ? plan.descripcion : "",
      modalidadPago: "Financiado",
    }));
    setInicial("");
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
  };

  const handleModalidadPrevisivoChange = (e) => {
    const { name, value } = e.target;
    setModalidadPrevisivo((prev) => ({
      ...prev,
      [name]: value,
    }));
  };

  const handleSeleccionarItem = (index) => {
    if (itemsSeleccionados.includes(index)) {
      setItemsSeleccionados(itemsSeleccionados.filter((i) => i !== index));
    } else {
      setItemsSeleccionados([...itemsSeleccionados, index]);
    }
  };

  const handleClienteChange = (e) => {
    const { name, value } = e.target;
    let updatedFields = { [name]: value };

    if (name === "fecha_nacimiento") {
      const edadCalculada = calcularEdad(value);
      updatedFields.edad = edadCalculada.toString();
    }

    setCliente((prev) => ({
      ...prev,
      ...updatedFields,
    }));
  };

  const handleFamiliarChange = (index, e) => {
    const { name, value } = e.target;
    const updatedFamiliares = cliente.familiares.map((familiar, i) => {
      if (i === index) {
        const updatedFamiliar = { ...familiar, [name]: value };

        if (name === "fecha_nacimiento") {
          const edadCalculada = calcularEdad(value);
          updatedFamiliar.edad = edadCalculada.toString();
        }

        if (
          updatedFamiliar.edad &&
          updatedFamiliar.parentesco &&
          planSeleccionado
        ) {
          const parentesco_id = parseInt(updatedFamiliar.parentesco, 10);
          const edad = parseInt(updatedFamiliar.edad, 10);
          const plan_id = parseInt(planSeleccionado.id, 10);

          axios
            .get(`${apiUrl}/api/previsivos/calcular_monto`, {
              params: { plan: plan_id, parentesco: parentesco_id, edad },
            })
            .then((response) => {
              updatedFamiliar.precio = parseFloat(response.data.monto);
              const nuevosFamiliares = [...cliente.familiares];
              nuevosFamiliares[i] = updatedFamiliar;
              setCliente((prev) => ({ ...prev, familiares: nuevosFamiliares }));
              calcularMontoAdicional(nuevosFamiliares);
            })
            .catch((error) => {
              console.error(error);
            });
        }

        return updatedFamiliar;
      }
      return familiar;
    });

    setCliente((prev) => ({
      ...prev,
      familiares: updatedFamiliares,
    }));
  };

  const calcularEdad = (fechaNacimiento) => {
    const hoy = new Date();
    const fecha_nacimiento = new Date(fechaNacimiento);
    let edad = hoy.getFullYear() - fecha_nacimiento.getFullYear();
    const mes = hoy.getMonth() - fecha_nacimiento.getMonth();

    if (mes < 0 || (mes === 0 && hoy.getDate() < fecha_nacimiento.getDate())) {
      edad--;
    }
    return edad;
  };

  const agregarFamiliar = () => {
    setCliente((prev) => ({
      ...prev,
      familiares: [
        ...prev.familiares,
        {
          nombre: "",
          cedula: "",
          fecha_nacimiento: "",
          edad: "",
          parentesco: "",
          precio: "",
          id_familiar: null,
        },
      ],
    }));
  };

  const eliminarFamiliar = (index) => {
    const updatedFamiliares = cliente.familiares.filter((_, i) => i !== index);
    setCliente((prev) => ({
      ...prev,
      familiares: updatedFamiliares,
    }));
    calcularMontoAdicional(updatedFamiliares);
  };

  const obtenerFechaFormateada = () => {
    const fecha = new Date();
    const options = {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
      hour12: true,
    };
    return fecha.toLocaleString("es-ES", options);
  };

  const obtenerFechaVencimiento = () => {
    const fecha = new Date();
    fecha.setFullYear(fecha.getFullYear() + 1);
    return fecha.toISOString().split("T")[0];
  };

  const calcularMontoAdicional = (familiares) => {
    let totalBase = 0;
    familiares.forEach((familiar) => {
      if (familiar.precio) {
        totalBase += parseFloat(familiar.precio);
      }
    });
    setModalidadPrevisivo((prev) => ({
      ...prev,
      monto_adicional: totalBase,
    }));
  };

  useEffect(() => {
    setModalidadPrevisivo((prev) => ({
      ...prev,
      fecha_actual: obtenerFechaFormateada(),
      fecha_vencimiento: obtenerFechaVencimiento(),
    }));
  }, []);

  useEffect(() => {
    const planPrecioBase = planSeleccionado
      ? parseFloat(planSeleccionado.financiado)
      : 0;
    const montoAdicionalBase = modalidadPrevisivo.monto_adicional || 0;
    const totalEnUSD = planPrecioBase + montoAdicionalBase;

    setModalidadPrevisivo((prev) => ({
      ...prev,
      total: totalEnUSD,
    }));
  }, [planSeleccionado, modalidadPrevisivo.monto_adicional]);

  useEffect(() => {
    if (modalidadPrevisivo.modalidadPago === "Financiado") {
      const inicialValido = parseFloat(inicial) || 0;
      const montoFinanciarCalculado = modalidadPrevisivo.total - inicialValido;
      setMontoFinanciar(
        montoFinanciarCalculado > 0 ? montoFinanciarCalculado : 0
      );
      setModalidadPrevisivo((prev) => ({
        ...prev,
        inicial: inicialValido,
        monto_financiar:
          montoFinanciarCalculado > 0 ? montoFinanciarCalculado : 0,
      }));
    } else {
      setMontoFinanciar(0);
      setModalidadPrevisivo((prev) => ({
        ...prev,
        inicial: 0,
        monto_financiar: 0,
      }));
    }
  }, [modalidadPrevisivo.total, inicial, modalidadPrevisivo.modalidadPago]);

  useEffect(() => {
    if (modalidadPrevisivo.modalidadPago === "Financiado" && cuotas > 0) {
      const montoCuotaCalculado = montoFinanciar / cuotas;
      setMontoCuotas(montoCuotaCalculado > 0 ? montoCuotaCalculado : 0);
      setModalidadPrevisivo((prev) => ({
        ...prev,
        montoPorCuotas: montoCuotaCalculado > 0 ? montoCuotaCalculado : 0,
      }));
    } else {
      setMontoCuotas(0);
      setModalidadPrevisivo((prev) => ({
        ...prev,
        montoPorCuotas: 0,
      }));
    }
  }, [montoFinanciar, cuotas, modalidadPrevisivo.modalidadPago]);

  useEffect(() => {
    if (modalidadPrevisivo.modalidadPago !== "Financiado") {
      setInicial("");
      setMontoFinanciar(0);
      setCuotas(1);
      setMontoCuotas(0);
      setModalidadPrevisivo((prev) => ({
        ...prev,
        inicial: 0,
        monto_financiar: 0,
        cuotas: 1,
        montoPorCuotas: 0,
      }));
    }
  }, [modalidadPrevisivo.modalidadPago]);

  useEffect(() => {
    const total = items.reduce(
      (sum, item) => sum + (parseFloat(item.precioTotal) || 0),
      0
    );
    setTotalContrato(total);
  }, [items]);

  const handleAgregar = () => {
    if (!planSeleccionado) return;

    const precioBaseUSD = planSeleccionado.financiado;
    const montoAdicionalBaseUSD = modalidadPrevisivo.monto_adicional;
    const precioTotalUSD = precioBaseUSD + montoAdicionalBaseUSD;

    const pagoInicial = parseFloat(inicial) || 0;
    const montoCuotaFinal = montoCuotas;

    const item = {
      codigo: "PRV-000001",
      descripcion: planSeleccionado.descripcion,
      precio: precioBaseUSD,
      modalidadPago: modalidadPrevisivo.modalidadPago,
      pagoInicial: pagoInicial,
      cuotas: cuotas,
      montoPorCuotas: montoCuotaFinal,
      precioFinanciado: modalidadPrevisivo.total,
      precioTotal: precioTotalUSD,
    };
    setItems([...items, item]);
  };

  const handleEliminarItems = () => {
    const nuevosItems = items.filter(
      (_, index) => !itemsSeleccionados.includes(index)
    );
    setItems(nuevosItems);
    setItemsSeleccionados([]);
  };

  const formatPrecio = (precio) => {
    return `$${parseFloat(precio || 0).toFixed(2)}`;
  };

  const buscarCliente = async (cedula) => {
    if (!cedula.trim()) return;
    try {
      const token = localStorage.getItem("access_token");
      const config = { headers: { Authorization: `Bearer ${token}` } };
      const response = await axios.get(
        `${apiUrl}/api/v1/clientes/${cedula}`,
        config
      );
      const clienteEncontrado = response.data;

      toast.success("El cliente ya está registrado. Se asociará el contrato.");
      setClienteSeleccionado(clienteEncontrado);
      setCliente({
        ...clienteEncontrado,
        fecha_nacimiento: clienteEncontrado.fecha_nacimiento
          ? new Date(clienteEncontrado.fecha_nacimiento)
              .toISOString()
              .split("T")[0]
          : "",
        documento_identidad: null,
        nombre_documento: "",
        familiares: clienteEncontrado.familiares || [], // Asegurar que familiares sea un arreglo
      });
    } catch (error) {
      if (error.response?.status === 404) {
        toast.info("El cliente no está registrado. Complete los datos.");
        setClienteSeleccionado(null);
        setCliente((prev) => ({ ...prev, cedula, familiares: [] }));
      } else {
        toast.error("Error al buscar cliente.");
        setClienteSeleccionado(null);
      }
    }
  };

  const handleFileChange = (event) => {
    const file = event.target.files[0];
    if (file) {
      setCliente((prev) => ({
        ...prev,
        documento_identidad: file,
        nombre_documento: file.name,
      }));
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    const token = localStorage.getItem("access_token");
    const decoded = jwtDecode(token);
    const vendedorId = decoded.sub;
    const supervisorId = decoded.supervisor_id;

    try {
      let clienteId = clienteSeleccionado ? clienteSeleccionado.id : null;

      if (!clienteId) {
        const clienteData = {
          vendedor_id: vendedorId,
          nombre: cliente.nombre,
          cedula: cliente.cedula,
          fecha_nacimiento: cliente.fecha_nacimiento || null,
          edad: cliente.edad || 0,
          telefono: cliente.telefono || "",
          celular: cliente.celular || "",
          email: cliente.email || "",
          direccion: cliente.direccion || "",
          nombre_documento: cliente.nombre_documento || "",
          documento_identidad: cliente.documento_identidad || null,
        };

        const formData = new FormData();
        Object.keys(clienteData).forEach((key) => {
          if (
            key === "documento_identidad" &&
            clienteData[key] instanceof File
          ) {
            formData.append(key, clienteData[key], clienteData[key].name);
          } else {
            formData.append(key, clienteData[key]);
          }
        });

        const config = {
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "multipart/form-data",
          },
        };

        const response = await axios.post(
          `${apiUrl}/api/clientes`,
          formData,
          config
        );
        clienteId = response.data.cliente_id;
        setClienteSeleccionado({ id: clienteId });
      }

      const contratoData = {
        tipo_servicio: "previsivo", // Enviar "previsivo" al backend
        descripcion: modalidadPrevisivo.descripcion,
        modalidadPago: modalidadPrevisivo.modalidadPago,
        codigo: "PRE-000001",
        cliente_id: clienteId,
        vendedor_id: vendedorId,
        supervisor_id: supervisorId,
        fecha_actual: modalidadPrevisivo.fecha_actual,
        precioPlan: modalidadPrevisivo.total,
        estado: "solicitud",
        fecha_vencimiento: new Date(
          modalidadPrevisivo.fecha_vencimiento
        ).toISOString(),
        detalles: items.map((item) => ({
          codigo: "PRE-000001",
          precio: item.precio,
          modalidadPago: item.modalidadPago,
          pagoInicial: parseFloat(item.pagoInicial) || 0,
          cuotas: parseInt(item.cuotas) || 0,
          montoPorCuotas: parseFloat(item.montoPorCuotas) || 0,
          precioTotal: item.precioTotal,
          precioFinanciado: modalidadPrevisivo.total,
          precioPlan: modalidadPrevisivo.total,
          detalles_adicionales: {
            tipo_servicio: "previsivo",
            tipo_contrato: modalidadPrevisivo.tipo_contrato,
          },
        })),
        familiares: (cliente.familiares || []).map((familiar) => ({
          nombre: familiar.nombre,
          cedula: familiar.cedula,
          fecha_nacimiento: familiar.fecha_nacimiento,
          edad: parseInt(familiar.edad),
          parentesco: parentescosMap[familiar.parentesco] || "",
        })),
      };

      const configContrato = {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
      };

      const contratoResponse = await axios.post(
        `${apiUrl}/api/previsivos`,
        [contratoData],
        configContrato
      );
      console.log("Contrato creado:", contratoResponse.data);
      toast.success("Cliente y contrato creados exitosamente");

      setTimeout(() => {
        setCliente({
          nombre: "",
          cedula: "",
          fecha_nacimiento: "",
          edad: "",
          telefono: "",
          celular: "",
          email: "",
          direccion: "",
          documento_identidad: null,
          nombre_documento: "",
          cantidad_parientes: "",
          familiares: [], // Asegurar que familiares sea un arreglo
        });
        setModalidadPrevisivo({
          descripcion: "",
          fecha_actual: "",
          fecha_vencimiento: "",
          monto_adicional: 0,
          total: 0,
          tipo_servicio: "previsivo",
          tipo_contrato: "", // Reiniciar como vacío
          modalidadPago: "Financiado",
          inicial: 0,
          monto_financiar: 0,
          cuotas: 1,
          montoPorCuotas: 0,
        });
        setItems([]);
        setClienteSeleccionado(null);
        setPlanSeleccionado(null);
        setInicial("");
        setCuotas(1);
        setMontoFinanciar(0);
        setMontoCuotas(0);
      }, 3000);
    } catch (error) {
      toast.error(`Error: ${error.response?.data?.detail || error.message}`);
    }
  };

  return (
    <Box
      component="form"
      onSubmit={handleSubmit}
      sx={{
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        sx={{
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          textAlign: "center",
        }}
        variant="h5"
      >
        SOLICITUD DE CONTRATO PREVISIVO
      </Typography>
      <Typography
        sx={{
          marginTop: "15px",
          marginBottom: "10px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
      >
        PLAN A SOLICITAR
      </Typography>
      <Grid container spacing={2}>
        <Grid item xs={12} sm={3}>
          <FormControl
            fullWidth
            size="small"
            variant="outlined"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel>Plan a Contratar</InputLabel>
            <Select
              label="Plan a Contratar"
              value={planSeleccionado ? planSeleccionado.id : ""}
              onChange={handlePlanChange}
            >
              <MenuItem value="" disabled>
                Selecciona un Plan
              </MenuItem>
              {planes.map((plan) => (
                <MenuItem key={plan.id} value={plan.id}>
                  {plan.descripcion} - ${plan.financiado.toFixed(2)}
                </MenuItem>
              ))}
            </Select>
          </FormControl>
        </Grid>
        {planSeleccionado && (
          <>
            <Grid item xs={12} sm={3}>
              <TextField
                fullWidth
                label="Precio del Plan (USD)"
                variant="outlined"
                size="small"
                value={formatPrecio(planSeleccionado.financiado)}
                InputProps={{ readOnly: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={12} sm={3}>
              <TextField
                fullWidth
                label="Fecha y Hora Actual"
                variant="outlined"
                size="small"
                value={modalidadPrevisivo.fecha_actual || ""}
                InputProps={{ readOnly: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={12} sm={3}>
              <TextField
                fullWidth
                label="Fecha Vencimiento"
                variant="outlined"
                size="small"
                type="date"
                value={modalidadPrevisivo.fecha_vencimiento}
                InputProps={{ readOnly: true }}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
          </>
        )}
      </Grid>

      <Typography
        sx={{
          marginTop: "15px",
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
      >
        DATOS DEL CLIENTE
      </Typography>
      <Grid container spacing={3}>
        <Grid item xs={12} sm={4}>
          <TextField
            fullWidth
            label="Nombre Solicitante"
            variant="outlined"
            size="small"
            name="nombre"
            value={cliente.nombre}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Cédula o Rif"
            variant="outlined"
            size="small"
            name="cedula"
            value={cliente.cedula}
            onChange={handleClienteChange}
            onBlur={(e) => buscarCliente(e.target.value)}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Fecha de Nacimiento"
            variant="outlined"
            type="date"
            size="small"
            name="fecha_nacimiento"
            value={cliente.fecha_nacimiento || ""}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
            InputLabelProps={{ shrink: true }}
          />
        </Grid>
        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Edad"
            variant="outlined"
            size="small"
            name="edad"
            value={cliente.edad}
            required
            sx={{ boxShadow: 3 }}
            InputProps={{ readOnly: true }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Teléfono"
            variant="outlined"
            size="small"
            name="telefono"
            value={cliente.telefono}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Celular"
            variant="outlined"
            size="small"
            name="celular"
            value={cliente.celular}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={6}>
          <TextField
            fullWidth
            label="Email"
            variant="outlined"
            size="small"
            name="email"
            value={cliente.email}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={6}>
          <TextField
            fullWidth
            label="Dirección"
            variant="outlined"
            size="small"
            name="direccion"
            value={cliente.direccion}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={4}>
          <input
            accept="image/*"
            id="file-upload"
            type="file"
            style={{ height: "1px", width: "1px", position: "absolute" }}
            onChange={handleFileChange}
          />
          <label htmlFor="file-upload">
            <Button
              variant="contained"
              component="span"
              sx={{
                boxShadow: 3,
                backgroundColor: "#0A4575",
                color: "#FFFFFF",
                "&:hover": { backgroundColor: "#083a5e" },
              }}
            >
              Subir Documento de Identidad
            </Button>
          </label>
        </Grid>
        <Grid item xs={12} sm={2}>
          {cliente.nombre_documento && (
            <Typography sx={{ marginTop: "10px", fontFamily: "Abel" }}>
              <b>{cliente.nombre_documento}</b>
            </Typography>
          )}
        </Grid>
      </Grid>

      <Typography
        sx={{
          marginTop: "10px",
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
      >
        ASOCIAR PARIENTES
      </Typography>
      <Grid container spacing={3} alignItems="center">
        <Grid item xs={12} sm={3}>
          <FormControl
            fullWidth
            variant="outlined"
            size="small"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel id="cantidad_parientes-label">
              ¿Desea Asociar algún Pariente?
            </InputLabel>
            <Select
              labelId="cantidad_parientes-label"
              name="cantidad_parientes"
              value={cliente.cantidad_parientes || ""}
              onChange={handleClienteChange}
              label="¿Desea Asociar algún Pariente?"
            >
              <MenuItem value="" disabled>
                Seleccione una opción
              </MenuItem>
              <MenuItem value="0">No</MenuItem>
              <MenuItem value="1">Sí</MenuItem>
            </Select>
          </FormControl>
        </Grid>
        {cliente.cantidad_parientes === "1" && (
          <Grid item xs={12} sm={6}>
            <Button
              sx={{
                fontFamily: "Abel, sans-serif",
                backgroundColor: "#0A4575",
                color: "#FFFFFF",
                marginRight: 2,
              }}
              variant="contained"
              onClick={agregarFamiliar}
            >
              Añadir Familiar
            </Button>
            <Button
              sx={{
                fontFamily: "Abel, sans-serif",
                backgroundColor: "#0A4575",
                color: "#FFFFFF",
              }}
              variant="contained"
              onClick={() => eliminarFamiliar(cliente.familiares.length - 1)}
              disabled={cliente.familiares.length === 0}
            >
              Eliminar Familiar
            </Button>
          </Grid>
        )}
      </Grid>

      {cliente.cantidad_parientes === "1" && (
        <Grid container spacing={3} sx={{ marginTop: 2 }}>
          {(cliente.familiares || []).map((familiar, index) => (
            <React.Fragment key={index}>
              <Grid item xs={12} sm={3}>
                <TextField
                  fullWidth
                  label="Nombre"
                  variant="outlined"
                  size="small"
                  name="nombre"
                  value={familiar.nombre}
                  onChange={(e) => handleFamiliarChange(index, e)}
                  required
                  sx={{ boxShadow: 3 }}
                />
              </Grid>
              <Grid item xs={12} sm={2}>
                <TextField
                  fullWidth
                  label="Cédula"
                  variant="outlined"
                  size="small"
                  name="cedula"
                  value={familiar.cedula}
                  onChange={(e) => handleFamiliarChange(index, e)}
                  required
                  sx={{ boxShadow: 3 }}
                />
              </Grid>
              <Grid item xs={12} sm={3}>
                <TextField
                  fullWidth
                  label="Fecha de Nacimiento"
                  variant="outlined"
                  type="date"
                  size="small"
                  name="fecha_nacimiento"
                  value={familiar.fecha_nacimiento || ""}
                  onChange={(e) => handleFamiliarChange(index, e)}
                  required
                  sx={{ boxShadow: 3 }}
                  InputLabelProps={{ shrink: true }}
                />
              </Grid>
              <Grid item xs={12} sm={2}>
                <TextField
                  fullWidth
                  label="Edad"
                  variant="outlined"
                  size="small"
                  name="edad"
                  value={familiar.edad}
                  onChange={(e) => handleFamiliarChange(index, e)}
                  required
                  sx={{ boxShadow: 3 }}
                  InputProps={{ readOnly: true }}
                />
              </Grid>
              <Grid item xs={12} sm={2}>
                <FormControl
                  fullWidth
                  variant="outlined"
                  size="small"
                  sx={{ boxShadow: 3 }}
                >
                  <InputLabel id={`familiar_parentesco_${index}`}>
                    Parentesco
                  </InputLabel>
                  <Select
                    labelId={`familiar_parentesco_${index}`}
                    name="parentesco"
                    value={familiar.parentesco}
                    onChange={(e) => handleFamiliarChange(index, e)}
                    label="Parentesco"
                  >
                    <MenuItem value="" disabled>
                      Seleccione una opción
                    </MenuItem>
                    {Object.entries(parentescosMap).map(([id, texto]) => (
                      <MenuItem key={id} value={id}>
                        {texto}
                      </MenuItem>
                    ))}
                  </Select>
                </FormControl>
              </Grid>
            </React.Fragment>
          ))}
        </Grid>
      )}

      <Box>
        <Typography
          sx={{
            marginTop: "20px",
            marginBottom: "10px",
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
          variant="h6"
        >
          MODALIDAD - LAPSOS DE CONTRATACIÓN
        </Typography>
      </Box>
      <Grid container spacing={2}>
        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Monto Adicional ($)"
            variant="outlined"
            size="small"
            value={modalidadPrevisivo.monto_adicional.toFixed(2)}
            InputProps={{ readOnly: true }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Total ($)"
            variant="outlined"
            size="small"
            value={modalidadPrevisivo.total.toFixed(2)}
            InputProps={{ readOnly: true }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={2}>
          <FormControl
            fullWidth
            variant="outlined"
            size="small"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel id="tipo_contrato">Tipo de Contrato</InputLabel>
            <Select
              labelId="tipo_contrato"
              name="tipo_contrato"
              value={modalidadPrevisivo.tipo_contrato}
              onChange={handleModalidadPrevisivoChange}
              label="Tipo de Contrato"
            >
              <MenuItem value="" disabled>
                Seleccione una opción
              </MenuItem>
              <MenuItem value="Nuevo Contrato">Nuevo Contrato</MenuItem>
              <MenuItem value="Renovacion">Renovación de Contrato</MenuItem>
            </Select>
          </FormControl>
        </Grid>
        <Grid item xs={2}>
          <FormControl
            fullWidth
            size="small"
            variant="outlined"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel id="modalidad-label">Modalidad de Pago</InputLabel>
            <Select
              labelId="modalidad-label"
              id="modalidad"
              label="Modalidad de Pago"
              value={modalidadPrevisivo.modalidadPago}
              onChange={handleModalidadPrevisivoChange}
              autoComplete="off"
            >
              <MenuItem value="" disabled>
                Seleccione
              </MenuItem>
              <MenuItem value="Financiado">Financiado</MenuItem>
            </Select>
          </FormControl>
        </Grid>

        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Inicial ($)"
            variant="outlined"
            size="small"
            type="number"
            value={inicial}
            onChange={(e) => setInicial(e.target.value)}
            inputProps={{ step: "0.01", min: "0" }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Monto a Financiar ($"
            variant="outlined"
            size="small"
            value={montoFinanciar.toFixed(2)}
            InputProps={{ readOnly: true }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Cuotas"
            variant="outlined"
            size="small"
            type="number"
            value={cuotas}
            onChange={(e) => {
              const value = parseInt(e.target.value) || 1;
              if (value >= 1 && value <= 24) setCuotas(value);
            }}
            inputProps={{ min: 1, max: 4 }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Monto por Cuotas ($)"
            variant="outlined"
            size="small"
            value={montoCuotas.toFixed(2)}
            InputProps={{ readOnly: true }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <Button
            variant="contained"
            color="primary"
            onClick={handleAgregar}
            disabled={!planSeleccionado}
            sx={{
              backgroundColor: "#0A4575",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Agregar al Contrato
          </Button>
        </Grid>
      </Grid>

      <Typography
        sx={{
          textAlign: "center",
          marginTop: "25px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
        gutterBottom
      >
        RESUMEN DEL CONTRATO
      </Typography>
      <TableContainer component={Paper}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}>
            <TableRow>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Código
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Descripción
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Modalidad de Pago
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Inicial
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Cuotas
              </TableCell>
              <TableCell
                sx={{ color: "#FFFFFF", fontSize: "16px", width: "110px" }}
              >
                Monto por Cuotas
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Total
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Seleccionar
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {items.map((item, index) => (
              <TableRow key={index}>
                <TableCell>{item.codigo}</TableCell>
                <TableCell>{item.descripcion}</TableCell>
                <TableCell>{item.modalidadPago}</TableCell>
                <TableCell>{formatPrecio(item.pagoInicial)}</TableCell>
                <TableCell>{item.cuotas}</TableCell>
                <TableCell>{formatPrecio(item.montoPorCuotas)}</TableCell>
                <TableCell>{formatPrecio(item.precioTotal)}</TableCell>
                <TableCell>
                  <Checkbox
                    checked={itemsSeleccionados.includes(index)}
                    onChange={() => handleSeleccionarItem(index)}
                    sx={{
                      color: "#0A4575",
                      "&.Mui-checked": { color: "#0A4575" },
                    }}
                  />
                </TableCell>
              </TableRow>
            ))}
            <TableRow>
              <TableCell colSpan={6} align="right">
                <Typography
                  sx={{
                    color: "#0A4575",
                    fontWeight: "bold",
                    fontSize: "16px",
                  }}
                >
                  Total:
                </Typography>
              </TableCell>
              <TableCell align="right">
                <Typography sx={{ fontWeight: "bold", color: "#000000" }}>
                  {formatPrecio(totalContrato)}
                </Typography>
              </TableCell>
              <TableCell />
            </TableRow>
          </TableBody>
        </Table>
      </TableContainer>

      {/* Nueva tabla para mostrar familiares */}
      {cliente.cantidad_parientes === "1" &&
        (cliente.familiares || []).length > 0 && (
          <>
            <Typography
              sx={{
                textAlign: "center",
                marginTop: "25px",
                fontFamily: "Bakbak One, sans-serif",
                color: "#0A4575",
              }}
              variant="h6"
              gutterBottom
            >
              FAMILIARES ASOCIADOS
            </Typography>
            <TableContainer component={Paper} sx={{ marginTop: 2 }}>
              <Table size="small">
                <TableHead
                  sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}
                >
                  <TableRow>
                    <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                      Nombre
                    </TableCell>
                    <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                      Cédula
                    </TableCell>
                    <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                      Fecha de Nacimiento
                    </TableCell>
                    <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                      Edad
                    </TableCell>
                    <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                      Parentesco
                    </TableCell>
                    <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                      Precio (USD)
                    </TableCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {(cliente.familiares || []).map((familiar, index) => (
                    <TableRow key={index}>
                      <TableCell>{familiar.nombre}</TableCell>
                      <TableCell>{familiar.cedula}</TableCell>
                      <TableCell>
                        {familiar.fecha_nacimiento
                          ? new Date(
                              familiar.fecha_nacimiento
                            ).toLocaleDateString("es-ES", {
                              day: "2-digit",
                              month: "2-digit",
                              year: "numeric",
                            })
                          : ""}
                      </TableCell>
                      <TableCell>{familiar.edad}</TableCell>
                      <TableCell>
                        {parentescosMap[familiar.parentesco] ||
                          "No especificado"}
                      </TableCell>
                      <TableCell>{formatPrecio(familiar.precio)}</TableCell>
                    </TableRow>
                  ))}
                  <TableRow>
                    <TableCell colSpan={5} align="right">
                      <Typography
                        sx={{
                          color: "#0A4575",
                          fontWeight: "bold",
                          fontSize: "16px",
                        }}
                      >
                        Total Adicional:
                      </Typography>
                    </TableCell>
                    <TableCell align="right">
                      <Typography sx={{ fontWeight: "bold", color: "#000000" }}>
                        {formatPrecio(modalidadPrevisivo.monto_adicional)}
                      </Typography>
                    </TableCell>
                  </TableRow>
                </TableBody>
              </Table>
            </TableContainer>
          </>
        )}

      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          marginTop: 3,
        }}
      >
        <Button
          variant="contained"
          component={Link}
          to="/ventas"
          startIcon={<ArrowBack />}
          sx={{
            backgroundColor: "#0A4575",
            color: "#FFFFFF",
            "&:hover": { backgroundColor: "#083a5e" },
          }}
        >
          Volver
        </Button>
        <Box>
          <Button
            variant="outlined"
            color="error"
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF !important",
              borderColor: "#0A4575",
              "&:hover": { backgroundColor: "#083a5e !important" },
              marginRight: 2,
            }}
            onClick={handleEliminarItems}
            disabled={itemsSeleccionados.length === 0}
          >
            Eliminar Ítems
          </Button>
          <Button
            type="submit"
            variant="contained"
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Guardar Solicitud
          </Button>
        </Box>
      </Box>
    </Box>
  );
};

export default Previsivos;
