import React, { useState, useEffect } from "react";
import {
  Grid,
  Box,
  TextField,
  Typography,
  Select,
  MenuItem,
  FormControl,
  InputLabel,
  Button,
  FormControlLabel,
  Checkbox,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";
import { CrearItemServicio } from "../utils/CrearItemServicio";

const apiUrl = process.env.REACT_APP_API_URL;

const Cremacion = ({ agregarAlPresupuesto, resetTrigger }) => {
  const [planes, setPlanes] = useState([]);
  const [planSeleccionado, setPlanSeleccionado] = useState(null);
  const [inicial, setInicial] = useState(0);
  const [cuotas, setCuotas] = useState(1);
  const [montoFinanciar, setMontoFinanciar] = useState(0);
  const [montoCuotas, setMontoCuotas] = useState(0);
  const [modalidad, setModalidad] = useState("");
  const [pagoTotalEnDivisas, setPagoTotalEnDivisas] = useState(false);
  const [pagoInicialEnDivisas, setPagoInicialEnDivisas] = useState(false);
  const [contadoChecked, setContadoChecked] = useState(false);

  const precioContado_bcv = planSeleccionado
    ? planSeleccionado.de_contado_bcv || 0
    : 0;
  const precioContado = planSeleccionado ? planSeleccionado.de_contado || 0 : 0;

  useEffect(() => {
    const fetchPlanes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(
          `${apiUrl}/api/planes/solicitudes?tipo_servicio=inhumacion`,
          { headers: { Authorization: `Bearer ${token}` } }
        );
        setPlanes(response.data);
      } catch (error) {
        console.error("Error al obtener los planes:", error);
        toast.error("Error al cargar los planes de Inhumación de Parcelas.");
      }
    };
    fetchPlanes();
  }, []);

  useEffect(() => {
    setPlanSeleccionado(null);
    setModalidad("");
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  }, [resetTrigger]);

  const handlePlanChange = (e) => {
    const planId = e.target.value;
    const plan = planes.find((p) => p.id === planId);
    setPlanSeleccionado(plan);
    setModalidad("");
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  };

  const handleModalidadChange = (e) => {
    setModalidad(e.target.value);
    setInicial(0);
    setCuotas(1);
    setMontoFinanciar(0);
    setMontoCuotas(0);
    setPagoTotalEnDivisas(false);
    setPagoInicialEnDivisas(false);
  };

  // Calcular montos
  useEffect(() => {
    if (!planSeleccionado || !modalidad) {
      setMontoFinanciar(0);
      setMontoCuotas(0);
      return;
    }

    if (modalidad === "De Contado") {
      // 1️⃣ Elegir precio base según el checkbox
      let precio = contadoChecked ? precioContado : precioContado_bcv;

      // 2️⃣ Aplicar 20% de descuento si está marcado "Pago en Divisas"
      if (contadoChecked) {
        precio = precio * 0.8; // 20% descuento
      }

      setMontoFinanciar(precio);
      setMontoCuotas(0);
    }

    if (modalidad === "Financiado") {
      const cuotaKey = `cuota_${cuotas}`;
      let total =
        planSeleccionado[cuotaKey] || planSeleccionado.financiado || 0;

      let montoInicialValido = parseFloat(inicial) || 0;
      if (pagoInicialEnDivisas) {
        montoInicialValido = montoInicialValido * 1.2; // 20% descuento en inicial
      }

      const saldo = total - montoInicialValido;
      setMontoFinanciar(saldo > 0 ? saldo : 0);

      const divisiones = Math.max(cuotas - 1, 1);
      setMontoCuotas(saldo > 0 ? saldo / divisiones : 0);
    }
  }, [
    modalidad,
    inicial,
    cuotas,
    planSeleccionado,
    pagoTotalEnDivisas,
    pagoInicialEnDivisas,
    contadoChecked, // 🔹 importante para recalcular al marcar/desmarcar
  ]);

  const handleAgregar = () => {
    if (!planSeleccionado || !modalidad) {
      toast.error("Por favor, seleccione un plan y una modalidad.");
      return;
    }

    if (modalidad === "Financiado" && inicial >= montoFinanciar + inicial) {
      toast.error(
        "El monto inicial no puede exceder el precio total financiado."
      );
      return;
    }

    const item = CrearItemServicio({
      codigo: "S000001",
      tipo_servicio: "cremacion",
      tipo_contrato: "cremacion",
      descripcion: planSeleccionado.descripcion,
      modalidadPago: modalidad,
      precioTotal: modalidad === "De Contado" ? montoFinanciar : montoFinanciar,
      pagoInicial: modalidad === "Financiado" ? inicial : 0,
      cuotas: modalidad === "Financiado" ? cuotas : 0,
      montoPorCuota: modalidad === "Financiado" ? montoCuotas : 0,
      detalles_adicionales: { puestos: planSeleccionado.puestos || 0 },
    });

    agregarAlPresupuesto(item);
    //toast.success("Solicitud de honra de exequias agregada al presupuesto.");
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        sx={{
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          textAlign: "center",
        }}
        variant="h5"
      >
        SOLICITUD DE CONTRATO INHUMACIÓN DE PARCELAS
      </Typography>

      <Box>
        <Typography
          sx={{
            marginTop: "15px",
            marginBottom: "10px",
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
          variant="h6"
        >
          PLAN A SOLICITAR
        </Typography>
      </Box>

      <Grid container spacing={2}>
        <Grid item xs={12} sm={3}>
          <FormControl
            fullWidth
            size="small"
            variant="outlined"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel id="plan-label">Plan</InputLabel>
            <Select
              labelId="plan-label"
              id="plan"
              label="Plan"
              value={planSeleccionado ? planSeleccionado.id : ""}
              onChange={handlePlanChange}
              autoComplete="off"
            >
              <MenuItem value="" disabled>
                Seleccione
              </MenuItem>
              {planes
                .filter(
                  (plan) => plan.is_active !== true && plan.is_active !== 1
                )
                .map((plan) => (
                  <MenuItem key={plan.id} value={plan.id}>
                    {plan.descripcion}
                  </MenuItem>
                ))}
            </Select>
          </FormControl>
        </Grid>

        {planSeleccionado && (
          <Grid item xs={12} sm={2}>
            <FormControl
              fullWidth
              size="small"
              variant="outlined"
              sx={{ boxShadow: 3 }}
            >
              <InputLabel id="modalidad-label">Modalidad</InputLabel>
              <Select
                labelId="modalidad-label"
                id="modalidad"
                label="Modalidad"
                value={modalidad}
                onChange={handleModalidadChange}
                autoComplete="off"
              >
                <MenuItem value="" disabled>
                  Seleccione
                </MenuItem>
                <MenuItem value="De Contado">De Contado</MenuItem>
                <MenuItem value="Financiado">Financiado</MenuItem>
              </Select>
            </FormControl>
          </Grid>
        )}

        {planSeleccionado && modalidad && (
          <>
            {modalidad === "De Contado" && (
              <>
                <Grid item xs={12} sm={2}>
                  <TextField
                    fullWidth
                    label="Precio al Contado"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <FormControlLabel
                    control={
                      <Checkbox
                        checked={contadoChecked}
                        onChange={(e) => setContadoChecked(e.target.checked)}
                        color="primary"
                      />
                    }
                    label="Pago en Divisas (20% Descuento)"
                  />
                </Grid>
                <Grid item xs={12} sm={2}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    fullWidth
                    sx={{
                      backgroundColor: "#0A4575",
                      "&:hover": { backgroundColor: "#083a5e" },
                    }}
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}

            {modalidad === "Financiado" && (
              <>
                <Grid item xs={12} sm={2}>
                  <TextField
                    fullWidth
                    label="Monto a Financiar"
                    variant="outlined"
                    size="small"
                    value={`$${montoFinanciar.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={12} sm={2}>
                  <TextField
                    fullWidth
                    label="Monto Inicial ($)"
                    variant="outlined"
                    size="small"
                    type="number"
                    value={inicial}
                    onChange={(e) =>
                      setInicial(parseFloat(e.target.value) || 0)
                    }
                    sx={{ boxShadow: 3 }}
                    inputProps={{ min: 0, step: "0.01" }}
                  />
                </Grid>
                <Grid item xs={12} sm={2}>
                  <FormControl
                    fullWidth
                    size="small"
                    variant="outlined"
                    sx={{ boxShadow: 3 }}
                  >
                    <InputLabel id="cuotas-label">Cuotas</InputLabel>
                    <Select
                      labelId="cuotas-label"
                      id="cuotas"
                      label="Cuotas"
                      value={cuotas}
                      onChange={(e) => setCuotas(parseInt(e.target.value) || 1)}
                      autoComplete="off"
                    >
                      {[1, 2, 3, 4, 5, 6].map((num) => (
                        <MenuItem key={num} value={num}>
                          {num}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                </Grid>
                <Grid item xs={12} sm={2}>
                  <TextField
                    fullWidth
                    label="Monto por Cuota"
                    variant="outlined"
                    size="small"
                    value={`$${montoCuotas.toFixed(2)}`}
                    sx={{ boxShadow: 3 }}
                    InputProps={{ readOnly: true }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <FormControlLabel
                    control={
                      <Checkbox
                        checked={pagoInicialEnDivisas}
                        onChange={(e) =>
                          setPagoInicialEnDivisas(e.target.checked)
                        }
                        color="primary"
                      />
                    }
                    label="Inicial en Divisas (20% Descuento)"
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <Button
                    variant="contained"
                    color="primary"
                    onClick={handleAgregar}
                    fullWidth
                    sx={{
                      backgroundColor: "#0A4575",
                      "&:hover": { backgroundColor: "#083a5e" },
                    }}
                  >
                    Crear Solicitud
                  </Button>
                </Grid>
              </>
            )}
          </>
        )}
      </Grid>
    </Box>
  );
};

export default Cremacion;
