import React, { useState, useEffect } from "react";
import axios from "axios";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Button,
} from "@mui/material";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";
import { jwtDecode } from "jwt-decode";

const apiUrl = process.env.REACT_APP_API_URL;

const ReporteCuotas = () => {
  const [cuotas, setCuotas] = useState([]);
  const token = localStorage.getItem("access_token");
  const vendedorId = token ? Number(jwtDecode(token).sub) : null;

  useEffect(() => {
    if (vendedorId) {
      fetchCuotas();
    }
  }, [vendedorId]);

  const fetchCuotas = async () => {
    try {
      const response = await axios.get(
        `${apiUrl}/api/contratos/cuotas?vendedor_id=${vendedorId}`,
        {
          headers: { Authorization: `Bearer ${token}` },
        }
      );
      setCuotas(response.data);
    } catch (error) {
      toast.error("Error al cargar las cuotas.");
    }
  };

  const handlePagoCuota = async (cuotaId) => {
    try {
      await axios.post(
        `${apiUrl}/api/contratos/pagar-cuota`,
        { cuota_id: cuotaId },
        { headers: { Authorization: `Bearer ${token}` } }
      );
      toast.success("Pago registrado exitosamente.");
      fetchCuotas(); // Recargar cuotas
    } catch (error) {
      toast.error("Error al registrar el pago.");
    }
  };

  const getButtonStyle = (fechaVencimiento, estado) => {
    const hoy = new Date();
    const vencimiento = new Date(fechaVencimiento);
    if (estado === "pagado") {
      return { backgroundColor: "#4CAF50", color: "#FFFFFF" }; // Verde con texto blanco
    } else if (vencimiento < hoy && estado === "pendiente") {
      return { backgroundColor: "#FF6347", color: "#FFFFFF" }; // Rojo con texto blanco
    }
    return { backgroundColor: "#f4d03f", color: "#000000" }; // Amarillo con texto negro
  };

  return (
    <Box
      sx={{
        maxWidth: "950px",
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        sx={{
          textAlign: "center",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          mb: 4,
        }}
        variant="h6"
      >
        REPORTE DE CUOTAS
      </Typography>
      <TableContainer component={Paper}>
        <Table>
          <TableHead sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}>
            <TableRow>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Contrato
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Fecha Vencimiento
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Monto
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Estado
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Acción
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {cuotas.map((cuota) => (
              <TableRow key={cuota.id}>
                <TableCell>{cuota.numero_contrato}</TableCell>
                <TableCell>
                  {new Date(cuota.fecha_vencimiento).toLocaleDateString()}
                </TableCell>
                <TableCell>${cuota.monto}</TableCell>
                <TableCell>
                  <Button
                    variant="contained"
                    sx={{
                      ...getButtonStyle(cuota.fecha_vencimiento, cuota.estado),
                      minWidth: "120px",
                    }}
                    disabled={cuota.estado === "pagado"}
                  >
                    {cuota.estado === "pagado"
                      ? "Pagado"
                      : cuota.estado === "vencido"
                      ? "Vencido"
                      : "Cuota Pendiente"}
                  </Button>
                </TableCell>
                <TableCell>
                  {cuota.estado !== "pagado" && (
                    <Button
                      variant="contained"
                      onClick={() => handlePagoCuota(cuota.id)}
                      sx={{
                        backgroundColor: "#0A4575",
                        "&:hover": { backgroundColor: "#083a5e" },
                      }}
                    >
                      Pagar Cuota
                    </Button>
                  )}
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </TableContainer>
      <Box sx={{ mt: 3, textAlign: "right" }}>
        <Button
          variant="contained"
          component={Link}
          to="/ventas"
          sx={{
            backgroundColor: "#0A4575",
            "&:hover": { backgroundColor: "#083a5e" },
          }}
        >
          Volver
        </Button>
      </Box>
    </Box>
  );
};

export default ReporteCuotas;
