import React, { useEffect, useState } from "react";
import axios from "axios";
import {
  Box,
  Typography,
  Button,
  Collapse,
  IconButton,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  CircularProgress,
  Alert,
  Modal,
  Grid,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  TextField,
} from "@mui/material";
import {
  ArrowBack,
  ExpandMore as ExpandMoreIcon,
  ExpandLess as ExpandLessIcon,
  Inbox as InboxIcon,
} from "@mui/icons-material";
import { format } from "date-fns";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";
import { useDollar } from "../../DollarPrice";
import { jwtDecode } from "jwt-decode";

// Valores fijos para pagos en Bolívares
const BANCO_RECEPTOR = "Banco Banesco";
const NUMERO_CUENTA = "0134-0319-82-3191111990";
const TELEFONO_PAGO_MOVIL = "0414 4282678";
const RIF = "J-408124637";

const ReporteCuotas = () => {
  const [contratos, setContratos] = useState([]);
  const [expandedContrato, setExpandedContrato] = useState(null);
  const [cuotas, setCuotas] = useState({});
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState(null);
  const [montoPorCuota, setMontoPorCuota] = useState(0);
  const [modalOpen, setModalOpen] = useState(false);
  const [cuotaSeleccionada, setCuotaSeleccionada] = useState(null);
  const [nroCuotaSeleccionada, setNroCuotaSeleccionada] = useState(null);
  const [contratoSeleccionado, setContratoSeleccionado] = useState(null);
  const { dollarPrice } = useDollar();
  const montoCuota = parseFloat(cuotas?.cuota || 0);
  const apiUrl = process.env.REACT_APP_API_URL;

  const getColor = (estado) => {
    if (estado === "pagado") return "#4caf50";
    if (estado === "vencido") return "#f44336";
    return "#ffeb3b";
  };

  useEffect(() => {
    const fetchContratos = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const res = await axios.get(
          `${apiUrl}/api/v1/cuotas/contratos-financiados`,
          {
            headers: { Authorization: `Bearer ${token}` },
          }
        );
        setContratos(res.data);
      } catch (err) {
        const errorMessage = err.response?.data?.detail
          ? Array.isArray(err.response.data.detail)
            ? err.response.data.detail.map((e) => e.msg).join("; ")
            : err.response.data.detail
          : "Error al cargar contratos";
        setError(errorMessage);
      } finally {
        setLoading(false);
      }
    };
    fetchContratos();
  }, [apiUrl]);

  const handleToggleContrato = async (contratoId) => {
    if (expandedContrato === contratoId) {
      setExpandedContrato(null);
      setContratoSeleccionado(null);
    } else {
      setExpandedContrato(contratoId);
      const contrato = contratos.find((c) => c.id === contratoId);
      setContratoSeleccionado(contrato);
      if (!cuotas[contratoId]) {
        try {
          const token = localStorage.getItem("access_token");
          const res = await axios.get(
            `${apiUrl}/api/v1/cuotas/cuotas/${contratoId}`,
            {
              headers: { Authorization: `Bearer ${token}` },
            }
          );
          setCuotas((prev) => ({ ...prev, [contratoId]: res.data }));
        } catch (err) {
          const errorMessage = err.response?.data?.detail
            ? Array.isArray(err.response.data.detail)
              ? err.response.data.detail.map((e) => e.msg).join("; ")
              : err.response.data.detail
            : "Error al cargar cuotas del contrato";
          if (
            err.response?.status === 404 ||
            errorMessage.includes("no se encontró")
          ) {
            setCuotas((prev) => ({ ...prev, [contratoId]: [] }));
          } else {
            setError(errorMessage);
          }
        }
      }
    }
  };

  const calcularResumen = (cuotasList, responseData = null) => {
    if (responseData) {
      return {
        pagadas: responseData.cuotas_pagadas,
        pendientes: responseData.cuotas_pendientes,
        totalPendiente: responseData.total_pendiente.toFixed(2),
        montoPorCuota:
          cuotasList.length > 0
            ? Number(cuotasList[0].cuotas).toFixed(2)
            : "0.00",
      };
    }

    const pagadas = cuotasList.filter((c) => c.estado_pago === "pagado").length;
    const pendientes = cuotasList.filter(
      (c) => c.estado_pago !== "pagado"
    ).length;

    const totalPendiente = cuotasList
      .filter((c) => c.estado_pago !== "pagado")
      .reduce((acc, c) => acc + Number(c.cuotas || 0), 0)
      .toFixed(2);

    return { pagadas, pendientes, totalPendiente };
  };

  const abrirModalPago = (cuota, index) => {
    const contrato = contratos.find((c) => c.id === cuota.contrato_id);
    setContratoSeleccionado(contrato);
    if (!cuota || !contrato) {
      toast.error("Faltan datos del contrato o cuota seleccionada.");
      return;
    }
    setCuotaSeleccionada(cuota);
    setNroCuotaSeleccionada(index + 1);
    setModalOpen(true);
    setMontoPorCuota(contrato.montoPorCuotas);
  };

  const cerrarModalPago = () => {
    setModalOpen(false);
    setCuotaSeleccionada(null);
    setNroCuotaSeleccionada(null);
    setContratoSeleccionado(null);
  };

  const handlePagoRegistrado = async (contratoId, responseData) => {
    if (!contratoId) {
      toast.error("ID del contrato no válido.");
      return;
    }
    try {
      const token = localStorage.getItem("access_token");
      const resCuotas = await axios.get(
        `${apiUrl}/api/v1/cuotas/cuotas/${contratoId}`,
        {
          headers: { Authorization: `Bearer ${token}` },
        }
      );
      setCuotas((prev) => ({ ...prev, [contratoId]: resCuotas.data }));

      const resContratos = await axios.get(
        `${apiUrl}/api/v1/cuotas/contratos-financiados`,
        {
          headers: { Authorization: `Bearer ${token}` },
        }
      );
      setContratos(resContratos.data);
      toast.success(responseData.message);
    } catch (err) {
      toast.error("Error al actualizar los datos del contrato");
    }
    cerrarModalPago();
  };

  const ReportePagosCuotasModal = ({
    open,
    onClose,
    cuota,
    numeroContrato,
    tipoServicio,
    monto,
    contratoId,
    nroCuota,
    onPagoRegistrado,
  }) => {
    const { dollarPrice } = useDollar();
    const [tipoPago, setTipoPago] = useState("USD");
    const [metodoPagoVes, setMetodoPagoVes] = useState("");
    const [pagoData, setPagoData] = useState({
      montoUsd: montoCuota.toFixed(2), // por defecto en USD
      montoVes: (montoCuota * dollarPrice).toFixed(2), // calculado si cambia a VES
      observaciones: "",
      comprobanteUsd: null,
      bancoEmisorVes: "",
      numeroTransferenciaVes: "",
      comprobanteTransferenciaVes: null,
      referenciaPagoMovilVes: "",
      comprobantePagoMovilVes: null,
      numeroContrato: numeroContrato || "",
      tipoServicio: tipoServicio || "",
      nroCuota: nroCuota || 1,
    });

    const [errorComprobanteUsd, setErrorComprobanteUsd] = useState(false);
    const [errorComprobanteVes, setErrorComprobanteVes] = useState(false);

    const banks = [
      { value: "", label: "Seleccione un Banco" },
      { value: "Banesco", label: "Banco Banesco Banco Universal, C.A." },
      { value: "100% Banco", label: "100% Banco Banco Universal, C.A." },
      { value: "Banca Amiga", label: "Banca Amiga, Banco Microfinanciero" },
      { value: "Bancaribe", label: "Bancaribe, C.A. Banco Universal" },
      { value: "Banco Activo", label: "Banco Activo" },
      { value: "Banco Bicentenario", label: "Banco Bicentenario" },
      { value: "Banco Caroní", label: "Banco Caroní, C.A. Banco Universal" },
      { value: "Banco de la FANB", label: "Banco de la FANB" },
      { value: "Banco de Venezuela", label: "Banco de Venezuela" },
      { value: "Banco del Tesoro", label: "Banco del Tesoro" },
      {
        value: "Banco Exterior",
        label: "Banco Exterior, C.A. Banco Universal",
      },
      {
        value: "Banco Internacional de Desarrollo",
        label: "Banco Internacional de Desarrollo",
      },
      { value: "Banco Mercantil", label: "Banco Mercantil" },
      { value: "BNC", label: "Banco Nacional de Crédito (BNC)" },
      { value: "Banco Plaza", label: "Banco Plaza, C.A. Banco Universal" },
      { value: "Banco Provincial", label: "Banco Provincial" },
      { value: "Banco Sofitasa", label: "Banco Sofitasa" },
      { value: "Bancrecer", label: "Bancrecer, S.A. Banco Microfinanciero" },
      { value: "Banplus", label: "Banplus Banco Universal, C.A." },
      { value: "Banco Fondo Común", label: "BFC Banco Fondo Común" },
      { value: "Mi Banco", label: "Mi Banco, Banco Microfinanciero, C.A." },
      {
        value: "Banco Venezolano de Credito",
        label: "Banco Venezolano de Credito",
      },
    ];

    useEffect(() => {
      const montoTotal = parseFloat(monto || 0);
      const montoUsd =
        tipoPago === "USD"
          ? montoTotal.toFixed(2)
          : tipoPago === "Ambos"
          ? pagoData.montoUsd
          : "";
      const montoVes =
        tipoPago === "VES"
          ? (montoTotal * dollarPrice).toFixed(2)
          : tipoPago === "Ambos" && pagoData.montoUsd
          ? (
              (montoTotal - parseFloat(pagoData.montoUsd || 0)) *
              dollarPrice
            ).toFixed(2)
          : "0.00";
      setPagoData((prev) => ({
        ...prev,
        montoUsd,
        montoVes,
        observaciones: "",
        comprobanteUsd: null,
        bancoEmisorVes: "",
        numeroTransferenciaVes: "",
        comprobanteTransferenciaVes: null,
        referenciaPagoMovilVes: "",
        comprobantePagoMovilVes: null,
        numeroContrato: numeroContrato || "",
        tipoServicio: tipoServicio || "",
        nroCuota: nroCuota || 1,
      }));
      setErrorComprobanteUsd(false);
      setErrorComprobanteVes(false);
    }, [
      monto,
      dollarPrice,
      numeroContrato,
      tipoServicio,
      open,
      nroCuota,
      tipoPago,
    ]);

    const handleTipoPagoChange = (event) => {
      const newTipoPago = event.target.value;
      setTipoPago(newTipoPago);

      if (newTipoPago === "USD") {
        setPagoData((prev) => ({
          ...prev,
          montoUsd: montoCuota.toFixed(2),
          montoVes: "0.00",
        }));
      } else if (newTipoPago === "VES") {
        setPagoData((prev) => ({
          ...prev,
          montoUsd: "0.00",
          montoVes: (montoCuota * dollarPrice).toFixed(2),
        }));
      }
    };

    const handleMetodoPagoVesChange = (event) => {
      setMetodoPagoVes(event.target.value);
      setPagoData((prev) => ({
        ...prev,
        bancoEmisorVes: "",
        numeroTransferenciaVes: "",
        comprobanteTransferenciaVes: null,
        referenciaPagoMovilVes: "",
        comprobantePagoMovilVes: null,
      }));
      setErrorComprobanteVes(false);
    };

    const handlePagoDataChange = (e) => {
      const { name, value } = e.target;
      setPagoData((prev) => {
        const newData = { ...prev, [name]: value };
        if (name === "montoUsd" && tipoPago === "Ambos") {
          const montoTotal = parseFloat(monto || 0);
          const montoUsd = parseFloat(value || 0);
          newData.montoVes =
            montoUsd <= montoTotal
              ? ((montoTotal - montoUsd) * dollarPrice).toFixed(2)
              : "0.00";
        }
        return newData;
      });
    };

    const handleComprobanteChange = (e, tipo) => {
      const file = e.target.files[0];
      if (file) {
        setPagoData((prev) => ({
          ...prev,
          [tipo]: file,
        }));
        if (tipo === "comprobanteUsd") {
          setErrorComprobanteUsd(false);
        } else {
          setErrorComprobanteVes(false);
        }
      }
    };

    const isFormValid = () => {
      if (tipoPago === "USD" || tipoPago === "Ambos") {
        if (!pagoData.montoUsd || !pagoData.comprobanteUsd) {
          return false;
        }
      }
      if (tipoPago === "VES" || tipoPago === "Ambos") {
        if (!pagoData.montoVes || !metodoPagoVes) {
          return false;
        }
        if (metodoPagoVes === "transferencia") {
          return (
            !!pagoData.bancoEmisorVes &&
            !!pagoData.numeroTransferenciaVes &&
            !!pagoData.comprobanteTransferenciaVes
          );
        }
        if (metodoPagoVes === "pagoMovil") {
          return (
            !!pagoData.bancoEmisorVes &&
            !!pagoData.referenciaPagoMovilVes &&
            !!pagoData.comprobantePagoMovilVes
          );
        }
      }
      return true;
    };

    const handleSubmit = async (e) => {
      e.preventDefault();

      const token = localStorage.getItem("access_token");
      let vendedorId = null;

      if (token) {
        try {
          const decoded = jwtDecode(token);
          vendedorId = decoded.sub ? Number(decoded.sub) : null;
        } catch (error) {
          toast.error("No se pudo identificar al vendedor.");
          return;
        }
      }

      try {
        const montoTotal = parseFloat(monto || 0);
        const montoUsd = parseFloat(pagoData.montoUsd || 0);
        const montoVes = parseFloat(pagoData.montoVes || 0);
        const montoPagado = montoUsd + montoVes / dollarPrice;

        if (tipoPago === "Ambos" && Math.abs(montoPagado - montoTotal) > 0.01) {
          toast.error("El monto pagado no coincide con el monto de la cuota.");
          return;
        }

        const formData = new FormData();
        formData.append("vendedor_id", vendedorId);
        formData.append("cuota_id", cuota.id);
        formData.append("contrato_id", contratoId);
        formData.append("numero_contrato", pagoData.numeroContrato);
        formData.append("tipo_servicio", pagoData.tipoServicio);
        formData.append("dollar_price", dollarPrice); // tipo de cambio actual
        formData.append("monto_restante", cuota.precioFinanciado); // saldo pendiente en la BD
        formData.append("metodo_pago", tipoPago);
        if (tipoPago === "VES")
          formData.append("banco_emisor", pagoData.bancoEmisorVes || "");
        if (tipoPago === "VES")
          formData.append("banco_receptor", BANCO_RECEPTOR || "");
        formData.append(
          "referencia",
          tipoPago === "VES" ? pagoData.referenciaPagoMovilVes : 0
        );
        formData.append("nro_cuota", pagoData.nroCuota);
        formData.append("monto_pagado", montoPagado.toFixed(2));

        if (pagoData.comprobanteUsd instanceof File) {
          formData.append("comprobante_usd", pagoData.comprobanteUsd);
        }
        if (pagoData.comprobanteTransferenciaVes instanceof File) {
          formData.append(
            "comprobante_ves",
            pagoData.comprobanteTransferenciaVes
          );
        } else if (pagoData.comprobantePagoMovilVes instanceof File) {
          formData.append("comprobante_ves", pagoData.comprobantePagoMovilVes);
        }
        formData.append("observaciones", pagoData.observaciones || "");
        formData.append("estado_pago", "pagado");

        const response = await axios.post(
          `${apiUrl}/api/v1/cuotas/pago-cuota`,
          formData,
          {
            headers: {
              Authorization: `Bearer ${token}`,
              "Content-Type": "multipart/form-data",
            },
          }
        );

        onPagoRegistrado(contratoId, response.data);
      } catch (error) {
        const errorMessage = error.response?.data?.detail
          ? Array.isArray(error.response.data.detail)
            ? error.response.data.detail.map((e) => e.msg).join("; ")
            : error.response.data.detail
          : "Error al registrar el pago de la cuota";
        toast.error(errorMessage);
      }
    };

    return (
      <Modal open={open} onClose={onClose}>
        <Box
          sx={{
            position: "absolute",
            top: "50%",
            left: "50%",
            transform: "translate(-50%, -50%)",
            width: "90%",
            maxWidth: "950px",
            bgcolor: "background.paper",
            boxShadow: 24,
            p: 4,
            borderRadius: 2,
            maxHeight: "90vh",
            overflowY: "auto",
          }}
        >
          <Box
            sx={{
              display: "flex",
              justifyContent: "space-between",
              alignItems: "center",
            }}
          >
            <Typography
              sx={{ fontFamily: "Bakbak One", color: "#0A4575" }}
              variant="h5"
            >
              REPORTAR PAGO - CUOTA #{pagoData.nroCuota}
            </Typography>
            <Typography
              sx={{
                fontFamily: "Bakbak One",
                color: "#0A4575",
                fontSize: "20px",
              }}
            >
              Monto: ${montoPorCuota}
            </Typography>
          </Box>
          <Grid container spacing={2}>
            <Grid item xs={12} sm={3}>
              <Typography
                sx={{
                  fontFamily: "Bakbak One",
                  color: "#0A4575",
                  fontSize: "20px",
                  mt: 1,
                }}
              >
                Servicio: {pagoData.tipoServicio}
              </Typography>
            </Grid>
            <Grid item xs={12} sm={3}>
              <Typography
                sx={{
                  fontFamily: "Bakbak One",
                  color: "#0A4575",
                  fontSize: "20px",
                  mt: 1,
                  mb: 2,
                }}
              >
                Contrato: {pagoData.numeroContrato}
              </Typography>
            </Grid>
          </Grid>
          <form onSubmit={handleSubmit}>
            <Grid container spacing={2}>
              <Grid item xs={12} sm={3}>
                <FormControl
                  fullWidth
                  variant="outlined"
                  size="small"
                  sx={{ boxShadow: 3 }}
                >
                  <InputLabel>Tipo de Pago</InputLabel>
                  <Select
                    value={tipoPago}
                    onChange={handleTipoPagoChange}
                    label="Tipo de Pago"
                    name="tipoPago"
                  >
                    <MenuItem value="USD">USD</MenuItem>
                    <MenuItem value="VES">VES</MenuItem>
                  </Select>
                </FormControl>
              </Grid>
              {(tipoPago === "USD" || tipoPago === "Ambos") && (
                <>
                  <Grid item xs={12} sm={3}>
                    <TextField
                      fullWidth
                      label="Monto en USD"
                      variant="outlined"
                      size="small"
                      name="montoUsd"
                      type="number"
                      value={pagoData.montoUsd}
                      onChange={handlePagoDataChange}
                      required
                      sx={{ boxShadow: 3 }}
                    />
                  </Grid>
                  <Grid item xs={12} sm={6}>
                    <input
                      accept="image/*,application/pdf"
                      style={{ display: "none" }}
                      id="comprobante-usd"
                      type="file"
                      onChange={(e) =>
                        handleComprobanteChange(e, "comprobanteUsd")
                      }
                      required
                    />
                    <label htmlFor="comprobante-usd">
                      <Button
                        variant="contained"
                        component="span"
                        sx={{
                          backgroundColor: "#0A4575",
                          color: "#FFFFFF",
                          "&:hover": { backgroundColor: "#083a5e" },
                        }}
                      >
                        Subir Comprobante USD
                      </Button>
                    </label>
                    {pagoData.comprobanteUsd && (
                      <Typography
                        sx={{ marginTop: "10px", fontFamily: "Abel" }}
                      >
                        <b>{pagoData.comprobanteUsd.name}</b>
                      </Typography>
                    )}
                    {errorComprobanteUsd && (
                      <Typography
                        sx={{
                          color: "red",
                          fontSize: "12px",
                          marginTop: "5px",
                        }}
                      >
                        Por favor, suba el comprobante en USD.
                      </Typography>
                    )}
                  </Grid>
                </>
              )}
              {(tipoPago === "VES" || tipoPago === "Ambos") && (
                <>
                  <Grid item xs={12} sm={3}>
                    <TextField
                      fullWidth
                      label="Monto en VES"
                      variant="outlined"
                      size="small"
                      name="montoVes"
                      type="number"
                      value={pagoData.montoVes}
                      onChange={handlePagoDataChange}
                      required
                      sx={{ boxShadow: 3 }}
                    />
                  </Grid>
                  <Grid item xs={12} sm={3}>
                    <FormControl
                      fullWidth
                      variant="outlined"
                      size="small"
                      sx={{ boxShadow: 3 }}
                    >
                      <InputLabel>Método de Pago VES</InputLabel>
                      <Select
                        value={metodoPagoVes}
                        onChange={handleMetodoPagoVesChange}
                        label="Método de Pago VES"
                        name="metodoPagoVes"
                      >
                        <MenuItem value="">Seleccione un método</MenuItem>
                        <MenuItem value="transferencia">
                          Transferencia Bancaria
                        </MenuItem>
                        <MenuItem value="pagoMovil">Pago Móvil</MenuItem>
                      </Select>
                    </FormControl>
                  </Grid>
                  {metodoPagoVes === "transferencia" && (
                    <Grid item xs={12}>
                      <Grid container spacing={2}>
                        <Grid item xs={12} sm={3}>
                          <FormControl
                            fullWidth
                            variant="outlined"
                            size="small"
                            sx={{ boxShadow: 3 }}
                          >
                            <InputLabel>Banco Emisor</InputLabel>
                            <Select
                              name="bancoEmisorVes"
                              value={pagoData.bancoEmisorVes}
                              onChange={handlePagoDataChange}
                              label="Banco Emisor"
                              required
                            >
                              {banks.map((bank) => (
                                <MenuItem key={bank.value} value={bank.value}>
                                  {bank.label}
                                </MenuItem>
                              ))}
                            </Select>
                          </FormControl>
                        </Grid>
                        <Grid item xs={12} sm={3}>
                          <TextField
                            fullWidth
                            label="Número de Transferencia"
                            variant="outlined"
                            size="small"
                            name="numeroTransferenciaVes"
                            value={pagoData.numeroTransferenciaVes}
                            onChange={handlePagoDataChange}
                            required
                            sx={{ boxShadow: 3 }}
                          />
                        </Grid>
                        <Grid item xs={12} sm={3}>
                          <input
                            accept="image/*,application/pdf"
                            style={{ display: "none" }}
                            id="comprobante-transferencia-ves"
                            type="file"
                            onChange={(e) =>
                              handleComprobanteChange(
                                e,
                                "comprobanteTransferenciaVes"
                              )
                            }
                            required
                          />
                          <label htmlFor="comprobante-transferencia-ves">
                            <Button
                              variant="contained"
                              component="span"
                              sx={{
                                backgroundColor: "#0A4575",
                                color: "#FFFFFF",
                                "&:hover": { backgroundColor: "#083a5e" },
                              }}
                            >
                              Subir Comprobante de Pago
                            </Button>
                          </label>
                          {pagoData.comprobanteTransferenciaVes && (
                            <Typography
                              sx={{ marginTop: "10px", fontFamily: "Abel" }}
                            >
                              <b>{pagoData.comprobanteTransferenciaVes.name}</b>
                            </Typography>
                          )}
                          {errorComprobanteVes && (
                            <Typography
                              sx={{
                                color: "red",
                                fontSize: "12px",
                                marginTop: "5px",
                              }}
                            >
                              Por favor, suba el comprobante de transferencia.
                            </Typography>
                          )}
                        </Grid>
                      </Grid>
                    </Grid>
                  )}
                  {metodoPagoVes === "pagoMovil" && (
                    <Grid item xs={12}>
                      <Grid container spacing={2}>
                        <Grid item xs={12} sm={3}>
                          <FormControl
                            fullWidth
                            variant="outlined"
                            size="small"
                            sx={{ boxShadow: 3 }}
                          >
                            <InputLabel>Banco Emisor</InputLabel>
                            <Select
                              name="bancoEmisorVes"
                              value={pagoData.bancoEmisorVes}
                              onChange={handlePagoDataChange}
                              label="Banco Emisor"
                              required
                            >
                              {banks.map((bank) => (
                                <MenuItem key={bank.value} value={bank.value}>
                                  {bank.label}
                                </MenuItem>
                              ))}
                            </Select>
                          </FormControl>
                        </Grid>
                        <Grid item xs={12} sm={3}>
                          <TextField
                            fullWidth
                            label="Referencia"
                            variant="outlined"
                            size="small"
                            name="referenciaPagoMovilVes"
                            value={pagoData.referenciaPagoMovilVes}
                            onChange={handlePagoDataChange}
                            required
                            sx={{ boxShadow: 3 }}
                          />
                        </Grid>
                        <Grid item xs={12} sm={3}>
                          <input
                            accept="image/*,application/pdf"
                            style={{ display: "none" }}
                            id="comprobante-pago-movil-ves"
                            type="file"
                            onChange={(e) =>
                              handleComprobanteChange(
                                e,
                                "comprobantePagoMovilVes"
                              )
                            }
                            required
                          />
                          <label htmlFor="comprobante-pago-movil-ves">
                            <Button
                              variant="contained"
                              component="span"
                              sx={{
                                backgroundColor: "#0A4575",
                                color: "#FFFFFF",
                                "&:hover": { backgroundColor: "#083a5e" },
                              }}
                            >
                              Subir Comprobante de Pago
                            </Button>
                          </label>
                          {pagoData.comprobantePagoMovilVes && (
                            <Typography
                              sx={{ marginTop: "10px", fontFamily: "Abel" }}
                            >
                              <b>{pagoData.comprobantePagoMovilVes.name}</b>
                            </Typography>
                          )}
                          {errorComprobanteVes && (
                            <Typography
                              sx={{
                                color: "red",
                                fontSize: "12px",
                                marginTop: "5px",
                              }}
                            >
                              Por favor, suba el comprobante de pago móvil.
                            </Typography>
                          )}
                        </Grid>
                      </Grid>
                    </Grid>
                  )}
                </>
              )}

              {(metodoPagoVes === "pagoMovil" ||
                metodoPagoVes === "transferencia") && (
                <Grid item xs={12} sm={6}>
                  <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                    Banco Receptor: {BANCO_RECEPTOR}
                  </Typography>
                  {metodoPagoVes === "pagoMovil" && (
                    <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                      Teléfono: {TELEFONO_PAGO_MOVIL}
                    </Typography>
                  )}
                  {metodoPagoVes === "transferencia" && (
                    <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                      Número de Cuenta: {NUMERO_CUENTA}
                    </Typography>
                  )}
                  <Typography sx={{ fontFamily: "Abel", fontWeight: "bold" }}>
                    RIF: {RIF}
                  </Typography>
                </Grid>
              )}

              <Box
                sx={{
                  display: "flex",
                  justifyContent: "flex-end",
                  alignItems: "center",
                  width: "100%",
                  mt: 3,
                  pr: 2,
                }}
              >
                <Button
                  variant="outlined"
                  onClick={onClose}
                  sx={{ mr: 2, minWidth: "120px" }}
                >
                  Cancelar
                </Button>
                <Button
                  type="submit"
                  variant="contained"
                  disabled={!isFormValid()}
                  sx={{
                    backgroundColor: "#0A4575",
                    color: "#FFFFFF",
                    "&:hover": { backgroundColor: "#083a5e" },
                    minWidth: "120px",
                  }}
                >
                  Reportar Pago
                </Button>
              </Box>
            </Grid>
          </form>
        </Box>
      </Modal>
    );
  };

  if (loading) return <CircularProgress />;
  if (error) return <Alert severity="error">{error}</Alert>;

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        variant="h6"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        CUOTAS DE CONTRATOS FINANCIADOS
      </Typography>
      <TableContainer component={Paper}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Número Contrato</TableCell>
              <TableCell sx={{ color: "#fff" }}>Vendedor</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
              <TableCell sx={{ color: "#fff" }}>fecha Contrato</TableCell>
              <TableCell sx={{ color: "#fff" }}>Tipo de Servicio</TableCell>
              <TableCell sx={{ color: "#fff" }}>Pago Inicial</TableCell>
              <TableCell sx={{ color: "#fff" }}>
                Monto Total con Descuento
              </TableCell>
              <TableCell sx={{ color: "#fff" }}>Precio Plan</TableCell>
              <TableCell sx={{ color: "#fff" }}>Acción</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {contratos.map((contrato) => (
              <React.Fragment key={contrato.id}>
                <TableRow>
                  <TableCell>{contrato.numero_contrato}</TableCell>
                  <TableCell>{contrato.vendedor || "-"}</TableCell>
                  <TableCell>{contrato.cliente || "Sin cliente"}</TableCell>
                  <TableCell>
                    {format(new Date(contrato.fecha_creacion), "dd/MM/yyyy")}
                  </TableCell>
                  <TableCell>{contrato.tipo_servicio}</TableCell>
                  <TableCell>
                    ${contrato.pagoInicial?.toFixed(2) || "0.00"}
                  </TableCell>
                  <TableCell>
                    ${contrato.items_adicionales.precioConDescuento.toFixed(2)}
                  </TableCell>
                  <TableCell>${contrato.precioPlan.toFixed(2)}</TableCell>
                  <TableCell>
                    <IconButton
                      onClick={() => handleToggleContrato(contrato.id)}
                    >
                      {expandedContrato === contrato.id ? (
                        <ExpandLessIcon />
                      ) : (
                        <ExpandMoreIcon />
                      )}
                    </IconButton>
                  </TableCell>
                </TableRow>
                <TableRow>
                  <TableCell colSpan={9} sx={{ padding: 0 }}>
                    <Collapse
                      in={expandedContrato === contrato.id}
                      timeout="auto"
                      unmountOnExit
                    >
                      <Box sx={{ padding: 2, backgroundColor: "#f9f9f9" }}>
                        <Typography
                          variant="h6"
                          align="left"
                          gutterBottom
                          sx={{
                            fontFamily: "Bakbak One, sans-serif",
                            color: "#0A4575",
                          }}
                        >
                          DETALLES DE CUOTAS
                        </Typography>
                        {cuotas[contrato.id] &&
                        cuotas[contrato.id].length > 0 ? (
                          <>
                            <Table size="small">
                              <TableHead sx={{ backgroundColor: "#0A4575" }}>
                                <TableRow>
                                  <TableCell
                                    sx={{
                                      color: "#FFFFFF",
                                      fontSize: "16px",
                                      width: "110px",
                                    }}
                                  >
                                    Cuota
                                  </TableCell>
                                  <TableCell
                                    sx={{
                                      color: "#FFFFFF",
                                      fontSize: "16px",
                                      width: "110px",
                                    }}
                                  >
                                    Fecha
                                  </TableCell>
                                  <TableCell
                                    sx={{
                                      color: "#FFFFFF",
                                      fontSize: "16px",
                                      width: "110px",
                                    }}
                                  >
                                    Monto por Cuotas
                                  </TableCell>
                                  <TableCell
                                    sx={{
                                      color: "#FFFFFF",
                                      fontSize: "16px",
                                      width: "110px",
                                    }}
                                  >
                                    Estado
                                  </TableCell>
                                  <TableCell
                                    sx={{
                                      color: "#FFFFFF",
                                      fontSize: "16px",
                                      width: "110px",
                                    }}
                                  >
                                    Acción
                                  </TableCell>
                                </TableRow>
                              </TableHead>
                              <TableBody>
                                {cuotas[contrato.id].map((cuota, index) => {
                                  const resumen = calcularResumen(
                                    cuotas[contrato.id]
                                  );
                                  return (
                                    <TableRow key={cuota.id || index}>
                                      <TableCell>#{index + 1}</TableCell>
                                      <TableCell>
                                        {format(
                                          new Date(cuota.fecha_vencimiento),
                                          "dd/MM/yyyy"
                                        )}
                                      </TableCell>
                                      <TableCell>
                                        ${cuota.cuotas.toFixed(2)}
                                      </TableCell>
                                      <TableCell>
                                        <Box
                                          sx={{
                                            backgroundColor: getColor(
                                              cuota.estado_pago
                                            ),
                                            color:
                                              cuota.estado_pago === "pagado"
                                                ? "white"
                                                : "black",
                                            px: 2,
                                            py: 0.5,
                                            borderRadius: 1,
                                            textAlign: "center",
                                          }}
                                        >
                                          {cuota.estado_pago.toUpperCase()}
                                        </Box>
                                      </TableCell>
                                      <TableCell>
                                        {cuota.estado_pago !== "pagado" && (
                                          <Button
                                            variant="contained"
                                            onClick={() =>
                                              abrirModalPago(cuota, index)
                                            }
                                          >
                                            Reportar Pago
                                          </Button>
                                        )}
                                      </TableCell>
                                    </TableRow>
                                  );
                                })}
                              </TableBody>
                            </Table>
                            <Grid
                              container
                              justifyContent="flex-end"
                              spacing={2}
                              sx={{ mt: 1 }}
                            >
                              <Grid item>
                                <Typography variant="subtitle1">
                                  <strong>Cuotas Pagadas:</strong>{" "}
                                  {calcularResumen(cuotas[contrato.id]).pagadas}
                                </Typography>
                              </Grid>
                              <Grid item>
                                <Typography variant="subtitle1">
                                  <strong>Cuotas Pendientes:</strong>{" "}
                                  {
                                    calcularResumen(cuotas[contrato.id])
                                      .pendientes
                                  }
                                </Typography>
                              </Grid>
                              <Grid item>
                                <Typography variant="subtitle1">
                                  <strong>Total Pendiente:</strong> $
                                  {
                                    calcularResumen(cuotas[contrato.id])
                                      .totalPendiente
                                  }
                                </Typography>
                              </Grid>
                            </Grid>
                          </>
                        ) : (
                          <Typography sx={{ mt: 2 }}>
                            Este contrato aún no tiene cuotas generadas.
                          </Typography>
                        )}
                      </Box>
                    </Collapse>
                  </TableCell>
                </TableRow>
              </React.Fragment>
            ))}
          </TableBody>
        </Table>
      </TableContainer>
      <ReportePagosCuotasModal
        open={modalOpen}
        onClose={cerrarModalPago}
        cuota={cuotaSeleccionada}
        numeroContrato={cuotaSeleccionada?.numero_contrato}
        tipoServicio={
          cuotaSeleccionada?.tipo_servicio ||
          contratos.find((c) => c.id === cuotaSeleccionada?.contrato_id)
            ?.tipo_servicio ||
          ""
        }
        monto={cuotaSeleccionada?.cuotas}
        contratoId={cuotaSeleccionada?.contrato_id}
        nroCuota={nroCuotaSeleccionada}
        onPagoRegistrado={handlePagoRegistrado}
      />
      <Button
        variant="contained"
        component={Link}
        to="/ventas"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
    </Box>
  );
};

export default ReporteCuotas;
