import React, { useState, useEffect } from "react";
import {
  Box,
  Typography,
  Grid,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  TextField,
  Button,
  Modal,
  Backdrop,
} from "@mui/material";
import axios from "axios";
import { jwtDecode } from "jwt-decode";
import { toast } from "react-toastify";
import { useDollar } from "../../DollarPrice";

const BANCO_RECEPTOR = "Banco Banesco";
const NUMERO_CUENTA = "0134-0319-82-3191111990";
const TELEFONO_PAGO_MOVIL = "0414-4282678";
const RIF = "J-408124637";
const apiUrl = process.env.REACT_APP_API_URL;

const PagoModal = ({ open, onClose, solicitud, detalle, onPagoRegistrado }) => {
  const { dollarPrice } = useDollar();

  // Estados principales
  const [numeroSolicitud, setNumeroSolicitud] = useState("");
  const [modalidadPago, setModalidadPago] = useState("");
  const [metodoPago, setMetodoPago] = useState("");
  const [inicial, setInicial] = useState(0);
  const [precioPlan, setPrecioPlan] = useState(0);
  const [precioConDescuento, setPrecioConDescuento] = useState(0);
  const [montoPagar, setMontoPagar] = useState(0);
  const [montoFinanciar, setMontoFinanciar] = useState(0);

  const [pagoData, setPagoData] = useState({
    montoUsd: 0,
    montoVes: 0,
    comprobanteUsd: null,
    bancoEmisorVes: "",
    referenciaPagoMovilVes: "",
    comprobanteVes: null,
    observaciones: "",
  });

  const banks = [
    { value: "", label: "Seleccione un Banco" },
    { value: "Banesco", label: "Banco Banesco Banco Universal, C.A." },
    { value: "100% Banco", label: "100% Banco Banco Universal, C.A." },
    { value: "Banca Amiga", label: "Banca Amiga, Banco Microfinanciero" },
    { value: "Bancaribe", label: "Bancaribe, C.A. Banco Universal" },
    { value: "Banco Activo", label: "Banco Activo" },
    { value: "Banco Bicentenario", label: "Banco Bicentenario" },
    { value: "Banco Caroní", label: "Banco Caroní, C.A. Banco Universal" },
    { value: "Banco de la FANB", label: "Banco de la FANB" },
    { value: "Banco de Venezuela", label: "Banco de Venezuela" },
    { value: "Banco del Tesoro", label: "Banco del Tesoro" },
    {
      value: "Banco Exterior",
      label: "Banco Exterior, C.A. Banco Universal",
    },
    {
      value: "Banco Internacional de Desarrollo",
      label: "Banco Internacional de Desarrollo",
    },
    { value: "Banco Mercantil", label: "Banco Mercantil" },
    { value: "BNC", label: "Banco Nacional de Crédito (BNC)" },
    { value: "Banco Plaza", label: "Banco Plaza, C.A. Banco Universal" },
    { value: "Banco Provincial", label: "Banco Provincial" },
    { value: "Banco Sofitasa", label: "Banco Sofitasa" },
    { value: "Bancrecer", label: "Bancrecer, S.A. Banco Microfinanciero" },
    { value: "Banplus", label: "Banplus Banco Universal, C.A." },
    { value: "Banco Fondo Común", label: "BFC Banco Fondo Común" },
    { value: "Mi Banco", label: "Mi Banco, Banco Microfinanciero, C.A." },
    {
      value: "Banco Venezolano de Credito",
      label: "Banco Venezolano de Credito",
    },
  ];

  // Cargar datos iniciales
  useEffect(() => {
    setNumeroSolicitud(solicitud?.numero_solicitud || "");
    setModalidadPago(solicitud?.modalidadPago || "");
    setMetodoPago(detalle?.metodoPago || "");
    setInicial(detalle?.pagoInicial || 0);
    setPrecioPlan(detalle?.precioPlan || 0);
    setPrecioConDescuento(detalle?.precioConDescuento || 0);
    setMontoFinanciar(detalle?.precioFinanciado || 0);
  }, [solicitud, detalle]);

  // Calcular monto a pagar
  useEffect(() => {
    let montoFinal = 0;

    if (modalidadPago === "De Contado") {
      montoFinal =
        metodoPago === "USD"
          ? Number(precioConDescuento) || 0
          : (Number(precioPlan) || 0) * (Number(dollarPrice) || 0);
    } else if (modalidadPago === "Financiado") {
      montoFinal =
        metodoPago === "USD"
          ? Number(inicial) || 0
          : (Number(inicial) || 0) * (Number(dollarPrice) || 0);
    }

    setMontoPagar(montoFinal);
  }, [
    modalidadPago,
    metodoPago,
    precioConDescuento,
    precioPlan,
    inicial,
    dollarPrice,
  ]);

  // Cambios en inputs
  const handlePagoDataChange = (e) => {
    const { name, value } = e.target;
    setPagoData((prev) => ({ ...prev, [name]: value }));
  };

  const handleComprobanteChange = (e, tipo) => {
    const file = e.target.files[0];
    if (file) setPagoData((prev) => ({ ...prev, [tipo]: file }));
  };

  // Validación
  const isFormValid = () => {
    if (metodoPago === "USD" && !pagoData.comprobanteUsd) return false;
    if (
      metodoPago === "VES" &&
      (!pagoData.bancoEmisorVes ||
        !pagoData.referenciaPagoMovilVes ||
        !pagoData.comprobanteVes)
    )
      return false;
    return true;
  };

  const formatDateToMySQL = (date) => {
    const d = new Date(date);
    return `${d.getFullYear()}-${String(d.getMonth() + 1).padStart(
      2,
      "0"
    )}-${String(d.getDate()).padStart(2, "0")} ${String(d.getHours()).padStart(
      2,
      "0"
    )}:${String(d.getMinutes()).padStart(2, "0")}:${String(
      d.getSeconds()
    ).padStart(2, "0")}`;
  };

  // Enviar pago
  const handleSubmit = async (e) => {
    e.preventDefault();
    try {
      const token = localStorage.getItem("access_token");
      if (!token) return toast.error("No hay token de autenticación.");

      const decoded = jwtDecode(token);
      const vendedorId = decoded.sub ? Number(decoded.sub) : null;
      if (!vendedorId)
        return toast.error("No se pudo identificar al vendedor.");

      const formData = new FormData();
      formData.append("numero_solicitud", solicitud.numero_solicitud);
      formData.append("cliente_id", solicitud.cliente_id);
      formData.append("vendedor_id", vendedorId);
      formData.append("solicitud_id", solicitud.id);
      formData.append("modalidadPago", modalidadPago);
      formData.append("tipo_servicio", solicitud.tipo_servicio);
      formData.append("dollarPrice", metodoPago === "VES" ? dollarPrice : 0);
      formData.append("metodoPago", metodoPago);
      if (metodoPago === "VES")
        formData.append("banco_emisor", pagoData.bancoEmisorVes || "");
      if (metodoPago === "VES")
        formData.append("banco_receptor", BANCO_RECEPTOR || "");
      formData.append(
        "referencia",
        metodoPago === "VES" ? pagoData.referenciaPagoMovilVes : 0
      );
      formData.append("fecha_pago", formatDateToMySQL(new Date()));
      formData.append("monto_usd", metodoPago === "USD" ? montoPagar : 0);
      formData.append("monto_ves", metodoPago === "VES" ? montoPagar : 0);
      formData.append("nro_cuota", 1);
      formData.append("monto_pagado", detalle.pagoInicial || "0");
      formData.append("descuento", detalle.descuento || "0");
      formData.append("monto_restante", detalle.precioFinanciado || "0");
      formData.append("monto_total", precioPlan || "0");
      formData.append(
        "estado_pago",
        modalidadPago === "Financiado" ? "inicial_pagada" : "pagado"
      );
      if (metodoPago === "USD" && pagoData.comprobanteUsd)
        formData.append("comprobante_usd", pagoData.comprobanteUsd);
      if (metodoPago === "VES" && pagoData.comprobanteVes)
        formData.append("comprobante_ves", pagoData.comprobanteVes);

      const response = await axios.post(
        `${apiUrl}/api/reporte-pagos/guardar-pago`,
        formData,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "multipart/form-data",
          },
        }
      );

      onPagoRegistrado(response.data);
      onClose();
    } catch (error) {
      console.error(error);
      toast.error(error.response?.data?.detail || "Error al registrar el pago");
    }
  };

  return (
    <Modal
      open={open}
      onClose={onClose}
      closeAfterTransition
      BackdropComponent={Backdrop}
    >
      <Box
        sx={{
          position: "absolute",
          top: "50%",
          left: "50%",
          transform: "translate(-50%, -50%)",
          width: "52%",
          bgcolor: "background.paper",
          boxShadow: 24,
          p: 4,
          borderRadius: 2,
        }}
      >
        <Box sx={{ display: "flex", justifyContent: "space-between", mb: 1 }}>
          <Typography
            sx={{ fontFamily: "Bakbak One", color: "#0A4575" }}
            variant="h6"
          >
            REPORTAR PAGO -{" "}
            <span style={{ color: "#000000" }}>#{numeroSolicitud}</span>
          </Typography>
          <Typography
            sx={{
              fontFamily: "Bakbak One",
              color: "#0A4575",
              fontSize: "20px",
            }}
          >
            Precio Total:{" "}
            <span style={{ color: "#000000" }}>{precioPlan.toFixed(2)}$</span>
          </Typography>
        </Box>

        <Grid container spacing={2} sx={{ mb: 2 }}>
          <Grid item xs={6}>
            <Typography sx={{ color: "#0A4575", fontSize: "18px" }}>
              Modalidad de Pago:{" "}
              <span style={{ color: "#000000" }}>{modalidadPago}</span>
            </Typography>
          </Grid>
          <Grid item xs={6}>
            <Typography sx={{ color: "#0A4575", fontSize: "18px" }}>
              Método de Pago:{" "}
              <span style={{ color: "#000000" }}>{metodoPago}</span>
            </Typography>
          </Grid>
        </Grid>

        <form onSubmit={handleSubmit}>
          <Grid container spacing={2}>
            {metodoPago === "USD" && (
              <>
                <Grid item xs={3}>
                  <TextField
                    fullWidth
                    label="Monto a Pagar ($)"
                    value={montoPagar.toFixed(2)}
                    InputProps={{ readOnly: true }}
                    size="small"
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={3}>
                  <input
                    type="file"
                    hidden
                    id="comprobante-usd"
                    onChange={(e) =>
                      handleComprobanteChange(e, "comprobanteUsd")
                    }
                  />
                  <label htmlFor="comprobante-usd">
                    <Button variant="contained" component="span">
                      Subir Comprobante USD
                    </Button>
                  </label>
                  {pagoData.comprobanteUsd && (
                    <Typography>{pagoData.comprobanteUsd.name}</Typography>
                  )}
                </Grid>
              </>
            )}

            {metodoPago === "VES" && (
              <>
                <Grid item xs={3}>
                  <TextField
                    fullWidth
                    label="Monto en VES"
                    value={montoPagar.toFixed(2)}
                    InputProps={{ readOnly: true }}
                    size="small"
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={3}>
                  <FormControl fullWidth size="small">
                    <InputLabel>Banco Emisor</InputLabel>
                    <Select
                      name="bancoEmisorVes"
                      value={pagoData.bancoEmisorVes}
                      onChange={handlePagoDataChange}
                      sx={{ boxShadow: 3 }}
                    >
                      {banks.map((bank) => (
                        <MenuItem key={bank.value} value={bank.value}>
                          {bank.label}
                        </MenuItem>
                      ))}
                    </Select>
                  </FormControl>
                </Grid>
                <Grid item xs={3}>
                  <TextField
                    fullWidth
                    label="Referencia"
                    name="referenciaPagoMovilVes"
                    value={pagoData.referenciaPagoMovilVes}
                    onChange={handlePagoDataChange}
                    size="small"
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={3}>
                  <input
                    type="file"
                    hidden
                    id="comprobante-ves"
                    onChange={(e) =>
                      handleComprobanteChange(e, "comprobanteVes")
                    }
                  />
                  <label htmlFor="comprobante-ves">
                    <Button variant="contained" component="span">
                      Subir Comprobante VES
                    </Button>
                  </label>
                  {pagoData.comprobanteVes && (
                    <Typography>{pagoData.comprobanteVes.name}</Typography>
                  )}
                </Grid>
                {metodoPago === "VES" && (
                  <Grid item xs={12} sm={12}>
                    <Box sx={{ display: "flex", gap: 2, alignItems: "center" }}>
                      <Typography
                        sx={{
                          fontFamily: "Bakbak One",
                          color: "#0A4575",
                          fontSize: "16px",
                        }}
                      >
                        Cuenta:{" "}
                        <span style={{ color: "#000000" }}>
                          {NUMERO_CUENTA}
                        </span>
                      </Typography>
                      <Typography
                        sx={{
                          fontFamily: "Bakbak One",
                          color: "#0A4575",
                          fontSize: "16px",
                        }}
                      >
                        {" "}
                        <span style={{ color: "#000000" }}>
                          {BANCO_RECEPTOR}
                        </span>
                      </Typography>
                      <Typography
                        sx={{
                          fontFamily: "Bakbak One",
                          color: "#0A4575",
                          fontSize: "16px",
                        }}
                      >
                        Teléfono:{" "}
                        <span style={{ color: "#000000" }}>
                          {TELEFONO_PAGO_MOVIL}
                        </span>
                      </Typography>
                      <Typography
                        sx={{
                          fontFamily: "Bakbak One",
                          color: "#0A4575",
                          fontSize: "16px",
                        }}
                      >
                        RIF: <span style={{ color: "#000000" }}>{RIF}</span>
                      </Typography>
                    </Box>
                  </Grid>
                )}
              </>
            )}
          </Grid>

          <Box sx={{ display: "flex", justifyContent: "flex-end", mt: 3 }}>
            <Button
              onClick={onClose}
              sx={{ mr: 2, background: "#cecdcdff", color: "#000" }}
            >
              Cancelar
            </Button>
            <Button type="submit" variant="contained" disabled={!isFormValid()}>
              Reportar Pago
            </Button>
          </Box>
        </form>
      </Box>
    </Modal>
  );
};

export default PagoModal;
