import React, { useEffect, useState } from "react";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Collapse,
  IconButton,
  Paper,
  TablePagination,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Dialog,
  DialogTitle,
  DialogContent,
} from "@mui/material";
import {
  ArrowBack,
  ExpandMore as ExpandMoreIcon,
  ExpandLess as ExpandLessIcon,
  Inbox as InboxIcon,
} from "@mui/icons-material";
import axios from "axios";
import { jwtDecode } from "jwt-decode";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";
import PagoModal from "./ReportePagosModal";
import { TablasEmail } from "./TablaEmail";

const apiUrl = process.env.REACT_APP_API_URL;

const ReportePagosPage = () => {
  const [solicitudes, setSolicitudes] = useState({ items: [], total: 0 });
  const [expandedSolicitud, setExpandedSolicitud] = useState(null);
  const [selectedSolicitud, setSelectedSolicitud] = useState(null);
  const [selectedDetalle, setSelectedDetalle] = useState(null);
  const [openPagoModal, setOpenPagoModal] = useState(false);
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [filter, setFilter] = useState("todos");
  const [detallesModalOpen, setDetallesModalOpen] = useState(false);
  const [reload, setReload] = useState(false);
  const [SolicitudData, setSolicitudData] = useState({
    solicitud: null,
    cliente: null,
    detalles: [],
    difunto: null,
    reporte_pagos: [],
  });

  // ---- CONSULTA PRINCIPAL ----
  useEffect(() => {
    const obtenerSolicitudes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        if (!token) return toast.error("No se encontró el token.");

        const decoded = jwtDecode(token);
        const vendedorId = decoded.sub ? Number(decoded.sub) : null;
        if (!vendedorId) return toast.error("No se pudo obtener el vendedor.");

        const config = {
          headers: { Authorization: `Bearer ${token}` },
          params: {
            page: page + 1,
            per_page: rowsPerPage,
            estado_pago: filter === "todos" ? undefined : filter,
          },
        };

        const response = await axios.get(
          `${apiUrl}/api/reporte-pagos/solicitudes/${vendedorId}`,
          config
        );
        setSolicitudes(response.data);
      } catch (error) {
        console.error("Error al obtener solicitudes:", error);
        toast.error("Error al obtener las solicitudes.");
        setSolicitudes({ items: [], total: 0 });
      }
    };

    obtenerSolicitudes();
  }, [page, rowsPerPage, filter, reload]);

  // ---- HANDLERS PRINCIPALES ----
  const handleToggleSolicitud = (id) =>
    setExpandedSolicitud(expandedSolicitud === id ? null : id);

  const handlePagoRegistrado = () => {
    toast.success("Pago registrado exitosamente");
    setReload((prev) => !prev); // esto fuerza que useEffect vuelva a ejecutarse
    setExpandedSolicitud(null); // Cierra el acordeón
  };

  const handleOpenPagoModal = (solicitud, detalle) => {
    setSelectedSolicitud(solicitud);
    setSelectedDetalle(detalle);
    setOpenPagoModal(true);
  };

  const handleClosePagoModal = () => {
    setSelectedSolicitud(null);
    setSelectedDetalle(null);
    setOpenPagoModal(false);
  };

  const mostrarDetallesSolicitud = async (solicitud, detalle) => {
    try {
      const token = localStorage.getItem("access_token");
      const decoded = jwtDecode(token);
      const vendedorId = decoded.sub ? Number(decoded.sub) : null;
      if (!vendedorId) return;

      const response = await axios.get(
        `${apiUrl}/api/v1/vendedores/detalles/${solicitud.cliente_id}/${solicitud.id}`,
        { headers: { Authorization: `Bearer ${token}` } }
      );

      if (!response.data.solicitud || !response.data.cliente)
        throw new Error("Datos incompletos del servidor");

      setSolicitudData({
        solicitud: {
          ...response.data.solicitud,
          email_enviado: detalle.email_enviado,
        },
        cliente: response.data.cliente,
        detalles: response.data.solicitud.detalles || [],
        difunto: response.data.difunto || null,
        reporte_pagos: response.data.reporte_pagos || [],
      });
      setDetallesModalOpen(true);
    } catch (error) {
      console.error("Error al obtener detalles:", error);
      toast.error("Error al obtener detalles de la solicitud.");
    }
  };

  const enviarCorreo = async () => {
    try {
      const token = localStorage.getItem("access_token");
      const decoded = jwtDecode(token);
      const user_email = decoded.email;

      if (!SolicitudData.solicitud || !SolicitudData.cliente || !user_email) {
        toast.error("Faltan datos para enviar el correo.");
        return;
      }

      const correoData = {
        solicitud: {
          ...SolicitudData.solicitud,
          documento_identidad: SolicitudData.cliente.cedula,
        },
        detalles: SolicitudData.detalles,
        cliente: {
          ...SolicitudData.cliente,
          vendedor_id: SolicitudData.solicitud.vendedor_id,
        },
        tipoServicio: SolicitudData.solicitud?.tipo_servicio || "",
        difunto: SolicitudData.difunto ? [SolicitudData.difunto] : [],
        reportePagos: SolicitudData.reporte_pagos,
        user_email,
      };

      await axios.post(`${apiUrl}/api/enviar_correo`, correoData, {
        headers: { Authorization: `Bearer ${token}` },
      });

      toast.success("Correo enviado con éxito");
      setDetallesModalOpen(false);
      setReload((prev) => !prev);
    } catch (error) {
      console.error("Error enviando correo:", error);
      toast.error("Error al enviar correo.");
    }
  };

  const handleCloseDetallesModal = () => {
    setDetallesModalOpen(false);
    setSolicitudData({
      solicitud: null,
      cliente: null,
      detalles: [],
      difunto: null,
      reporte_pagos: [],
    });
  };

  const getEstadoPagoTexto = (estado) =>
    estado === "inicial_pagada"
      ? "INICIAL PAGADA"
      : estado === "pagado"
      ? "PAGADO"
      : "PENDIENTE";

  // ---- RENDER ----
  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 2,
        boxShadow: 3,
        backgroundColor: "#fff",
        borderRadius: 2,
      }}
    >
      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          mb: 2,
        }}
      >
        <Typography
          variant="h6"
          sx={{ fontFamily: "Bakbak One", color: "#0A4575" }}
        >
          REPORTE DE PAGOS
        </Typography>
        <FormControl
          variant="outlined"
          size="small"
          sx={{ minWidth: 200, boxShadow: 3 }}
        >
          <InputLabel>Filtrar por Estado</InputLabel>
          <Select
            label="Filtrar por Estado"
            value={filter}
            onChange={(e) => setFilter(e.target.value)}
          >
            <MenuItem value="todos">Todos</MenuItem>
            <MenuItem value="pendiente">Pendiente</MenuItem>
            <MenuItem value="pagado">Pagados</MenuItem>
            <MenuItem value="inicial_pagada">Inicial Pagada</MenuItem>
          </Select>
        </FormControl>
      </Box>

      <TableContainer component={Paper} sx={{ boxShadow: 3 }}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Nº Solicitud</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cliente</TableCell>
              <TableCell sx={{ color: "#fff" }}>Servicio</TableCell>
              <TableCell sx={{ color: "#fff" }}>Fecha</TableCell>
              <TableCell sx={{ color: "#fff" }}>Modalidad</TableCell>
              <TableCell sx={{ color: "#fff" }}>Método de Pago</TableCell>
              <TableCell sx={{ color: "#fff" }}>Descuento</TableCell>
              <TableCell sx={{ color: "#fff" }}>Precio con Descuento</TableCell>
              <TableCell sx={{ color: "#fff" }}>Precio Plan</TableCell>
              <TableCell sx={{ color: "#fff" }}>Estado</TableCell>
              <TableCell sx={{ color: "#fff" }}>Acciones</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {solicitudes.items.length > 0 ? (
              solicitudes.items
                .flatMap((cliente) =>
                  cliente.solicitudes
                    .filter((s) =>
                      filter === "todos" ? true : s.estadoPago === filter
                    )
                    .map((s) => ({ cliente, solicitud: s }))
                )
                .map(({ cliente, solicitud }) => (
                  <React.Fragment key={solicitud.id}>
                    <TableRow>
                      <TableCell>{solicitud.numero_solicitud}</TableCell>
                      <TableCell>{cliente.nombre}</TableCell>
                      <TableCell>{solicitud.tipo_servicio}</TableCell>
                      <TableCell>
                        {new Date(solicitud.fecha_creacion).toLocaleDateString(
                          "es-ES"
                        )}
                      </TableCell>
                      <TableCell>{solicitud.modalidadPago}</TableCell>
                      <TableCell>
                        {solicitud.items_adicionales?.metodoPago}
                      </TableCell>
                      <TableCell>
                        ${solicitud.items_adicionales?.descuento}
                      </TableCell>
                      <TableCell>
                        $
                        {solicitud.items_adicionales?.precioConDescuento.toFixed(
                          2
                        )}
                      </TableCell>
                      <TableCell>${solicitud.precioPlan.toFixed(2)}</TableCell>
                      <TableCell>
                        <Button
                          variant="contained"
                          sx={{
                            backgroundColor:
                              solicitud.estadoPago === "pagado" ||
                              solicitud.estadoPago === "inicial_pagada"
                                ? "#4CAF50"
                                : "#F44336",
                            color: "#fff",
                            pointerEvents: "none",
                          }}
                        >
                          {getEstadoPagoTexto(solicitud.estadoPago)}
                        </Button>
                      </TableCell>
                      <TableCell>
                        <IconButton
                          onClick={() => handleToggleSolicitud(solicitud.id)}
                        >
                          {expandedSolicitud === solicitud.id ? (
                            <ExpandLessIcon />
                          ) : (
                            <ExpandMoreIcon />
                          )}
                        </IconButton>
                      </TableCell>
                    </TableRow>
                    <TableRow>
                      <TableCell colSpan={11} sx={{ p: 0 }}>
                        <Collapse
                          in={expandedSolicitud === solicitud.id}
                          timeout="auto"
                          unmountOnExit
                        >
                          <Box sx={{ p: 2 }}>
                            <Typography
                              variant="h6"
                              sx={{
                                fontFamily: "Bakbak One",
                                color: "#0A4575",
                              }}
                            >
                              DETALLES DE SOLICITUD
                            </Typography>
                            <Table size="small">
                              <TableHead sx={{ backgroundColor: "#0A4575" }}>
                                <TableRow>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Código
                                  </TableCell>
                                  <TableCell
                                    sx={{ color: "#fff", width: "15%" }}
                                  >
                                    Descripción
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Cuotas
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Inicial
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Monto/Cuota
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Precio Financiado
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Precio con Descuento
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Acción
                                  </TableCell>
                                </TableRow>
                              </TableHead>
                              <TableBody>
                                {solicitud.detalles.map((d) => (
                                  <TableRow key={d.id}>
                                    <TableCell>{d.codigo}</TableCell>
                                    <TableCell>
                                      {d.descripcion || solicitud.descripcion}
                                    </TableCell>
                                    <TableCell>{d.cuotas}</TableCell>
                                    <TableCell>
                                      ${d.pagoInicial.toFixed(2)}
                                    </TableCell>
                                    <TableCell>
                                      ${d.montoPorCuotas.toFixed(2)}
                                    </TableCell>
                                    <TableCell>
                                      ${d.precioFinanciado.toFixed(2)}
                                    </TableCell>
                                    <TableCell>
                                      ${d.precioConDescuento.toFixed(2)}
                                    </TableCell>
                                    <TableCell>
                                      <Button
                                        onClick={() =>
                                          handleOpenPagoModal(solicitud, d)
                                        }
                                        disabled={
                                          d.estadoPago === "inicial_pagada" ||
                                          d.estadoPago === "pagado"
                                        }
                                        sx={{
                                          backgroundColor:
                                            d.estadoPago === "inicial_pagada" ||
                                            d.estadoPago === "pagado"
                                              ? "#B0BEC5"
                                              : "#0A4575",
                                          color: "#fff",
                                          "&:hover": {
                                            backgroundColor:
                                              d.estadoPago ===
                                                "inicial_pagada" ||
                                              d.estadoPago === "pagado"
                                                ? "#B0BEC5"
                                                : "#083a5e",
                                          },
                                        }}
                                      >
                                        Reportar Pago
                                      </Button>
                                      <Button
                                        sx={{
                                          ml: 2,
                                          backgroundColor:
                                            d.email_enviado ||
                                            d.estadoPago === "pendiente"
                                              ? "#B0BEC5"
                                              : "#0A4575",
                                          color: "#fff",
                                          "&:hover": {
                                            backgroundColor:
                                              d.email_enviado ||
                                              d.estadoPago === "pendiente"
                                                ? "#B0BEC5"
                                                : "#083a5e",
                                          },
                                        }}
                                        onClick={() =>
                                          mostrarDetallesSolicitud(solicitud, d)
                                        }
                                        disabled={
                                          d.email_enviado ||
                                          d.estadoPago === "pendiente"
                                        }
                                      >
                                        Enviar Solicitud
                                      </Button>
                                    </TableCell>
                                  </TableRow>
                                ))}
                              </TableBody>
                            </Table>
                          </Box>
                        </Collapse>
                      </TableCell>
                    </TableRow>
                  </React.Fragment>
                ))
            ) : (
              <TableRow>
                <TableCell colSpan={11}>
                  <Box
                    sx={{
                      display: "flex",
                      flexDirection: "column",
                      alignItems: "center",
                      py: 2,
                    }}
                  >
                    <InboxIcon sx={{ fontSize: 32, color: "#0A4575", mb: 1 }} />
                    <Typography>No hay reporte de pagos disponible.</Typography>
                  </Box>
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
        <TablePagination
          rowsPerPageOptions={[5, 10, 25]}
          component="div"
          count={
            filter === "todos"
              ? solicitudes.total
              : solicitudes.items.flatMap((c) =>
                  c.solicitudes.filter((s) =>
                    filter === "todos" ? true : s.estadoPago === filter
                  )
                ).length
          }
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={(e, newPage) => setPage(newPage)}
          onRowsPerPageChange={(e) => {
            setRowsPerPage(parseInt(e.target.value, 10));
            setPage(0);
          }}
          labelRowsPerPage="Filas por página:"
        />
      </TableContainer>

      {/* MODAL DETALLES + EMAIL */}
      <Dialog
        open={detallesModalOpen}
        onClose={handleCloseDetallesModal}
        fullWidth
        maxWidth={false}
        PaperProps={{
          sx: { maxWidth: "1300px" },
        }}
      >
        <DialogTitle
          sx={{
            textAlign: "center",
            fontFamily: "Bakbak One",
            color: "#0A4575",
          }}
        >
          DETALLES DE LA SOLICITUD
        </DialogTitle>
        <DialogContent>
          <TablasEmail
            tipoServicio={SolicitudData?.solicitud?.tipo_servicio}
            cliente={SolicitudData?.cliente}
            solicitud={SolicitudData?.solicitud}
            detalles={SolicitudData?.detalles}
            difunto={SolicitudData?.difunto}
            reportePagos={SolicitudData?.reporte_pagos}
          />
          <Box sx={{ display: "flex", justifyContent: "flex-end", mt: 2 }}>
            <Button
              variant="contained"
              onClick={enviarCorreo}
              disabled={SolicitudData?.solicitud?.email_enviado}
            >
              Enviar por Correo Electrónico
            </Button>
          </Box>
        </DialogContent>
      </Dialog>

      {/* MODAL PAGO */}
      <PagoModal
        open={openPagoModal}
        onClose={handleClosePagoModal}
        solicitud={selectedSolicitud}
        detalle={selectedDetalle}
        onPagoRegistrado={handlePagoRegistrado}
      />

      <Button
        variant="contained"
        component={Link}
        to="/ventas"
        startIcon={<ArrowBack />}
        sx={{ mt: 2, backgroundColor: "#0A4575", color: "#fff" }}
      >
        Volver
      </Button>
    </Box>
  );
};

export default ReportePagosPage;
