import React, { useState, useEffect } from "react";
import axios from "axios";
import {
  Box,
  Button,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Grid,
  Checkbox,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
} from "@mui/material";
import { ArrowBack } from "@mui/icons-material";
import { toast } from "react-toastify";
import "react-toastify/dist/ReactToastify.css";
import { Link } from "react-router-dom";
import { jwtDecode } from "jwt-decode";

const ServicioFunerario = () => {
  const [funerariosItems, setFunerariosItems] = useState([]);
  const [funerariosItemsSeleccionados, setFunerariosItemsSeleccionados] =
    useState([]);
  const [tipoSala, setTipoSala] = useState("");
  const [planSeleccionado, setPlanSeleccionado] = useState(null);
  const [mostrarTraslado, setMostrarTraslado] = useState(true);
  const [mostrarPreparacion, setMostrarPreparacion] = useState(true);
  const [mostrarAtaud, setMostrarAtaud] = useState(true);
  const [precioVelacion, setPrecioVelacion] = useState("");
  const [precioTraslado, setPrecioTraslado] = useState("");
  const [precioPreparacion, setPrecioPreparacion] = useState("");
  const [precioAtaud, setPrecioAtaud] = useState("");
  const [selectCliente, setSelectCliente] = useState("");
  const [clienteSeleccionado, setClienteSeleccionado] = useState(null);
  const [incluirDifunto, setIncluirDifunto] = useState("no"); // Nuevo estado para controlar si se incluye difunto
  const apiUrl = process.env.REACT_APP_API_URL;

  // Estado para almacenar los planes desde el backend
  const [planesFunerarios, setPlanesFunerarios] = useState({
    sencilla: [],
    vip: [],
  });

  const [cliente, setCliente] = useState({
    nombre: "",
    cedula: "",
    fecha_nacimiento: "",
    edad: "",
    telefono: "",
    celular: "",
    email: "",
    direccion: "",
    nombre_documento: "",
    documento_identidad: null,
  });

  const [difunto, setDifunto] = useState({
    nombre_difunto: "",
    cedula_difunto: "",
    fecha_nac_difunto: "",
    edad_difunto: "",
    parentesco_difunto: "",
    fecha_defuncion: "",
    causa_defuncion: "",
    direccion_defuncion: "",
  });

  // Obtener los planes desde el backend
  useEffect(() => {
    const fetchPlanesFunerarios = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const response = await axios.get(`${apiUrl}/api/planes/funerarios`, {
          headers: { Authorization: `Bearer ${token}` },
        });

        // Adaptar los datos recibidos a la estructura que espera el componente
        const planes = response.data;

        // Filtrar planes vacíos
        const planesValidos = planes.filter(
          (plan) => plan.tipo_servicio && plan.precio_velacion > 0
        );

        const sencilla = planesValidos
          .filter((plan) => plan.tipo_capilla.includes("salas del crematorio"))
          .map((plan) => ({
            id: plan.id,
            tipo: plan.tipo_capilla,
            precio_traslado: plan.precio_traslado,
            velacion: `${plan.tiempo_velacion} HORAS`,
            precio_v: plan.precio_velacion,
            precio_d: plan.preparacion_difunto_sencilla,
            precio_a: plan.alquiler_ataud_sencillo,
            precio: 0,
          }));

        const vip = planesValidos
          .filter((plan) => plan.tipo_capilla.includes("sala VIP en funeraria"))
          .map((plan) => ({
            id: plan.id,
            tipo: plan.tipo_capilla,
            precio_traslado: plan.precio_traslado,
            velacion: `${plan.tiempo_velacion} HORAS`,
            precio_v: plan.precio_velacion,
            precio_d: plan.preparacion_difunto_sencilla,
            precio_a: plan.alquiler_ataud_sencillo,
            precio: 0,
          }));

        setPlanesFunerarios({
          sencilla,
          vip,
        });
      } catch (error) {
        console.error("Error al obtener los planes funerarios:", error);
        toast.error("Error al cargar los planes funerarios.");
      }
    };
    fetchPlanesFunerarios();
  }, [apiUrl]);

  const handleClienteChange = (e) => {
    const { name, value } = e.target;
    let updatedFields = { [name]: value };
    if (name === "fecha_nacimiento") {
      const edadCalculada = calcularEdad(value);
      updatedFields.edad = edadCalculada.toString();
    }
    setCliente((prev) => ({
      ...prev,
      ...updatedFields,
    }));
  };

  const calcularEdad = (fechaNacimiento) => {
    const hoy = new Date();
    const fecha_nacimiento = new Date(fechaNacimiento);
    let edad = hoy.getFullYear() - fecha_nacimiento.getFullYear();
    const mes = hoy.getMonth() - fecha_nacimiento.getMonth();
    if (mes < 0 || (mes === 0 && hoy.getDate() < fecha_nacimiento.getDate())) {
      edad--;
    }
    return edad;
  };

  const handleFileChange = (event) => {
    const file = event.target.files[0];
    if (file) {
      setCliente((prev) => ({
        ...prev,
        documento_identidad: file,
        nombre_documento: file.name,
      }));
    }
  };

  const handleTipoSalaChange = (e) => {
    setTipoSala(e.target.value);
    setPlanSeleccionado(null);
    setPrecioVelacion("");
    setPrecioTraslado("");
    setPrecioPreparacion("");
    setPrecioAtaud("");
  };

  const handlePlanChange = (e) => {
    const plan = planesFunerarios[tipoSala].find(
      (p) => p.id === e.target.value
    );
    setPlanSeleccionado(plan);

    const precioV = plan.precio_v;
    const precioTraslado = plan.precio_traslado;
    const precioD = plan.precio_d;
    const precioA = plan.precio_a;

    setPrecioVelacion(precioV.toString() || "");
    setPrecioTraslado(precioTraslado.toString() || "");
    setPrecioPreparacion(precioD.toString() || "");
    setPrecioAtaud(precioA.toString() || "");
  };

  const totalForm = () => {
    let total = 0;
    total += parseFloat(precioVelacion) || 0;
    if (mostrarTraslado) total += parseFloat(precioTraslado) || 0;
    if (mostrarPreparacion) total += parseFloat(precioPreparacion) || 0;
    if (mostrarAtaud) total += parseFloat(precioAtaud) || 0;
    return total;
  };

  const precioTotalContrato = funerariosItems.reduce((total, item) => {
    let subtotal = item.precioVelacion || 0;
    if (mostrarTraslado) subtotal += item.traslado || 0;
    if (mostrarPreparacion) subtotal += item.preparacion || 0;
    if (mostrarAtaud) subtotal += item.alquiler || 0;
    return total + subtotal;
  }, 0);

  const agregarContrato = () => {
    if (planSeleccionado) {
      const precioBaseVelacion = parseFloat(precioVelacion) || 0;
      const precioBaseTraslado = parseFloat(precioTraslado) || 0;
      const precioBasePreparacion = parseFloat(precioPreparacion) || 0;
      const precioBaseAtaud = parseFloat(precioAtaud) || 0;
      const precioTotal =
        precioBaseVelacion +
        (mostrarTraslado ? precioBaseTraslado : 0) +
        (mostrarPreparacion ? precioBasePreparacion : 0) +
        (mostrarAtaud ? precioBaseAtaud : 0);

      const item = {
        descripcion: planSeleccionado.tipo,
        traslado: precioBaseTraslado,
        velacion: planSeleccionado.velacion,
        precioVelacion: precioBaseVelacion,
        preparacion: precioBasePreparacion,
        alquiler: precioBaseAtaud,
        precioTotal: precioTotal,
      };

      setFunerariosItems((prevItems) => [...prevItems, item]);
      setTipoSala("");
      setPrecioVelacion("");
      setPrecioTraslado("");
      setPrecioPreparacion("");
      setPrecioAtaud("");
      setPlanSeleccionado(null);
    }
  };

  const handleSeleccionarFunerariosItem = (index) => {
    const seleccionado = funerariosItemsSeleccionados.includes(index);
    if (seleccionado) {
      setFunerariosItemsSeleccionados(
        funerariosItemsSeleccionados.filter((i) => i !== index)
      );
    } else {
      setFunerariosItemsSeleccionados([...funerariosItemsSeleccionados, index]);
    }
  };

  const handleEliminarFunerariosItems = () => {
    const nuevosFunerariosItems = funerariosItems.filter(
      (_, index) => !funerariosItemsSeleccionados.includes(index)
    );
    setFunerariosItems(nuevosFunerariosItems);
    setFunerariosItemsSeleccionados([]);
  };

  const handleDifuntoChange = (e) => {
    const { name, value } = e.target;
    let updatedFields = { [name]: value };
    if (name === "fecha_nac_difunto") {
      const edadCalculada = calcularEdad(value);
      updatedFields.edad_difunto = edadCalculada.toString();
    }
    setDifunto((prev) => ({
      ...prev,
      ...updatedFields,
    }));
  };

  const handleIncluirDifuntoChange = (e) => {
    setIncluirDifunto(e.target.value);
    if (e.target.value === "no") {
      setDifunto({
        nombre_difunto: "",
        cedula_difunto: "",
        fecha_nac_difunto: "",
        edad_difunto: "",
        parentesco_difunto: "",
        fecha_defuncion: "",
        causa_defuncion: "",
        direccion_defuncion: "",
      });
    }
  };

  const buscarCliente = async (cedula) => {
    if (!cedula.trim()) return;
    try {
      const token = localStorage.getItem("access_token");
      const config = { headers: { Authorization: `Bearer ${token}` } };
      const response = await axios.get(
        `${apiUrl}/api/v1/clientes/${cedula}`,
        config
      );
      const clienteEncontrado = response.data;
      toast.success("El cliente ya está registrado. Se asociará el contrato.");
      setClienteSeleccionado(clienteEncontrado);
      setCliente({
        nombre: clienteEncontrado.nombre,
        cedula: clienteEncontrado.cedula,
        fecha_nacimiento: clienteEncontrado.fecha_nacimiento
          ? new Date(clienteEncontrado.fecha_nacimiento)
              .toISOString()
              .split("T")[0]
          : "",
        edad: clienteEncontrado.edad,
        telefono: clienteEncontrado.telefono,
        celular: clienteEncontrado.celular,
        email: clienteEncontrado.email,
        direccion: clienteEncontrado.direccion,
        documento_identidad: null,
        nombre_documento: "",
      });
    } catch (error) {
      if (error.response?.status === 404) {
        toast.info("Cliente no encontrado. Agregue un nuevo cliente.");
      } else {
        console.error("Error al buscar cliente:", error);
        toast.error("Error al buscar cliente.");
      }
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    const token = localStorage.getItem("access_token");
    const decoded = jwtDecode(token);
    const vendedorId = decoded.sub;

    let clienteId;

    try {
      if (selectCliente === "existente" || clienteSeleccionado) {
        if (!clienteSeleccionado || !clienteSeleccionado.id) {
          toast.error("Debe seleccionar un cliente válido.");
          return;
        }
        clienteId = clienteSeleccionado.id;
      } else if (selectCliente === "nuevo") {
        if (
          !cliente.nombre ||
          !cliente.cedula ||
          !cliente.fecha_nacimiento ||
          !cliente.telefono ||
          !cliente.email ||
          !cliente.direccion
        ) {
          toast.error(
            "Por favor, complete todos los campos obligatorios del cliente."
          );
          return;
        }

        const formData = new FormData();
        Object.keys(cliente).forEach((key) => {
          if (key === "documento_identidad" && cliente[key] instanceof File) {
            formData.append(key, cliente[key], cliente[key].name);
          } else {
            formData.append(key, cliente[key]);
          }
        });

        const response = await axios.post(
          `${apiUrl}/api/v1/clientes`,
          formData,
          {
            headers: {
              Authorization: `Bearer ${token}`,
              "Content-Type": "multipart/form-data",
            },
          }
        );
        clienteId = response.data.id;
      }

      const config = {
        headers: {
          Authorization: `Bearer ${token}`,
          "Content-Type": "application/json",
        },
      };

      const contratoData = {
        cliente_id: clienteId,
        vendedor_id: vendedorId,
        tipo_servicio: "Servicio Funerario",
        fecha_creacion: new Date().toISOString(),
        fecha_vencimiento: new Date(
          new Date().setDate(new Date().getDate() + 3)
        ).toISOString(),
        monto_total: precioTotalContrato,
        detalles: funerariosItems.map((item) => ({
          tipo_contrato: "Servicio Funerario",
          codigo: "FUN-00000",
          descripcion: item.descripcion,
          modalidadPago: "De Contado",
          pagoInicial: 0,
          cuotas: 0,
          montoPorCuota: 0,
          precioTotal: item.precioTotal,
          detalles_adicionales: {
            traslado: item.traslado,
            tiempo_velacion: item.velacion,
            precio_velacion: item.precioVelacion,
            incluye_preparacion: item.preparacion,
            incluye_ataud: item.alquiler,
          },
        })),
        difuntos:
          incluirDifunto === "sí"
            ? [
                {
                  nombre_difunto: difunto.nombre_difunto || null,
                  cedula_difunto: difunto.cedula_difunto || null,
                  fecha_nac_difunto: difunto.fecha_nac_difunto || null,
                  edad_difunto: difunto.edad_difunto || null,
                  parentesco_difunto: difunto.parentesco_difunto || null,
                  fecha_defuncion: difunto.fecha_defuncion || null,
                  causa_defuncion: difunto.causa_defuncion || null,
                  direccion_defuncion: difunto.direccion_defuncion || null,
                },
              ]
            : [],
      };

      try {
        const contratoResponse = await axios.post(
          `${apiUrl}/api/v1/solicitudes`,
          [contratoData],
          config
        );
        const contratoId = contratoResponse.data[0].id;

        toast.success("Contrato registrado exitosamente");

        setTimeout(() => {
          setCliente({
            nombre: "",
            cedula: "",
            fecha_nacimiento: "",
            edad: "",
            telefono: "",
            celular: "",
            email: "",
            direccion: "",
            documento_identidad: null,
          });
          setFunerariosItems([]);
          setFunerariosItemsSeleccionados([]);
          setTipoSala("");
          setDifunto({
            nombre_difunto: "",
            cedula_difunto: "",
            fecha_nac_difunto: "",
            edad_difunto: "",
            parentesco_difunto: "",
            fecha_defuncion: "",
            causa_defuncion: "",
            direccion_defuncion: "",
          });
          setIncluirDifunto("no");
          setPrecioVelacion("");
          setPrecioTraslado("");
          setPrecioPreparacion("");
          setPrecioAtaud("");
          setPlanSeleccionado(null);
        }, 3000);
      } catch (error) {
        toast.error(`Error: ${error.response?.data?.detail || error.message}`);
        throw error;
      }
    } catch (error) {
      toast.error(`Error: ${error.response?.data?.detail || error.message}`);
    }
  };

  return (
    <Box
      component="form"
      onSubmit={handleSubmit}
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        sx={{
          marginBottom: "5px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
          textAlign: "center",
        }}
        variant="h5"
      >
        SOLICITUD DE SERVICIO FUNERARIO
      </Typography>

      <Typography
        sx={{
          marginTop: "15px",
          marginBottom: "10px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
      >
        PLAN A SOLICITAR
      </Typography>

      <Grid container spacing={3} alignItems="center">
        <Grid item xs={12} sm={3}>
          <FormControl
            fullWidth
            size="small"
            variant="outlined"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel>Tipo de Capilla</InputLabel>
            <Select
              value={tipoSala}
              label="Tipo de Capilla"
              onChange={handleTipoSalaChange}
            >
              <MenuItem value="" disabled selected>
                Selecciona el tipo de Capilla
              </MenuItem>
              <MenuItem value="sencilla">Sala Edificio Crematorio</MenuItem>
              <MenuItem value="vip">Sala VIP en Funeraria</MenuItem>
            </Select>
          </FormControl>
        </Grid>

        <Grid item xs={12} sm={2}>
          <FormControl
            fullWidth
            size="small"
            variant="outlined"
            sx={{ boxShadow: 3 }}
            disabled={!tipoSala}
          >
            <InputLabel>Tiempo de Velación</InputLabel>
            <Select
              value={planSeleccionado ? planSeleccionado.id : ""}
              label="Tiempo de Velación"
              onChange={handlePlanChange}
            >
              <MenuItem value="" disabled>
                Selecciona un Plan
              </MenuItem>
              {tipoSala &&
                planesFunerarios[tipoSala].map((plan) => (
                  <MenuItem key={plan.id} value={plan.id}>
                    {plan.velacion}
                  </MenuItem>
                ))}
            </Select>
          </FormControl>
        </Grid>

        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label={`Precio Velación`}
            variant="outlined"
            size="small"
            value={precioVelacion ? parseFloat(precioVelacion).toFixed(2) : ""}
            onChange={(e) => setPrecioVelacion(e.target.value)}
            sx={{ boxShadow: 3 }}
          />
        </Grid>

        {mostrarTraslado && (
          <Grid item xs={12} sm={2}>
            <TextField
              fullWidth
              label={`Precio Traslado`}
              variant="outlined"
              size="small"
              value={
                precioTraslado ? parseFloat(precioTraslado).toFixed(2) : ""
              }
              onChange={(e) => setPrecioTraslado(e.target.value)}
              sx={{ boxShadow: 3 }}
            />
          </Grid>
        )}

        {mostrarPreparacion && (
          <Grid item xs={12} sm={3}>
            <TextField
              fullWidth
              label={`Preparación del Difunto`}
              variant="outlined"
              size="small"
              value={
                precioPreparacion
                  ? parseFloat(precioPreparacion).toFixed(2)
                  : ""
              }
              onChange={(e) => setPrecioPreparacion(e.target.value)}
              sx={{ boxShadow: 3 }}
            />
          </Grid>
        )}

        {mostrarAtaud && (
          <Grid item xs={12} sm={2}>
            <TextField
              fullWidth
              label={`Alquiler del Ataud`}
              variant="outlined"
              size="small"
              value={precioAtaud ? parseFloat(precioAtaud).toFixed(2) : ""}
              onChange={(e) => setPrecioAtaud(e.target.value)}
              sx={{ boxShadow: 3 }}
            />
          </Grid>
        )}

        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label={`Precio Total`}
            variant="outlined"
            size="small"
            value={planSeleccionado ? totalForm().toFixed(2) : ""}
            InputProps={{ readOnly: true }}
            sx={{ boxShadow: 3 }}
          />
        </Grid>

        <Grid item xs={12} sm={3}>
          <Button
            variant="contained"
            color="primary"
            onClick={agregarContrato}
            disabled={!planSeleccionado}
            sx={{
              backgroundColor: "#0A4575",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Añadir Contrato
          </Button>
        </Grid>
      </Grid>

      <Grid container spacing={2} sx={{ mt: 1 }}>
        <Grid item xs={3}>
          <Checkbox
            checked={mostrarTraslado}
            onChange={(e) => setMostrarTraslado(e.target.checked)}
            sx={{ color: "#0A4575", "&.Mui-checked": { color: "#0A4575" } }}
          />
          <Typography
            variant="body1"
            sx={{ display: "inline", fontFamily: "Abel" }}
          >
            Servicio de Traslado
          </Typography>
        </Grid>
        <Grid item xs={3}>
          <Checkbox
            checked={mostrarPreparacion}
            onChange={(e) => setMostrarPreparacion(e.target.checked)}
            sx={{ color: "#0A4575", "&.Mui-checked": { color: "#0A4575" } }}
          />
          <Typography
            variant="body1"
            sx={{ display: "inline", fontFamily: "Abel" }}
          >
            Preparación del Difunto
          </Typography>
        </Grid>
        <Grid item xs={3}>
          <Checkbox
            checked={mostrarAtaud}
            onChange={(e) => setMostrarAtaud(e.target.checked)}
            sx={{ color: "#0A4575", "&.Mui-checked": { color: "#0A4575" } }}
          />
          <Typography
            variant="body1"
            sx={{ display: "inline", fontFamily: "Abel" }}
          >
            Alquiler de Ataúd
          </Typography>
        </Grid>
      </Grid>

      <Typography
        sx={{
          marginTop: "25px",
          marginBottom: "7px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
      >
        DATOS DEL CLIENTE
      </Typography>

      <Grid container spacing={3}>
        <Grid item xs={12} sm={4}>
          <TextField
            fullWidth
            label="Nombre cliente"
            variant="outlined"
            size="small"
            name="nombre"
            value={cliente.nombre}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Cédula o Rif"
            variant="outlined"
            size="small"
            name="cedula"
            value={cliente.cedula}
            onChange={handleClienteChange}
            onBlur={(e) => buscarCliente(e.target.value)}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Fecha de Nacimiento"
            variant="outlined"
            type="date"
            size="small"
            name="fecha_nacimiento"
            value={cliente.fecha_nacimiento || ""}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
            InputLabelProps={{ shrink: true }}
          />
        </Grid>
        <Grid item xs={12} sm={2}>
          <TextField
            fullWidth
            label="Edad"
            variant="outlined"
            size="small"
            name="edad"
            value={cliente.edad}
            required
            sx={{ boxShadow: 3 }}
            InputProps={{ readOnly: true }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Teléfono"
            variant="outlined"
            size="small"
            name="telefono"
            value={cliente.telefono}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3}>
          <TextField
            fullWidth
            label="Celular"
            variant="outlined"
            size="small"
            name="celular"
            value={cliente.celular}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={6}>
          <TextField
            fullWidth
            label="Email"
            variant="outlined"
            size="small"
            name="email"
            value={cliente.email}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={5}>
          <TextField
            fullWidth
            label="Dirección"
            variant="outlined"
            size="small"
            name="direccion"
            value={cliente.direccion}
            onChange={handleClienteChange}
            required
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={4}>
          <input
            accept="image/*"
            id="file-upload"
            type="file"
            style={{ height: "1px", width: "1px", position: "absolute" }}
            onChange={handleFileChange}
          />
          <label htmlFor="file-upload">
            <Button
              variant="contained"
              component="span"
              sx={{
                boxShadow: 3,
                backgroundColor: "#0A4575",
                color: "#FFFFFF",
                "&:hover": { backgroundColor: "#083a5e" },
              }}
            >
              Subir Documento de Identidad
            </Button>
          </label>
        </Grid>
        <Grid item xs={12} sm={2}>
          {cliente.nombre_documento && (
            <Typography sx={{ marginTop: "10px", fontFamily: "Abel" }}>
              <b>{cliente.nombre_documento}</b>
            </Typography>
          )}
        </Grid>
      </Grid>

      <Typography
        sx={{
          marginTop: "25px",
          marginBottom: "7px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
      >
        DATOS DEL DIFUNTO
      </Typography>

      <Grid container spacing={3}>
        <Grid item xs={12} sm={3}>
          <FormControl
            fullWidth
            size="small"
            variant="outlined"
            sx={{ boxShadow: 3 }}
          >
            <InputLabel>Incluir datos del difunto</InputLabel>
            <Select
              value={incluirDifunto}
              label="Incluir datos del difunto"
              onChange={handleIncluirDifuntoChange}
            >
              <MenuItem value="no">No</MenuItem>
              <MenuItem value="sí">Sí</MenuItem>
            </Select>
          </FormControl>
        </Grid>
      </Grid>

      {incluirDifunto === "sí" && (
        <>
          <Grid container spacing={3} sx={{ mt: 1 }}>
            <Grid item xs={12} sm={3}>
              <TextField
                fullWidth
                label="Nombre"
                variant="outlined"
                size="small"
                name="nombre_difunto"
                value={difunto.nombre_difunto || ""}
                onChange={handleDifuntoChange}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={12} sm={2}>
              <TextField
                fullWidth
                label="Cédula"
                variant="outlined"
                size="small"
                name="cedula_difunto"
                value={difunto.cedula_difunto || ""}
                onChange={handleDifuntoChange}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={12} sm={3}>
              <TextField
                fullWidth
                label="Fecha de Nacimiento"
                variant="outlined"
                type="date"
                size="small"
                name="fecha_nac_difunto"
                value={difunto.fecha_nac_difunto || ""}
                onChange={handleDifuntoChange}
                sx={{ boxShadow: 3 }}
                InputLabelProps={{ shrink: true }}
              />
            </Grid>
            <Grid item xs={12} sm={2}>
              <TextField
                fullWidth
                label="Edad"
                variant="outlined"
                size="small"
                name="edad_difunto"
                value={difunto.edad_difunto || ""}
                onChange={handleDifuntoChange}
                sx={{ boxShadow: 3 }}
                InputProps={{ readOnly: true }}
              />
            </Grid>
            <Grid item xs={12} sm={2}>
              <FormControl
                fullWidth
                variant="outlined"
                size="small"
                sx={{ boxShadow: 3 }}
              >
                <InputLabel id="parentesco_difunto">Parentesco</InputLabel>
                <Select
                  labelId="parentesco_difunto"
                  name="parentesco_difunto"
                  value={difunto.parentesco_difunto || ""}
                  onChange={handleDifuntoChange}
                  label="Parentesco"
                >
                  <MenuItem value="" disabled>
                    Seleccione una opción
                  </MenuItem>
                  <MenuItem value="Padre/Madre">Padre/Madre</MenuItem>
                  <MenuItem value="Hijo(a)">Hijo(a)</MenuItem>
                  <MenuItem value="Esposo(a)/Conyugue">
                    Esposo(a)/Conyugue
                  </MenuItem>
                  <MenuItem value="Hermano(a)">Hermano(a)</MenuItem>
                  <MenuItem value="Abuelo(a)">Abuelo(a)</MenuItem>
                  <MenuItem value="Tío(a)">Tío(a)</MenuItem>
                  <MenuItem value="Primo(a)">Primo(a)</MenuItem>
                  <MenuItem value="Otro(a)">Otro(a)</MenuItem>
                </Select>
              </FormControl>
            </Grid>
          </Grid>

          <Typography
            sx={{
              marginTop: "25px",
              marginBottom: "7px",
              fontFamily: "Bakbak One, sans-serif",
              color: "#0A4575",
            }}
            variant="h6"
          >
            DATOS DE DIFUNCIÓN
          </Typography>

          <Grid container spacing={3}>
            <Grid item xs={12} sm={3}>
              <TextField
                fullWidth
                label="Fecha de Defunción"
                variant="outlined"
                type="date"
                size="small"
                name="fecha_defuncion"
                value={difunto.fecha_defuncion || ""}
                onChange={handleDifuntoChange}
                sx={{ boxShadow: 3 }}
                InputLabelProps={{ shrink: true }}
              />
            </Grid>
            <Grid item xs={12} sm={4}>
              <TextField
                fullWidth
                label="Causa de Defunción"
                variant="outlined"
                size="small"
                name="causa_defuncion"
                value={difunto.causa_defuncion || ""}
                onChange={handleDifuntoChange}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
            <Grid item xs={12} sm={4}>
              <TextField
                fullWidth
                label="Dirección de Defunción"
                variant="outlined"
                size="small"
                name="direccion_defuncion"
                value={difunto.direccion_defuncion || ""}
                onChange={handleDifuntoChange}
                sx={{ boxShadow: 3 }}
              />
            </Grid>
          </Grid>
        </>
      )}

      <Typography
        sx={{
          textAlign: "center",
          marginTop: "25px",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
        gutterBottom
      >
        RESUMEN DE SERVICIOS FUNERARIOS
      </Typography>

      <TableContainer component={Paper}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575", color: "#FFFFFF" }}>
            <TableRow>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Tipo de Capilla
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Traslado
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Horas Velación
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Precio Velación
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Preparación Difunto
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Alquiler Ataud
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Total
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Seleccionar
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {funerariosItems.map((item, index) => (
              <TableRow key={index}>
                <TableCell>{item.descripcion}</TableCell>
                <TableCell>${item.traslado.toFixed(2)}</TableCell>
                <TableCell>{item.velacion}</TableCell>
                <TableCell>${item.precioVelacion.toFixed(2)}</TableCell>
                <TableCell>${item.preparacion.toFixed(2)}</TableCell>
                <TableCell>${item.alquiler.toFixed(2)}</TableCell>
                <TableCell>${item.precioTotal.toFixed(2)}</TableCell>
                <TableCell>
                  <Checkbox
                    checked={funerariosItemsSeleccionados.includes(index)}
                    onChange={() => handleSeleccionarFunerariosItem(index)}
                    sx={{
                      color: "#0A4575",
                      "&.Mui-checked": { color: "#0A4575" },
                    }}
                  />
                </TableCell>
              </TableRow>
            ))}
            <TableRow>
              <TableCell colSpan={6} sx={{ textAlign: "right" }}>
                <Typography
                  component="div"
                  sx={{
                    display: "flex",
                    justifyContent: "flex-end",
                    fontFamily: "Bakbak One, sans-serif",
                    color: "#0A4575",
                    fontSize: "18px",
                  }}
                >
                  Total:
                </Typography>
              </TableCell>
              <TableCell>
                <Typography sx={{ fontWeight: "bold", color: "#000000" }}>
                  ${precioTotalContrato.toFixed(2)}
                </Typography>
              </TableCell>
              <TableCell />
            </TableRow>
          </TableBody>
        </Table>
      </TableContainer>

      <Box
        sx={{
          display: "flex",
          justifyContent: "space-between",
          alignItems: "center",
          marginTop: 3,
        }}
      >
        <Button
          variant="contained"
          component={Link}
          to="/ventas"
          startIcon={<ArrowBack />}
          sx={{
            backgroundColor: "#0A4575",
            color: "#FFFFFF",
            "&:hover": { backgroundColor: "#083a5e" },
          }}
        >
          Volver
        </Button>

        <Box>
          <Button
            variant="outlined"
            color="error"
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF !important",
              borderColor: "#0A4575",
              "&:hover": { backgroundColor: "#083a5e !important" },
              marginRight: 2,
            }}
            onClick={handleEliminarFunerariosItems}
            disabled={funerariosItemsSeleccionados.length === 0}
          >
            Eliminar Ítems
          </Button>

          <Button
            type="submit"
            variant="contained"
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Guardar Solicitud
          </Button>
        </Box>
      </Box>
    </Box>
  );
};

export default ServicioFunerario;
