import React, { useEffect, useState } from "react";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Collapse,
  IconButton,
  Paper,
  TablePagination,
} from "@mui/material";
import {
  ArrowBack,
  ExpandMore as ExpandMoreIcon,
  ExpandLess as ExpandLessIcon,
} from "@mui/icons-material";
import axios from "axios";
import { jwtDecode } from "jwt-decode";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";

const apiUrl = process.env.REACT_APP_API_URL;

const Solicitudes = () => {
  const [clientes, setClientes] = useState({ items: [], total: 0 });
  const [expandedCliente, setExpandedCliente] = useState(null);
  const [expandedContrato, setExpandedContrato] = useState(null);
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);

  const formatDate = (dateString) => {
    const date = new Date(dateString);
    return date.toLocaleDateString("es-ES", {
      year: "numeric",
      month: "2-digit",
      day: "2-digit",
    });
  };

  useEffect(() => {
    const obtenerClientes = async () => {
      try {
        const token = localStorage.getItem("access_token");
        const decoded = jwtDecode(token);
        const vendedorId = decoded.sub ? Number(decoded.sub) : null;

        if (!vendedorId) {
          toast.error("No se pudo obtener el ID del vendedor.");
          return;
        }

        const config = {
          headers: { Authorization: `Bearer ${token}` },
        };

        const response = await axios.get(
          `${apiUrl}/api/v1/clientes/lista/${vendedorId}?page=${
            page + 1
          }&per_page=${rowsPerPage}`,
          config
        );
        setClientes(response.data);
      } catch (error) {
        toast.error("Error al obtener la lista de clientes.");
        console.error(error);
      }
    };

    obtenerClientes();
  }, [page, rowsPerPage]);

  const handleToggleCliente = (clienteId) => {
    setExpandedCliente(expandedCliente === clienteId ? null : clienteId);
    setExpandedContrato(null); // Cerrar los detalles de contratos al cambiar de cliente
  };

  const handleToggleContrato = (contratoId) => {
    setExpandedContrato(expandedContrato === contratoId ? null : contratoId);
  };

  const handleChangePage = (event, newPage) => {
    setPage(newPage);
  };

  const handleChangeRowsPerPage = (event) => {
    setRowsPerPage(parseInt(event.target.value, 10));
    setPage(0);
  };

  const getEstadoPagoTexto = (estadoPago) => {
    if (estadoPago === "pagado" || estadoPago === "inicial_pagada") {
      return estadoPago === "inicial_pagada" ? "INICIAL PAGADA" : "PAGADO";
    }
    return "PENDIENTE";
  };

  const getEstadoTexto = (estado) => {
    if (estado === "contrato") {
      return "CONTRATO";
    }
    return "SOLICITUD";
  };

  return (
    <Box
      sx={{
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        variant="h6"
        align="center"
        gutterBottom
        sx={{ fontFamily: "Bakbak One, sans-serif", color: "#0A4575" }}
      >
        SOLICITUDES DE CONTRATOS
      </Typography>

      <TableContainer component={Paper} sx={{ boxShadow: 3 }}>
        <Table size="small">
          <TableHead sx={{ backgroundColor: "#0A4575" }}>
            <TableRow>
              <TableCell sx={{ color: "#fff" }}>Nombre</TableCell>
              <TableCell sx={{ color: "#fff" }}>Cédula</TableCell>
              <TableCell sx={{ color: "#fff" }}>Teléfono</TableCell>
              <TableCell sx={{ color: "#fff" }}>Dirección</TableCell>
              <TableCell sx={{ color: "#fff" }}>Correo electrónico</TableCell>
              <TableCell sx={{ color: "#fff" }}></TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {clientes.items.length > 0 ? (
              clientes.items.map((cliente) => (
                <React.Fragment key={cliente.id}>
                  <TableRow>
                    <TableCell>{cliente.nombre}</TableCell>
                    <TableCell>{cliente.cedula}</TableCell>
                    <TableCell>{cliente.celular}</TableCell>
                    <TableCell>{cliente.direccion}</TableCell>
                    <TableCell>{cliente.email}</TableCell>
                    <TableCell>
                      <IconButton
                        onClick={() => handleToggleCliente(cliente.id)}
                      >
                        {expandedCliente === cliente.id ? (
                          <ExpandLessIcon />
                        ) : (
                          <ExpandMoreIcon />
                        )}
                      </IconButton>
                    </TableCell>
                  </TableRow>
                  <TableRow>
                    <TableCell colSpan={6} sx={{ padding: 2 }}>
                      <Collapse
                        in={expandedCliente === cliente.id}
                        timeout="auto"
                        unmountOnExit
                      >
                        <Box>
                          <Typography
                            variant="h6"
                            gutterBottom
                            sx={{
                              fontFamily: "Bakbak One, sans-serif",
                              color: "#0A4575",
                            }}
                          >
                            SOLICITUD
                          </Typography>
                          <TableContainer component={Paper}>
                            <Table size="small" sx={{ width: "100%" }}>
                              <TableHead sx={{ backgroundColor: "#0A4575" }}>
                                <TableRow>
                                  <TableCell sx={{ color: "#fff" }}>
                                    No. Transacción
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Tipo de Servicio
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Descripción
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Modalidad de Pago
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Fecha Solicitud
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Estado
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}>
                                    Estado del Pago
                                  </TableCell>
                                  <TableCell sx={{ color: "#fff" }}></TableCell>
                                </TableRow>
                              </TableHead>
                              <TableBody>
                                {cliente.solicitudes.length > 0 ? (
                                  cliente.solicitudes.map((solicitud) => (
                                    <React.Fragment key={solicitud.id}>
                                      <TableRow>
                                        <TableCell>
                                          {solicitud.numero_solicitud ||
                                            solicitud.numero_solicitud}
                                        </TableCell>
                                        <TableCell>
                                          {solicitud.tipo_servicio}
                                        </TableCell>
                                        <TableCell>
                                          {solicitud.descripcion}
                                        </TableCell>
                                        <TableCell>
                                          {solicitud.modalidadPago}
                                        </TableCell>
                                        <TableCell>
                                          {formatDate(solicitud.fecha_creacion)}
                                        </TableCell>
                                        <TableCell>
                                          <Button
                                            variant="contained"
                                            sx={{
                                              backgroundColor:
                                                solicitud.estado === "solicitud"
                                                  ? "#F44336"
                                                  : "#4CAF50",
                                              color: "#FFFFFF",
                                              "&:hover": {
                                                backgroundColor:
                                                  solicitud.estado ===
                                                  "solicitud"
                                                    ? "#D32F2F"
                                                    : "#45A049",
                                              },
                                              pointerEvents: "none",
                                            }}
                                          >
                                            {getEstadoTexto(solicitud.estado)}
                                          </Button>
                                        </TableCell>
                                        <TableCell>
                                          <Button
                                            variant="contained"
                                            sx={{
                                              backgroundColor:
                                                solicitud.estadoPago ===
                                                  "pagado" ||
                                                solicitud.estadoPago ===
                                                  "inicial_pagada"
                                                  ? "#4CAF50"
                                                  : "#F44336",
                                              color: "#FFFFFF",
                                              "&:hover": {
                                                backgroundColor:
                                                  solicitud.estadoPago ===
                                                    "pagado" ||
                                                  solicitud.estadoPago ===
                                                    "inicial_pagada"
                                                    ? "#45A049"
                                                    : "#D32F2F",
                                              },
                                              pointerEvents: "none",
                                            }}
                                          >
                                            {getEstadoPagoTexto(
                                              solicitud.estadoPago
                                            )}
                                          </Button>
                                        </TableCell>
                                        <TableCell>
                                          <IconButton
                                            onClick={() =>
                                              handleToggleContrato(solicitud.id)
                                            }
                                          >
                                            {expandedContrato ===
                                            solicitud.id ? (
                                              <ExpandLessIcon />
                                            ) : (
                                              <ExpandMoreIcon />
                                            )}
                                          </IconButton>
                                        </TableCell>
                                      </TableRow>
                                      <TableRow>
                                        <TableCell
                                          colSpan={8}
                                          sx={{ padding: 0, width: "100%" }}
                                        >
                                          <Collapse
                                            in={
                                              expandedContrato === solicitud.id
                                            }
                                            timeout="auto"
                                            unmountOnExit
                                          >
                                            <Box
                                              sx={{
                                                margin: 2,
                                                backgroundColor: "#fafafa",
                                              }}
                                            >
                                              <Typography
                                                variant="h6"
                                                gutterBottom
                                                sx={{
                                                  fontFamily:
                                                    "Bakbak One, sans-serif",
                                                  color: "#0A4575",
                                                }}
                                              >
                                                DETALLES DE LA SOLICITUD
                                              </Typography>
                                              <TableContainer
                                                component={Paper}
                                                sx={{
                                                  boxShadow: 3,
                                                  width: "100%",
                                                  margin: 0,
                                                  padding: 0,
                                                }}
                                              >
                                                <Table size="small">
                                                  <TableHead
                                                    sx={{
                                                      backgroundColor:
                                                        "#0A4575",
                                                    }}
                                                  >
                                                    <TableRow>
                                                      <TableCell
                                                        sx={{ color: "#fff" }}
                                                      >
                                                        Método Pago
                                                      </TableCell>
                                                      <TableCell
                                                        sx={{ color: "#fff" }}
                                                      >
                                                        Pago Inicial
                                                      </TableCell>
                                                      <TableCell
                                                        sx={{ color: "#fff" }}
                                                      >
                                                        Cuotas
                                                      </TableCell>
                                                      <TableCell
                                                        sx={{ color: "#fff" }}
                                                      >
                                                        Monto por Cuotas
                                                      </TableCell>
                                                      <TableCell
                                                        sx={{ color: "#fff" }}
                                                      >
                                                        Descuento
                                                      </TableCell>
                                                      <TableCell
                                                        sx={{ color: "#fff" }}
                                                      >
                                                        Precio total con
                                                        Descuento
                                                      </TableCell>
                                                      <TableCell
                                                        sx={{ color: "#fff" }}
                                                      >
                                                        Precio Total
                                                      </TableCell>
                                                    </TableRow>
                                                  </TableHead>
                                                  <TableBody>
                                                    {solicitud.detalles.map(
                                                      (detalle) => (
                                                        <TableRow
                                                          key={detalle.id}
                                                        >
                                                          <TableCell
                                                            sx={{
                                                              maxWidth: 250,
                                                              overflow:
                                                                "hidden",
                                                              textOverflow:
                                                                "ellipsis",
                                                              whiteSpace:
                                                                "nowrap",
                                                            }}
                                                          >
                                                            {detalle.metodoPago}
                                                          </TableCell>
                                                          <TableCell>
                                                            $
                                                            {
                                                              detalle.pagoInicial
                                                            }
                                                          </TableCell>
                                                          <TableCell>
                                                            {detalle.cuotas}
                                                          </TableCell>
                                                          <TableCell>
                                                            $
                                                            {
                                                              detalle.montoPorCuotas
                                                            }
                                                          </TableCell>

                                                          <TableCell>
                                                            ${detalle.descuento}
                                                          </TableCell>
                                                          <TableCell>
                                                            $
                                                            {
                                                              detalle.precioConDescuento
                                                            }
                                                          </TableCell>
                                                          <TableCell>
                                                            $
                                                            {detalle.precioPlan}
                                                          </TableCell>
                                                        </TableRow>
                                                      )
                                                    )}
                                                  </TableBody>
                                                </Table>
                                              </TableContainer>
                                            </Box>
                                          </Collapse>
                                        </TableCell>
                                      </TableRow>
                                    </React.Fragment>
                                  ))
                                ) : (
                                  <TableRow>
                                    <TableCell colSpan={7} align="center">
                                      Este cliente no posee solicitudes
                                      asociadas.
                                    </TableCell>
                                  </TableRow>
                                )}
                              </TableBody>
                            </Table>
                          </TableContainer>
                        </Box>
                      </Collapse>
                    </TableCell>
                  </TableRow>
                </React.Fragment>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={5} align="center">
                  Usted no posee clientes registrados.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
        <TablePagination
          rowsPerPageOptions={[5, 10, 25]}
          component="div"
          count={clientes.total}
          rowsPerPage={rowsPerPage}
          page={page}
          onPageChange={handleChangePage}
          onRowsPerPageChange={handleChangeRowsPerPage}
          labelRowsPerPage="Filas por página:"
          labelDisplayedRows={({ from, to, count }) =>
            `${from}-${to} de ${count}`
          }
        />
      </TableContainer>

      <Button
        variant="contained"
        component={Link}
        to="/ventas"
        startIcon={<ArrowBack />}
        sx={{
          marginTop: "15px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": { backgroundColor: "#083a5e" },
        }}
      >
        Volver
      </Button>
    </Box>
  );
};

export default Solicitudes;
