import React, { useState } from "react";
import axios from "axios";
import {
  Grid,
  Box,
  Button,
  TextField,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
} from "@mui/material";
import { ArrowBack } from "@mui/icons-material";
import { toast } from "react-toastify";
import { Link } from "react-router-dom";
const apiUrl = process.env.REACT_APP_API_URL;

const SubirDocumento = () => {
  const [cedula, setCedula] = useState("");
  const [cliente, setCliente] = useState(null);
  const [documento, setDocumento] = useState(null);

  const handleBuscarCliente = async () => {
    if (!cedula.trim()) {
      toast.error("Por favor, ingrese la cédula del cliente.");
      return;
    }

    try {
      const token = localStorage.getItem("access_token");
      const response = await axios.get(`${apiUrl}/api/v1/clientes/${cedula}`, {
        headers: {
          Authorization: `Bearer ${token}`,
        },
      });
      setCliente(response.data);
      setDocumento(null); // Reiniciar el documento al buscar un nuevo cliente
    } catch (error) {
      console.error("Error:", error.response?.data?.detail || error.message);
      toast.error(
        `Error al buscar el cliente: ${
          error.response?.data?.detail || error.message
        }`
      );
      setCliente(null);
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    if (!cliente || !documento) {
      toast.error("Por favor, seleccione un cliente y un documento.");
      return;
    }

    const formData = new FormData();
    formData.append("documento_identidad", documento);
    formData.append("nombre_documento", documento.name);

    try {
      const token = localStorage.getItem("access_token");
      const response = await axios.post(
        `${apiUrl}/api/clientes/${cliente.cedula}/upload-documento/`,
        formData,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "multipart/form-data",
          },
        }
      );
      toast.success("Documento subido con éxito.");
      setCliente(response.data); // Actualizar los datos del cliente con la respuesta
      setDocumento(null);
    } catch (error) {
      console.error("Error:", error.response?.data?.detail || error.message);
      toast.error(
        `Error al subir el documento: ${
          error.response?.data?.detail || error.message
        }`
      );
    }
  };

  return (
    <Box
      sx={{
        maxWidth: "950px",
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        variant="h6"
        sx={{ fontFamily: "Bakbak One", color: "#0A4575", marginBottom: 2 }}
      >
        Subir Documento de Identidad
      </Typography>
      <Grid container spacing={2} alignItems="center">
        <Grid item xs={12} sm={6} md={4}>
          <TextField
            fullWidth
            label="Cédula del Cliente"
            variant="outlined"
            size="small"
            value={cedula}
            onChange={(e) => setCedula(e.target.value)}
            sx={{ boxShadow: 3 }}
          />
        </Grid>
        <Grid item xs={12} sm={3} md={2}>
          <Button
            variant="contained"
            fullWidth
            onClick={handleBuscarCliente}
            sx={{
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Buscar
          </Button>
        </Grid>
      </Grid>

      {cliente && (
        <Box sx={{ marginTop: 3 }}>
          <TableContainer component={Paper}>
            <Table size="small">
              <TableHead sx={{ backgroundColor: "#0A4575" }}>
                <TableRow>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Nombre
                  </TableCell>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Dirección
                  </TableCell>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Teléfono
                  </TableCell>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Nombre Documento
                  </TableCell>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Acciones
                  </TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                <TableRow>
                  <TableCell>{cliente.nombre}</TableCell>
                  <TableCell>{cliente.direccion}</TableCell>
                  <TableCell>{cliente.telefono}</TableCell>
                  <TableCell>
                    {cliente.nombre_documento || "No subido"}
                  </TableCell>
                  <TableCell>
                    {cliente.nombre_documento ? (
                      <Typography sx={{ fontFamily: "Abel", color: "#0A4575" }}>
                        <Button
                          sx={{
                            backgroundColor: "#4CAF50",
                            color: "#FFFFFF",
                            "&:hover": { backgroundColor: "#45A049" },
                            pointerEvents: "none",
                          }}
                        >
                          DOCUMENTO SUBIDO
                        </Button>
                      </Typography>
                    ) : (
                      <>
                        <input
                          accept="image/*,application/pdf"
                          style={{ display: "none" }}
                          id="upload-documento"
                          type="file"
                          onChange={(e) => setDocumento(e.target.files[0])}
                        />
                        <label htmlFor="upload-documento">
                          <Button
                            variant="contained"
                            component="span"
                            sx={{
                              backgroundColor: "#0A4575",
                              color: "#FFFFFF",
                              "&:hover": { backgroundColor: "#083a5e" },
                              marginRight: 1,
                            }}
                          >
                            Subir Documento de Identidad
                          </Button>
                        </label>
                        <Button
                          variant="contained"
                          onClick={handleSubmit}
                          disabled={!documento}
                          sx={{
                            backgroundColor: "#0A4575",
                            color: "#FFFFFF",
                            "&:hover": { backgroundColor: "#083a5e" },
                          }}
                        >
                          Guardar
                        </Button>
                      </>
                    )}
                  </TableCell>
                </TableRow>
              </TableBody>
            </Table>
          </TableContainer>
          {documento && !cliente.nombre_documento && (
            <Typography sx={{ marginTop: 1, fontFamily: "Abel" }}>
              Documento seleccionado: {documento.name}
            </Typography>
          )}
          <Button
            variant="contained"
            component={Link}
            to="/ventas"
            startIcon={<ArrowBack />}
            sx={{
              marginTop: "15px",
              backgroundColor: "#0A4575",
              color: "#FFFFFF",
              "&:hover": { backgroundColor: "#083a5e" },
            }}
          >
            Volver
          </Button>
        </Box>
      )}
    </Box>
  );
};

export default SubirDocumento;
