import React, { useEffect, useState } from "react";
import {
  Box,
  Typography,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Button,
  Dialog,
  DialogTitle,
  DialogContent,
} from "@mui/material";
import axios from "axios";
import { toast } from "react-toastify";
import { jwtDecode } from "jwt-decode";
import { Link } from "react-router-dom";

const apiUrl = process.env.REACT_APP_API_URL;

const VendedoresList = () => {
  const [vendedores, setVendedores] = useState([]);
  const [contratos, setContratos] = useState([]);
  const [openDialog, setOpenDialog] = useState(false);
  const [selectedVendedor, setSelectedVendedor] = useState(null);

  const token = localStorage.getItem("access_token");
  let promotorId = null;

  if (token) {
    try {
      const decoded = jwtDecode(token);
      promotorId = decoded.sub ? Number(decoded.sub) : null;
    } catch (error) {
      console.error("Error al decodificar el token:", error);
    }
  }

  useEffect(() => {
    if (!promotorId) {
      console.error("Error: promotorId es null");
      return;
    }

    const obtenerVendedores = async () => {
      try {
        const config = { headers: { Authorization: `Bearer ${token}` } };
        const response = await axios.get(
          `${apiUrl}/api/v1/vendedores/list/${promotorId}`,
          config
        );
        setVendedores(response.data);
      } catch (error) {
        toast.info("No posee vendedores a su cargo.");
      }
    };

    obtenerVendedores();
  }, [token, promotorId]);

  // 🔹 Obtener contratos del vendedor seleccionado
  const verContratos = async (vendedorId) => {
    try {
      const config = { headers: { Authorization: `Bearer ${token}` } };
      const response = await axios.get(
        `${apiUrl}/api/vendedor/contratos/${vendedorId}`,
        config
      );
      setContratos(response.data);
      setSelectedVendedor(vendedorId);
      setOpenDialog(true);
    } catch (error) {
      toast.error("Error al obtener los contratos del vendedor.");
    }
  };

  const handleCloseDialog = () => {
    setOpenDialog(false);
    setContratos([]);
  };

  return (
    <Box
      component="form"
      sx={{
        maxWidth: "950px",
        margin: "0 auto",
        padding: 3,
        boxShadow: 3,
        backgroundColor: "#ffffff",
        borderRadius: 2,
      }}
    >
      <Typography
        sx={{
          textAlign: "center",
          fontFamily: "Bakbak One, sans-serif",
          color: "#0A4575",
        }}
        variant="h6"
        gutterBottom
      >
        LISTA DE VENDEDORES
      </Typography>

      <TableContainer>
        <Table>
          <TableHead>
            <TableRow sx={{ backgroundColor: "#0A4575" }}>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Vendedor
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Correo Electrónico
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Teléfono
              </TableCell>
              <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                Detalles
              </TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {vendedores.length > 0 ? (
              vendedores.map((vendedor) => (
                <TableRow key={vendedor.id}>
                  <TableCell>{vendedor.nombre}</TableCell>
                  <TableCell>{vendedor.email}</TableCell>
                  <TableCell>{vendedor.telefono || "No disponible"}</TableCell>
                  <TableCell>
                    <Button
                      variant="contained"
                      color="primary"
                      onClick={() => verContratos(vendedor.id)}
                    >
                      Ver Detalles
                    </Button>
                  </TableCell>
                </TableRow>
              ))
            ) : (
              <TableRow>
                <TableCell colSpan={4} align="center">
                  Usted no poseé vendedores asignados.
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </TableContainer>
      {/* Botón "Volver" a la izquierda */}
      <Button
        variant="contained"
        component={Link} // <--- Añadir esto
        to="/ventas" // <--- Asegúrate de que esta ruta existe en tu App
        sx={{
          marginTop: "10px",
          backgroundColor: "#0A4575",
          color: "#FFFFFF",
          "&:hover": {
            backgroundColor: "#083a5e",
          },
        }}
      >
        Volver
      </Button>

      {/* 🔹 Diálogo para mostrar los contratos del vendedor */}
      <Dialog
        open={openDialog}
        onClose={handleCloseDialog}
        maxWidth="md"
        fullWidth
      >
        <DialogTitle
          sx={{
            textAlign: "center",
            fontFamily: "Bakbak One, sans-serif",
            color: "#0A4575",
          }}
        >
          Contratos del Vendedor
        </DialogTitle>
        <DialogContent>
          <TableContainer>
            <Table>
              <TableHead>
                <TableRow sx={{ backgroundColor: "#0A4575" }}>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Número de Transacción
                  </TableCell>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Tipo de Servicio
                  </TableCell>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Precio
                  </TableCell>
                  <TableCell sx={{ color: "#FFFFFF", fontSize: "16px" }}>
                    Estado
                  </TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {contratos.map((contrato) => (
                  <TableRow key={contrato.id}>
                    <TableCell>{contrato.numero_contrato}</TableCell>
                    <TableCell>{contrato.tipo_servicio}</TableCell>
                    <TableCell>{contrato.precio_total}</TableCell>
                    <TableCell>{contrato.estado}</TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>
        </DialogContent>
      </Dialog>
    </Box>
  );
};

export default VendedoresList;
