import React, { useState } from 'react';
import axios from 'axios';
import {
  Box,
  Button,
  Typography,
  Select,
  MenuItem,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Grid,
  Checkbox,
  TextField,
  FormControl,
  InputLabel
} from '@mui/material';

import Cremacion from './Cremacion';
import Exequia from './Exequia';
import Inhumacion from './Inhumacion';
import Columbarios from './Columbarios';
import Parcelas from './Parcelas';
import NichosAtaud from './NichosAtaud';
import Previsivos from './Previsivos';
import ServicioFunerario from './ServicioFunerario';

const VentasContrato = () => {
  const [itemSeleccionado, setItemSeleccionado] = useState('');
  const [items, setItems] = useState([]); // Aquí se guardarán los productos añadidos 
  const [itemsSeleccionados, setItemsSeleccionados] = useState([]); // Para eliminar los ítems seleccionados
  const [funerariosItems, setFunerariosItems] = useState([]);
  const [funerariosItemsSeleccionados, setFunerariosItemsSeleccionados] = useState([]);
  const [cliente, setCliente] = useState({
    cliente: '',
    cedula: '',
    fechaNac: '',
    edad: '',
    telefono: '',
    celular: '',
    correo: '',
    direccion: '',
    fileName: ''
  });

  const setClienteData = (clienteData) => {
    setCliente(clienteData);
  }

  const handleProductoChange = (e) => {
    setItemSeleccionado(e.target.value);
  };

  const handleAgregarItem = (nicho, plan, precio) => {
    if (plan) {
      const item = {
        codigo: nicho,
        descripcion: plan.servicio || 'Sin descripción',
        pagoInicial: plan.inicial && precio === plan.precioFinanciado ? plan.inicial : '',  // Solo si es financiado
        cuotas: plan.cuotas && precio === plan.precioFinanciado ? plan.cuotas : '',          // Solo si es financiado
        montoPorCuota: plan.precio_cuotas && precio === plan.precioFinanciado ? plan.precio_cuotas : '', // Solo si es financiado
        precioTotal: precio || 0, // Ajustamos para usar el precio de contado o financiado correctamente
        modalidadPago: precio === plan.precioContado ? 'Contado' : 'Financiado',
      };
      setItems([...items, item]);
    }
  };

  const agregarItemGenerico = (descripcion, plan, precio) => {
    const nuevoItem = {
      codigo: '',
      descripcion: plan.servicio,
      modalidadPago: '',
      pagoInicial: '',
      montoPorCuota: '',
      precioTotal: precio
    };
    setItems((prevItems) => [...prevItems, nuevoItem]);
  };

  const agregarItemFunerario = (nuevoItem) => {
    setFunerariosItems((prevItems) => [...prevItems, nuevoItem]);
  };

  const agregarItemPrevisivo = (item) => {
    setItems((prevItems) => [...prevItems, item]);
  };

  // Función para renderizar el componente según el valor seleccionado
  const renderFormularioProducto = () => {
    switch (itemSeleccionado) {
      case 'cremacion':
        return <Cremacion agregarAlPresupuesto={agregarItemGenerico} />;
      case 'parcela':
        return <Parcelas agregarAlPresupuesto={agregarItemGenerico} />;
        case 'honra de exequia':
          return <Exequia agregarAlPresupuesto={agregarItemGenerico} />;        
        case 'inhumación de cenizas':
          return <Inhumacion agregarAlPresupuesto={agregarItemGenerico} />;        
        case 'nichos de columbarios':
        return <Columbarios agregarAlPresupuesto={handleAgregarItem} />;
      case 'nichos de atatud':
        return <NichosAtaud agregarAlPresupuesto={agregarItemGenerico} />;
      case 'servicio funenario':
        return <ServicioFunerario agregarAlPresupuesto={agregarItemFunerario} />;
        case 'previsivo':
          return (
            <Previsivos
              agregarAlPresupuesto={agregarItemPrevisivo}
              setClienteData={setClienteData}             
            />
          );     
      default:
        return null;
    }
  };

  // Funciones de eliminación separadas
  const handleEliminarItems = () => {
    const nuevosItems = items.filter((_, index) => !itemsSeleccionados.includes(index));
    setItems(nuevosItems);
    setItemsSeleccionados([]); // Reiniciar los ítems seleccionados
  };

  const handleEliminarFunerariosItems = () => {
    const nuevosFunerariosItems = funerariosItems.filter((_, index) => !funerariosItemsSeleccionados.includes(index));
    setFunerariosItems(nuevosFunerariosItems);
    setFunerariosItemsSeleccionados([]); // Reiniciar los ítems seleccionados de funerarios
  };

  // Manejar la selección de los ítems para eliminar
  const handleSeleccionarItem = (index) => {
    const seleccionado = itemsSeleccionados.includes(index);
    if (seleccionado) {
      setItemsSeleccionados(itemsSeleccionados.filter((i) => i !== index));
    } else {
      setItemsSeleccionados([...itemsSeleccionados, index]);
    }
  };

  // Manejar la selección de los ítems funerarios para eliminar
  const handleSeleccionarFunerariosItem = (index) => {
    const seleccionado = funerariosItemsSeleccionados.includes(index);
    if (seleccionado) {
      setFunerariosItemsSeleccionados(funerariosItemsSeleccionados.filter((i) => i !== index));
    } else {
      setFunerariosItemsSeleccionados([...funerariosItemsSeleccionados, index]);
    }
  };

  // Calcula la edad a partir de la fecha de nacimiento
  const calcularEdad = (fechaNacimiento) => {
    const hoy = new Date();
    const fechaNac = new Date(fechaNacimiento);
    let edad = hoy.getFullYear() - fechaNac.getFullYear();
    const mes = hoy.getMonth() - fechaNac.getMonth();

    if (mes < 0 || (mes === 0 && hoy.getDate() < fechaNac.getDate())) {
      edad--;
    }

    return edad;
  };

  // Actualiza los datos del cliente
  const handleClienteChange = (e) => {
    const { name, value } = e.target;
    let updatedFields = { [name]: value };

    if (name === 'fechaNac') {
      const edadCalculada = calcularEdad(value);
      updatedFields.edad = edadCalculada.toString();
    }

    setCliente((prev) => ({
      ...prev,
      ...updatedFields,
    }));
  };

  const handleFileChange = async (event) => {
    const file = event.target.files[0];
    if (file) {
      const formData = new FormData();
      formData.append('file', file);

      try {
        const response = await axios.post('http://localhost:8001/api/upload', formData, {
          headers: {
            'Content-Type': 'multipart/form-data',
          },
        });

        setCliente((prev) => ({
          ...prev,
          fileName: file.name,
          documentoIdentidad: response.data.url, // Assuming the API returns the file URL
        }));
      } catch (error) {
        console.error('Error uploading file:', error);
        // Handle error (e.g., show error message to user)
      }
    }
  };

  const handleSubmit = async (e) => {
    e.preventDefault();
    
    try {
      // First, create or update the client
      const clienteResponse = await axios.post('http://localhost:8001/api/clientes', cliente);
      const clienteId = clienteResponse.data.id;

      // Then, create the contract
      const contratoData = {
        clienteId,
        items: [...items, ...funerariosItems],
        // Add other necessary contract fields
      };
      
      const contratoResponse = await axios.post('http://localhost:8001/api/contratos', contratoData);
      
      console.log('Contrato creado:', contratoResponse.data);
      // Handle success (e.g., show success message, redirect, etc.)
    } catch (error) {
      console.error('Error al crear el contrato:', error);
      // Handle error (e.g., show error message to user)
    }
  };

  return (
    <Box component="form" 
      onSubmit={handleSubmit}       
      sx={{
        maxWidth: '950px',
        margin: '0 auto',  
        padding: 3,      
        boxShadow: 3,
        backgroundColor: '#ffffff',
        borderRadius: 2,
      }}
    >

      <Typography sx={{ textAlign: 'center', marginBottom: '15px', fontFamily: 'Bakbak One, sans-serif', color: '#0A4575' }} variant="h5" gutterBottom>
        NUEVO CONTRATO
      </Typography>

      {/* Selección del item */}
      <Grid container spacing={3} alignItems="center">
        <Grid item xs={12} sm={4}>
          <FormControl fullWidth size="small" variant="outlined" sx={{ boxShadow: 3 }}>
            <InputLabel>Seleccione un Producto Servicio </InputLabel>
            <Select value={itemSeleccionado} label=" Seleccione un Producto o Servicio " onChange={handleProductoChange}>
              <MenuItem value="" disabled>
                Selecciona el Producto o Servicio
              </MenuItem>
              <MenuItem value="certificado funerario">Certificado Funerario</MenuItem>
              <MenuItem value="cremacion">Cremación</MenuItem>              
              <MenuItem value="parcela">Fosas/Parcelas</MenuItem>
              <MenuItem value="honra de exequia">Honra de Exequias</MenuItem>
              <MenuItem value="inhumación de cenizas">Inhumación de Cenizas</MenuItem>
              <MenuItem value="nichos de atatud">Nichos de Ataud</MenuItem>
              <MenuItem value="nichos de columbarios">Nichos de Columbarios</MenuItem>                                          
              <MenuItem value="servicio funenario">Servicio Funerario</MenuItem>                                          
              <MenuItem value="previsivo">Servicio Previsivo</MenuItem>
            </Select>
          </FormControl>  
        </Grid>
      </Grid>

      {renderFormularioProducto()}
      {itemSeleccionado !== 'servicio funenario' && (
        <>
          {itemSeleccionado !== 'previsivo' && (
            <>
                {/* DATOS DEL CLIENTE */}
                <Typography
                  sx={{ marginTop: '10px', marginBottom: '5px', fontFamily: 'Bakbak One, sans-serif', color: '#0A4575' }}
                  variant="h6"
                >
                  DATOS DEL CLIENTE
                </Typography>

                <Grid container spacing={3}>
                  <Grid item xs={12} sm={4}>
                  <TextField
                    fullWidth
                    label="Nombre cliente *"
                    variant="outlined"
                    size="small"
                    name="cliente"
                    value={cliente.cliente}
                    onChange={handleClienteChange}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <TextField
                    fullWidth
                    label="Cédula o Rif *"
                    variant="outlined"
                    size="small"
                    name="cedula"
                    value={cliente.cedula}
                    onChange={handleClienteChange}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                    <TextField
                      fullWidth
                      label="Fecha de Nacimiento"
                      variant="outlined"
                      type="date"
                      size="small"
                      name="fechaNac"
                      value={cliente.fechaNac || ''}
                      onChange={handleClienteChange}
                      required
                      sx={{ boxShadow: 3 }}
                      InputLabelProps={{
                        shrink: true,
                      }}
                    />
                  </Grid>
                  <Grid item xs={12} sm={2}>
                    <TextField
                      fullWidth
                      label="Edad"
                      variant="outlined"
                      size="small"
                      name="edad"
                      value={cliente.edad}
                      required
                      sx={{ boxShadow: 3 }}
                      InputProps={{
                        readOnly: true,
                      }}
                    />
                  </Grid>
                <Grid item xs={12} sm={3}>
                  <TextField
                    fullWidth
                    label="Teléfono *"
                    variant="outlined"
                    size="small"
                    name="telefono"
                    value={cliente.telefono}
                    onChange={handleClienteChange}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={3}>
                  <TextField
                    fullWidth
                    label="Celular *"
                    variant="outlined"
                    size="small"
                    name="celular"
                    value={cliente.celular}
                    onChange={handleClienteChange}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={6}>
                  <TextField
                    fullWidth
                    label="Email *"
                    variant="outlined"
                    size="small"
                    name="correo"
                    value={cliente.correo}
                    onChange={handleClienteChange}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={5}>
                  <TextField
                    fullWidth
                    label="Dirección *"
                    variant="outlined"
                    size="small"
                    name="direccion"
                    value={cliente.direccion}
                    onChange={handleClienteChange}
                    required
                    sx={{ boxShadow: 3 }}
                  />
                </Grid>
                <Grid item xs={12} sm={4}>            
                  <input
                    accept="image/*" // puedes restringir el tipo de archivo permitido
                    style={{ display: 'none' }} // Ocultar el input real
                    id="file-upload"
                    type="file"                
                    onChange={handleFileChange}
                  />
                  
                  <label htmlFor="file-upload">
                    <Button
                      variant="contained"
                      component="span"
                      sx={{
                        backgroundColor: '#0A4575', // Estilo personalizado
                        color: '#FFFFFF',
                        '&:hover': {
                          backgroundColor: '#083a5e',
                        },
                      }}
                    >
                      Subir Documento de Identidad
                    </Button>
                  </label>
                </Grid>
                <Grid item xs={12} sm={3}>
                  {/* Mostrar el nombre del archivo seleccionado */}
                  {cliente.fileName && (
                    <Typography sx={{ marginTop: '10px', fontFamily: 'Abel' }}>
                      Documento: {cliente.fileName}
                    </Typography>
                  )}            
                </Grid>
              </Grid>
            </>
          )}        
          <Typography sx={{ textAlign: 'center', marginTop: '25px', fontFamily: 'Bakbak One, sans-serif', color: '#0A4575' }} variant="h6" gutterBottom>
            RESUMEN DEL CONTRATO
          </Typography>

          {/* Tabla de Presupuesto */}
          <TableContainer component={Paper}>
            <Table>
              <TableHead sx={{ backgroundColor: '#0A4575', color: '#FFFFFF' }}>
                <TableRow>
                  <TableCell sx={{ color: '#FFFFFF', fontSize: '16px', width: '110px' }}>Código</TableCell>
                  <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Descripción</TableCell>
                  <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Modalidad</TableCell>
                  <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Inicial</TableCell>
                  <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Cuotas</TableCell>
                  <TableCell sx={{ color: '#FFFFFF', fontSize: '16px', width: '110px' }}>Monto Cuota</TableCell>
                  <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Total</TableCell>
                  <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Seleccionar</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {items.map((item, index) => (
                  <TableRow key={index}>
                    <TableCell>{item.codigo}</TableCell>
                    <TableCell>{item.descripcion}</TableCell>
                    <TableCell>{item.modalidadPago}</TableCell>
                    <TableCell>{item.pagoInicial !== '' ? `$${item.pagoInicial}` : ''}</TableCell>
                    <TableCell>{item.cuotas}</TableCell>
                    <TableCell>{item.montoPorCuota !== '' ? `$${item.montoPorCuota}` : ''}</TableCell>
                    <TableCell>{`$${item.precioTotal}`}</TableCell>
                    <TableCell>
                      <Checkbox
                        checked={itemsSeleccionados.includes(index)}
                        onChange={() => handleSeleccionarItem(index)}
                        sx={{
                          color: '#0A4575',
                          '&.Mui-checked': {
                            color: '#0A4575',
                          },
                        }}
                      />
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>

          <Box sx={{ display: 'flex', justifyContent: 'flex-end', marginTop: 3 }}>
            {/* Botón para eliminar ítems seleccionados */}
            <Button
              variant="outlined"
              color="error"
              sx={{ 
                backgroundColor: '#0A4575', // Color de fondo del botón
                color: '#FFFFFF !important', // Color del texto
                borderColor: '#0A4575', // Color del borde para mantener consistencia
                '&:hover': {
                  backgroundColor: '#083a5e !important', // Color de fondo al pasar el cursor
                },
                marginRight: 2,
              }}
              onClick={handleEliminarItems}
              disabled={itemsSeleccionados.length === 0} // Deshabilitar si no hay ítems seleccionados
            >
              Eliminar Ítems
            </Button> 
          </Box>
        </>  
      )}
          {itemSeleccionado === 'servicio funenario' && (
            <>
              <Typography sx={{ textAlign: 'center', marginTop: '25px', fontFamily: 'Bakbak One, sans-serif', color: '#0A4575' }} variant="h6" gutterBottom>
                RESUMEN DE SERVICIOS FUNERARIOS
              </Typography>
              {/* Nueva Tabla de Servicios Funerarios */}
              <TableContainer component={Paper}>
                <Table>
                  <TableHead sx={{ backgroundColor: '#0A4575', color: '#FFFFFF' }}>
                    <TableRow>
                      <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Traslado</TableCell>
                      <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Tiempo de Velación</TableCell>                      
                      <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Precio a Pagar</TableCell>
                      <TableCell sx={{ color: '#FFFFFF', fontSize: '16px' }}>Seleccionar</TableCell>
                    </TableRow>
                  </TableHead>
                  <TableBody>
                    {funerariosItems.map((item, index) => (
                      <TableRow key={index}>
                        <TableCell>{item.traslado}</TableCell>
                        <TableCell>{item.velacion}</TableCell>
                        <TableCell>{`$${item.precio}`}</TableCell>
                        <TableCell>
                          <Checkbox
                            checked={funerariosItemsSeleccionados.includes(index)}
                            onChange={() => handleSeleccionarFunerariosItem(index)}
                            sx={{
                              color: '#0A4575',
                              '&.Mui-checked': {
                                color: '#0A4575',
                              },
                            }}
                          />
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </TableContainer>
            <Box sx={{ display: 'flex', justifyContent: 'flex-end', marginTop: 3 }}>
              {/* Botón para eliminar ítems seleccionados */}
              <Button
                variant="outlined"
                color="error"
                sx={{ 
                  backgroundColor: '#0A4575', // Color de fondo del botón
                  color: '#FFFFFF !important', // Color del texto
                  borderColor: '#0A4575', // Color del borde para mantener consistencia
                  '&:hover': {
                    backgroundColor: '#083a5e !important', // Color de fondo al pasar el cursor
                  },
                  marginRight: 2,
                }}
                onClick={handleEliminarFunerariosItems}
                disabled={funerariosItemsSeleccionados.length === 0} // Deshabilitar si no hay ítems seleccionados
              >
                Eliminar Ítems
              </Button> 
            </Box>              
            </>
          )}
      
          <Box sx={{ display: 'flex', justifyContent: 'flex-end', marginTop: 3 }}>
            <Button
              type="submit"
              variant="contained"
              sx={{ 
                backgroundColor: '#0A4575', // Color de fondo
                color: '#FFFFFF', // Color del texto
                '&:hover': {
                  backgroundColor: '#000000', // Color de fondo al pasar el cursor
                }
              }}
            >
              Guardar Contrato
            </Button>
          </Box>        
    </Box>  
  );
};

export default VentasContrato;
